"use strict";
/**
 *  Copyright 2021 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.AddAwsServiceEndpoint = exports.ServiceEndpointTypes = exports.buildVpc = void 0;
const ec2 = require("@aws-cdk/aws-ec2");
const security_group_helper_1 = require("./security-group-helper");
const utils_1 = require("./utils");
function buildVpc(scope, props) {
    var _a;
    if (props === null || props === void 0 ? void 0 : props.existingVpc) {
        return props === null || props === void 0 ? void 0 : props.existingVpc;
    }
    let cumulativeProps = props === null || props === void 0 ? void 0 : props.defaultVpcProps;
    if (props === null || props === void 0 ? void 0 : props.userVpcProps) {
        cumulativeProps = utils_1.overrideProps(cumulativeProps, props === null || props === void 0 ? void 0 : props.userVpcProps);
    }
    if (props === null || props === void 0 ? void 0 : props.constructVpcProps) {
        cumulativeProps = utils_1.overrideProps(cumulativeProps, props === null || props === void 0 ? void 0 : props.constructVpcProps);
    }
    const vpc = new ec2.Vpc(scope, "Vpc", cumulativeProps);
    // Add VPC FlowLogs with the default setting of trafficType:ALL and destination: CloudWatch Logs
    const flowLog = vpc.addFlowLog("FlowLog");
    // Add Cfn Nag suppression for PUBLIC subnets to suppress WARN W33: EC2 Subnet should not have MapPublicIpOnLaunch set to true
    vpc.publicSubnets.forEach((subnet) => {
        const cfnSubnet = subnet.node.defaultChild;
        cfnSubnet.cfnOptions.metadata = {
            cfn_nag: {
                rules_to_suppress: [{
                        id: 'W33',
                        reason: 'Allow Public Subnets to have MapPublicIpOnLaunch set to true'
                    }]
            }
        };
    });
    // Add Cfn Nag suppression for CloudWatchLogs LogGroups data is encrypted
    const cfnLogGroup = (_a = flowLog.logGroup) === null || _a === void 0 ? void 0 : _a.node.defaultChild;
    cfnLogGroup.cfnOptions.metadata = {
        cfn_nag: {
            rules_to_suppress: [{
                    id: 'W84',
                    reason: 'By default CloudWatchLogs LogGroups data is encrypted using the CloudWatch server-side encryption keys (AWS Managed Keys)'
                }]
        }
    };
    return vpc;
}
exports.buildVpc = buildVpc;
var ServiceEndpointTypes;
(function (ServiceEndpointTypes) {
    ServiceEndpointTypes["DYNAMODB"] = "DDB";
    ServiceEndpointTypes["SNS"] = "SNS";
    ServiceEndpointTypes["SQS"] = "SQS";
    ServiceEndpointTypes["S3"] = "S3";
    ServiceEndpointTypes["STEPFUNCTIONS"] = "STEPFUNCTIONS";
})(ServiceEndpointTypes = exports.ServiceEndpointTypes || (exports.ServiceEndpointTypes = {}));
var EndpointTypes;
(function (EndpointTypes) {
    EndpointTypes["GATEWAY"] = "Gateway";
    EndpointTypes["INTERFACE"] = "Interface";
})(EndpointTypes || (EndpointTypes = {}));
const endpointSettings = [
    {
        endpointName: ServiceEndpointTypes.DYNAMODB,
        endpointType: EndpointTypes.GATEWAY,
        endpointGatewayService: ec2.GatewayVpcEndpointAwsService.DYNAMODB,
    },
    {
        endpointName: ServiceEndpointTypes.S3,
        endpointType: EndpointTypes.GATEWAY,
        endpointGatewayService: ec2.GatewayVpcEndpointAwsService.S3,
    },
    {
        endpointName: ServiceEndpointTypes.SNS,
        endpointType: EndpointTypes.INTERFACE,
        endpointInterfaceService: ec2.InterfaceVpcEndpointAwsService.SNS,
    },
    {
        endpointName: ServiceEndpointTypes.SQS,
        endpointType: EndpointTypes.INTERFACE,
        endpointInterfaceService: ec2.InterfaceVpcEndpointAwsService.SQS,
    },
];
function AddAwsServiceEndpoint(scope, vpc, interfaceTag) {
    if (!vpc.node.children.some((child) => child.node.id === interfaceTag)) {
        const service = endpointSettings.find((endpoint) => endpoint.endpointName === interfaceTag);
        if (!service) {
            throw new Error("Unsupported Service sent to AddServiceEndpoint");
        }
        if (service.endpointType === EndpointTypes.GATEWAY) {
            vpc.addGatewayEndpoint(interfaceTag, {
                service: service.endpointGatewayService,
            });
        }
        if (service.endpointType === EndpointTypes.INTERFACE) {
            const endpointDefaultSecurityGroup = security_group_helper_1.buildSecurityGroup(scope, "ReplaceEndpointDefaultSecurityGroup", {
                vpc,
                allowAllOutbound: true,
            }, [{ peer: ec2.Peer.ipv4(vpc.vpcCidrBlock), connection: ec2.Port.tcp(443) }], []);
            vpc.addInterfaceEndpoint(interfaceTag, {
                service: service.endpointInterfaceService,
                securityGroups: [endpointDefaultSecurityGroup],
            });
        }
    }
    return;
}
exports.AddAwsServiceEndpoint = AddAwsServiceEndpoint;
//# sourceMappingURL=data:application/json;base64,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