"use strict";
/**
 *  Copyright 2021 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
// Imports
const assert_1 = require("@aws-cdk/assert");
require("@aws-cdk/assert/jest");
const aws_glue_1 = require("@aws-cdk/aws-glue");
const aws_iam_1 = require("@aws-cdk/aws-iam");
const aws_s3_1 = require("@aws-cdk/aws-s3");
const core_1 = require("@aws-cdk/core");
const defaults = require("..");
// --------------------------------------------------------------
// Test deployment with role creation
// --------------------------------------------------------------
test('Test deployment with role creation', () => {
    // Stack
    const stack = new core_1.Stack();
    const _jobID = 'glueetl';
    const _jobRole = new aws_iam_1.Role(stack, 'CustomETLJobRole', {
        assumedBy: new aws_iam_1.ServicePrincipal('glue.amazonaws.com')
    });
    const cfnJobProps = defaults.DefaultGlueJobProps(_jobRole, {
        name: _jobID,
        pythonVersion: '3',
        scriptLocation: 's3://fakelocation/script'
    }, 'testETLJob', {}, '1.0');
    const _database = defaults.createGlueDatabase(stack, defaults.DefaultGlueDatabaseProps());
    defaults.buildGlueJob(stack, {
        glueJobProps: cfnJobProps,
        database: _database,
        table: defaults.createGlueTable(stack, _database, undefined, [{
                name: "id",
                type: "int",
                comment: ""
            }], 'kinesis', { STREAM_NAME: 'testStream' })
    });
    // Assertion 1
    expect(assert_1.SynthUtils.toCloudFormation(stack)).toMatchSnapshot();
    // Assertion 2
    expect(stack).toHaveResourceLike('AWS::Glue::Job', {
        Type: "AWS::Glue::Job",
        Properties: {
            Command: {
                Name: "glueetl",
                PythonVersion: "3",
                ScriptLocation: "s3://fakelocation/script"
            },
            Role: {
                "Fn::GetAtt": [
                    "CustomETLJobRole90A83A66",
                    "Arn"
                ]
            },
            SecurityConfiguration: "testETLJob"
        }
    }, assert_1.ResourcePart.CompleteDefinition);
});
// --------------------------------------------------------------
// Pass an existing Glue Job
// --------------------------------------------------------------
test('Create a Glue Job outside the construct', () => {
    // Stack
    const stack = new core_1.Stack();
    const _existingCfnJob = new aws_glue_1.CfnJob(stack, 'ExistingJob', {
        command: {
            name: 'pythonshell',
            pythonVersion: '2',
            scriptLocation: 's3://existingFakeLocation/existingScript'
        },
        role: new aws_iam_1.Role(stack, 'ExistingJobRole', {
            assumedBy: new aws_iam_1.ServicePrincipal('glue.amazon.com'),
            description: 'Existing role'
        }).roleArn,
        glueVersion: '1',
        allocatedCapacity: 2,
        maxCapacity: 4,
        numberOfWorkers: 2,
        workerType: 'Standard'
    });
    const _database = defaults.createGlueDatabase(stack, defaults.DefaultGlueDatabaseProps());
    defaults.buildGlueJob(stack, {
        existingCfnJob: _existingCfnJob,
        outputDataStore: {
            datastoreType: defaults.SinkStoreType.S3
        },
        database: _database,
        table: defaults.createGlueTable(stack, _database, undefined, [{
                name: "id",
                type: "int",
                comment: ""
            }], 'kinesis', { STREAM_NAME: 'testStream' })
    });
    expect(assert_1.SynthUtils.toCloudFormation(stack)).toMatchSnapshot();
    expect(stack).toHaveResourceLike('AWS::Glue::Job', {
        Type: "AWS::Glue::Job",
        Properties: {
            AllocatedCapacity: 2,
            Command: {
                Name: "pythonshell",
                PythonVersion: "2",
                ScriptLocation: "s3://existingFakeLocation/existingScript",
            },
            GlueVersion: "1",
            MaxCapacity: 4,
            NumberOfWorkers: 2,
            Role: {
                "Fn::GetAtt": [
                    "ExistingJobRole8F750976",
                    "Arn",
                ],
            },
            WorkerType: "Standard",
        }
    }, assert_1.ResourcePart.CompleteDefinition);
});
// --------------------------------------------------------------
// Provide additional parameters other than default ones
// --------------------------------------------------------------
test('Test custom deployment properties', () => {
    // Stack
    const stack = new core_1.Stack();
    const _commandName = 'glueetl';
    const cfnJobProps = {
        command: {
            name: _commandName,
            pythonVersion: '3',
            scriptLocation: 's3://existingFakeLocation/existingScript'
        },
        role: new aws_iam_1.Role(stack, 'ExistingJobRole', {
            assumedBy: new aws_iam_1.ServicePrincipal('glue.amazon.com'),
            description: 'Existing role'
        }).roleArn,
        glueVersion: '1',
        allocatedCapacity: 2,
        maxCapacity: 4,
        numberOfWorkers: 2,
        workerType: 'Standard'
    };
    const _database = defaults.createGlueDatabase(stack);
    defaults.buildGlueJob(stack, {
        glueJobProps: cfnJobProps,
        outputDataStore: {
            datastoreType: defaults.SinkStoreType.S3
        },
        database: _database,
        table: defaults.createGlueTable(stack, _database, undefined, [{
                name: "id",
                type: "int",
                comment: ""
            }], 'kinesis', { STREAM_NAME: 'testStream' })
    });
    // Assertion 1
    expect(assert_1.SynthUtils.toCloudFormation(stack)).toMatchSnapshot();
    // check if Glue Job Resource was created correctly
    expect(stack).toHaveResourceLike('AWS::Glue::Job', {
        Properties: {
            AllocatedCapacity: 2,
            Command: {
                Name: "glueetl",
                PythonVersion: "3",
                ScriptLocation: "s3://existingFakeLocation/existingScript",
            },
            GlueVersion: "1",
            MaxCapacity: 4,
            NumberOfWorkers: 2,
            Role: {
                "Fn::GetAtt": [
                    "ExistingJobRole8F750976",
                    "Arn",
                ],
            },
            SecurityConfiguration: "ETLJobSecurityConfig",
            WorkerType: "Standard",
        },
        Type: "AWS::Glue::Job"
    }, assert_1.ResourcePart.CompleteDefinition);
    // check if the role is created
    expect(stack).toHaveResourceLike('AWS::IAM::Role', {
        Type: "AWS::IAM::Role",
        Properties: {
            AssumeRolePolicyDocument: {
                Statement: [
                    {
                        Action: "sts:AssumeRole",
                        Effect: "Allow",
                        Principal: {
                            Service: "glue.amazon.com"
                        }
                    }
                ],
                Version: "2012-10-17"
            },
            Description: "Existing role"
        }
    }, assert_1.ResourcePart.CompleteDefinition);
    // check if the security config is created
    expect(stack).toHaveResourceLike('AWS::Glue::SecurityConfiguration', {
        Properties: {
            EncryptionConfiguration: {
                JobBookmarksEncryption: {
                    JobBookmarksEncryptionMode: "CSE-KMS",
                    KmsKeyArn: {
                        "Fn::Join": [
                            "", [
                                "arn:", {
                                    Ref: "AWS::Partition",
                                },
                                ":kms:", {
                                    Ref: "AWS::Region",
                                },
                                ":", {
                                    Ref: "AWS::AccountId",
                                },
                                ":alias/aws/glue",
                            ],
                        ],
                    },
                },
                S3Encryptions: [{
                        S3EncryptionMode: "SSE-S3",
                    }],
            },
            Name: "ETLJobSecurityConfig",
        },
        Type: "AWS::Glue::SecurityConfiguration",
    }, assert_1.ResourcePart.CompleteDefinition);
});
// --------------------------------------------------------------
// Do not supply parameters and error out
// --------------------------------------------------------------
test('Do no supply glueJobProps or existingCfnJob and error out', () => {
    const stack = new core_1.Stack();
    try {
        const _database = defaults.createGlueDatabase(stack);
        defaults.buildGlueJob(stack, {
            outputDataStore: {
                datastoreType: defaults.SinkStoreType.S3
            },
            database: _database,
            table: defaults.createGlueTable(stack, _database, defaults.DefaultGlueTableProps(_database, [{
                    name: "id",
                    type: "int",
                    comment: ""
                }], 'kinesis', { STREAM_NAME: 'testStream' }))
        });
    }
    catch (error) {
        expect(error).toBeInstanceOf(Error);
    }
});
// --------------------------------------------------------------
// Allow the construct to create the job role required
// --------------------------------------------------------------
test('Test deployment with role creation', () => {
    // Stack
    const stack = new core_1.Stack();
    const _jobID = 'glueetl';
    const cfnJobProps = {
        command: {
            name: _jobID,
            pythonVersion: '3',
            scriptLocation: 's3://fakelocation/script'
        }
    };
    const _database = defaults.createGlueDatabase(stack);
    defaults.buildGlueJob(stack, {
        glueJobProps: cfnJobProps,
        outputDataStore: {
            datastoreType: defaults.SinkStoreType.S3
        },
        database: _database,
        table: defaults.createGlueTable(stack, _database, undefined, [{
                name: "id",
                type: "int",
                comment: ""
            }], 'kinesis', { STREAM_NAME: 'testStream' })
    });
    // Assertion 1
    expect(assert_1.SynthUtils.toCloudFormation(stack)).toMatchSnapshot();
    expect(stack).toHaveResourceLike('AWS::IAM::Role', {
        Type: "AWS::IAM::Role",
        Properties: {
            AssumeRolePolicyDocument: {
                Statement: [{
                        Action: "sts:AssumeRole",
                        Effect: "Allow",
                        Principal: {
                            Service: "glue.amazonaws.com"
                        }
                    }],
                Version: "2012-10-17"
            },
            Description: "Service role that Glue custom ETL jobs will assume for exeuction"
        }
    }, assert_1.ResourcePart.CompleteDefinition);
});
// --------------------------------------------------------------
// Test deployment when output location is provided
// --------------------------------------------------------------
test('Test deployment with role creation', () => {
    // Stack
    const stack = new core_1.Stack();
    const _jobID = 'glueetl';
    const cfnJobProps = {
        command: {
            name: _jobID,
            pythonVersion: '3',
            scriptLocation: 's3://fakelocation/script'
        }
    };
    const _database = defaults.createGlueDatabase(stack);
    defaults.buildGlueJob(stack, {
        glueJobProps: cfnJobProps,
        outputDataStore: {
            datastoreType: defaults.SinkStoreType.S3,
            existingS3OutputBucket: new aws_s3_1.Bucket(stack, 'OutputBucket', {
                versioned: false,
                bucketName: 'outputbucket',
                encryption: aws_s3_1.BucketEncryption.S3_MANAGED,
                removalPolicy: core_1.RemovalPolicy.DESTROY
            })
        },
        database: _database,
        table: defaults.createGlueTable(stack, _database, undefined, [{
                name: "id",
                type: "int",
                comment: ""
            }], 'kinesis', { STREAM_NAME: 'testStream' })
    });
    // Assertion 1
    expect(assert_1.SynthUtils.toCloudFormation(stack)).toMatchSnapshot();
    expect(stack).toHaveResourceLike('AWS::S3::Bucket', {
        Type: 'AWS::S3::Bucket',
        Properties: {
            BucketEncryption: {
                ServerSideEncryptionConfiguration: [{
                        ServerSideEncryptionByDefault: {
                            SSEAlgorithm: "AES256"
                        }
                    }]
            },
            BucketName: "outputbucket"
        },
        UpdateReplacePolicy: "Delete",
        DeletionPolicy: "Delete"
    }, assert_1.ResourcePart.CompleteDefinition);
});
// --------------------------------------------------------------
// Test deployment when script location not provided - throw error
// --------------------------------------------------------------
test('Test deployment with role creation', () => {
    // Stack
    const stack = new core_1.Stack();
    const _jobID = 'glueetl';
    const cfnJobProps = {
        command: {
            name: _jobID,
            pythonVersion: '3'
        }
    };
    const _database = defaults.createGlueDatabase(stack);
    try {
        defaults.buildGlueJob(stack, {
            glueJobProps: cfnJobProps,
            outputDataStore: {
                datastoreType: defaults.SinkStoreType.S3,
                existingS3OutputBucket: new aws_s3_1.Bucket(stack, 'OutputBucket', {
                    versioned: false
                })
            },
            database: _database,
            table: defaults.createGlueTable(stack, _database, undefined, [{
                    name: "id",
                    type: "int",
                    comment: ""
                }], 'kinesis', { STREAM_NAME: 'testStream' })
        });
    }
    catch (error) {
        expect(error).toBeInstanceOf(Error);
        expect(error.message).toEqual('Script location has to be provided as an s3 Url location. Script location cannot be empty');
    }
});
// --------------------------------------------------------------
// Dont pass Job Command attributes and it should throw an error
// --------------------------------------------------------------
test('Test for incorrect Job Command property', () => {
    const stack = new core_1.Stack();
    try {
        const _database = defaults.createGlueDatabase(stack);
        defaults.buildGlueJob(stack, {
            glueJobProps: {},
            database: _database,
            table: defaults.createGlueTable(stack, _database, undefined, [{
                    name: "id",
                    type: "int",
                    comment: ""
                }], 'kinesis', { STREAM_NAME: 'testStream' })
        });
    }
    catch (error) {
        expect(error).toBeInstanceOf(Error);
    }
});
// --------------------------------------------------------------
// Check for CfnJob.JobCommandProperty type
// --------------------------------------------------------------
test('check for JobCommandProperty type', () => {
    const stack = new core_1.Stack();
    try {
        const _database = defaults.createGlueDatabase(stack);
        defaults.buildGlueJob(stack, {
            glueJobProps: {
                command: {
                    fakekey: 'fakevalue'
                }
            },
            database: _database,
            table: defaults.createGlueTable(stack, _database, undefined, [{
                    name: "id",
                    type: "int",
                    comment: ""
                }], 'kinesis', { STREAM_NAME: 'testStream' })
        });
    }
    catch (error) {
        expect(error).toBeInstanceOf(Error);
    }
});
//# sourceMappingURL=data:application/json;base64,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