"use strict";
/**
 *  Copyright 2021 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
const core_1 = require("@aws-cdk/core");
const iam = require("@aws-cdk/aws-iam");
const kms = require("@aws-cdk/aws-kms");
const ec2 = require("@aws-cdk/aws-ec2");
const defaults = require("../");
const assert_1 = require("@aws-cdk/assert");
require("@aws-cdk/assert/jest");
// --------------------------------------------------------------
// Test minimal deployment with no properties
// --------------------------------------------------------------
test('Test minimal deployment with no properties', () => {
    // Stack
    const stack = new core_1.Stack();
    const sagemakerRole = new iam.Role(stack, "SagemakerRole", {
        assumedBy: new iam.ServicePrincipal("sagemaker.amazonaws.com"),
    });
    // Build SageMaker Notebook Instance
    defaults.buildSagemakerNotebook(stack, {
        role: sagemakerRole,
    });
    // Assertion
    expect(assert_1.SynthUtils.toCloudFormation(stack)).toMatchSnapshot();
});
// --------------------------------------------------------------
// Test deployment with VPC
// --------------------------------------------------------------
test('Test deployment with VPC', () => {
    // Stack
    const stack = new core_1.Stack();
    const sagemakerRole = new iam.Role(stack, "SagemakerRole", {
        assumedBy: new iam.ServicePrincipal("sagemaker.amazonaws.com"),
    });
    let sagemaker;
    let vpc;
    let sg;
    // Build SageMaker Notebook Instance
    [sagemaker, vpc, sg] = defaults.buildSagemakerNotebook(stack, {
        role: sagemakerRole,
    });
    // Assertion
    expect(assert_1.SynthUtils.toCloudFormation(stack)).toMatchSnapshot();
    expect(vpc === null || vpc === void 0 ? void 0 : vpc.privateSubnets.length).toEqual(2);
    expect(vpc === null || vpc === void 0 ? void 0 : vpc.publicSubnets.length).toEqual(2);
    expect(sagemaker.instanceType).toEqual('ml.t2.medium');
    expect(sg).toBeInstanceOf(ec2.SecurityGroup);
});
// --------------------------------------------------------------
// Test deployment witout VPC
// --------------------------------------------------------------
test('Test deployment w/o VPC', () => {
    // Stack
    const stack = new core_1.Stack();
    const sagemakerRole = new iam.Role(stack, "SagemakerRole", {
        assumedBy: new iam.ServicePrincipal("sagemaker.amazonaws.com"),
    });
    // Build SageMaker Notebook Instance
    defaults.buildSagemakerNotebook(stack, {
        role: sagemakerRole,
        deployInsideVpc: false
    });
    // Assertion
    expect(assert_1.SynthUtils.toCloudFormation(stack)).toMatchSnapshot();
});
// --------------------------------------------------------------
// Test deployment in existing VPC
// --------------------------------------------------------------
test('Test deployment w/ existing VPC', () => {
    // Stack
    const stack = new core_1.Stack();
    const sagemakerRole = new iam.Role(stack, "SagemakerRole", {
        assumedBy: new iam.ServicePrincipal("sagemaker.amazonaws.com"),
    });
    // Build SageMaker Notebook Instance
    defaults.buildSagemakerNotebook(stack, {
        role: sagemakerRole,
        deployInsideVpc: true,
        sagemakerNotebookProps: {
            subnetId: 'subnet-deadbeef',
            securityGroupIds: ['sg-deadbeef']
        }
    });
    expect(stack).toHaveResource("AWS::SageMaker::NotebookInstance", {
        DirectInternetAccess: "Disabled",
        SecurityGroupIds: [
            "sg-deadbeef"
        ],
        SubnetId: "subnet-deadbeef"
    });
});
// --------------------------------------------------------------
// Test deployment with override
// --------------------------------------------------------------
test('Test deployment w/ override', () => {
    // Stack
    const stack = new core_1.Stack();
    const sagemakerRole = new iam.Role(stack, "SagemakerRole", {
        assumedBy: new iam.ServicePrincipal("sagemaker.amazonaws.com"),
    });
    const key = new kms.Key(stack, 'MyEncryptionKey');
    // Build SageMaker Notebook Instance
    defaults.buildSagemakerNotebook(stack, {
        role: sagemakerRole,
        sagemakerNotebookProps: {
            instanceType: 'ml.c4.2xlarge',
            kmsKeyId: key.keyArn
        }
    });
    expect(stack).toHaveResource("AWS::SageMaker::NotebookInstance", {
        DirectInternetAccess: "Disabled",
        InstanceType: "ml.c4.2xlarge",
        KmsKeyId: {
            "Fn::GetAtt": [
                "MyEncryptionKeyD795679F",
                "Arn"
            ]
        }
    });
});
// --------------------------------------------------------------
// Test exception
// --------------------------------------------------------------
test('Test exception', () => {
    // Stack
    const stack = new core_1.Stack();
    const sagemakerRole = new iam.Role(stack, "SagemakerRole", {
        assumedBy: new iam.ServicePrincipal("sagemaker.amazonaws.com"),
    });
    expect(() => {
        // Build SageMaker Notebook Instance
        defaults.buildSagemakerNotebook(stack, {
            role: sagemakerRole,
            deployInsideVpc: true,
            sagemakerNotebookProps: {
                subnetId: 'subnet-deadbeef',
            }
        });
    }).toThrowError();
});
//# sourceMappingURL=data:application/json;base64,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