# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['coingecko_py',
 'coingecko_py.scripts',
 'coingecko_py.swagger_generated',
 'coingecko_py.swagger_generated.swagger_client',
 'coingecko_py.swagger_generated.swagger_client.api',
 'coingecko_py.swagger_generated.swagger_client.models',
 'coingecko_py.utils']

package_data = \
{'': ['*'], 'coingecko_py': ['swagger_data/*']}

install_requires = \
['certifi>=2017.4.17,<2018.0.0',
 'python-dotenv>=0.19.2,<0.20.0',
 'python_dateutil>=2.5.3,<3.0.0',
 'requests>=2.27.1,<3.0.0',
 'six>=1.10,<2.0',
 'urllib3>=1.25.10,<2.0.0']

entry_points = \
{'console_scripts': ['generate_client = '
                     'coingecko_py.scripts.swagger:generate_client',
                     'generate_test_data = '
                     'coingecko_py.scripts.swagger:generate_test_data',
                     'generate_test_data_template = '
                     'coingecko_py.scripts.swagger:generate_test_data_template',
                     'get_api_version = '
                     'coingecko_py.scripts.github:get_api_version',
                     'get_cov = coingecko_py.scripts.github:get_cov_percent',
                     'get_cov_color = '
                     'coingecko_py.scripts.github:get_cov_color',
                     'get_data = '
                     'coingecko_py.scripts.swagger:generate_test_data',
                     'is_client_updated = '
                     'coingecko_py.scripts.github:github_specs_equal',
                     'test = tests.scripts.runner:run_tests']}

setup_kwargs = {
    'name': 'coingecko-py',
    'version': '1.0.0',
    'description': 'A python API client for the coingecko API.',
    'long_description': '# coingecko_py \n\n![Api Version](https://img.shields.io/endpoint?url=https://gist.githubusercontent.com/brycemorrow4564/e454b900fe6518d21bdd25c9508d8a64/raw/coingecko_py_apiversion__heads_master.json)\n![Api Updated](https://img.shields.io/endpoint?url=https://gist.githubusercontent.com/brycemorrow4564/e454b900fe6518d21bdd25c9508d8a64/raw/coingecko_py_clientupdated__heads_master.json)\n![Tests](https://github.com/brycemorrow4564/coingecko_py/actions/workflows/ci.yml/badge.svg)\n![Coverage](https://img.shields.io/endpoint?url=https://gist.githubusercontent.com/brycemorrow4564/e454b900fe6518d21bdd25c9508d8a64/raw/coingecko_py__heads_master.json)\n\n[![PyPi Version](https://img.shields.io/pypi/v/coingecko_py.svg)](https://pypi.org/project/coingecko_py/)\n![GitHub](https://img.shields.io/github/license/brycemorrow4564/coingecko_py)\n\n## An Advanced API Client For The Coingecko API\n\nThe base api client class and its documentation are automatically generated with [`swagger-codegen`](https://swagger.io/tools/swagger-codegen/) from the OpenAPI specification available [here](https://www.coingecko.com/api/documentations/v3/swagger.json) on the coingecko website. \n\n> The documentation for the api client can be found [here](./docs/API.md). \n\nThis ensures that all endpoints and their corresponding parameters are **100%** correct. Furthermore, \nthe "Client Updated" badge you see at the top of this README is a live check that the spec used to \ngenerate the client code matches the latest version of the spec available on the coingecko website. \nThis badge is updated once a day as a part of the CICD pipeline. \n\nAdditionally, the base api client has been extended to provide additional functionality like\n\n- Abstracting away complexities associated with server side rate limiting when sending many api requests. \n- Page range queries (bounded and unbounded). \n\n## Outline \n\n[Installation](#installation)\n\n[Usage](#usage)\n\n[API Reference](./docs/API.md)\n\n[Advanced Features - Mitigate Rate Limiting](#advanced-features---mitigate-rate-limiting)\n\n[Advanced Features - Page Range Queries](#advanced-features---page-range-queries)\n\n[Client Configuration](#client-configuration)\n\n[Summary](#summary)\n\n[Development and Testing](#development-and-testing)\n\n[License](#license)\n\n## Installation \n\nThis package is currently only available through **PyPI**. You can install \nit by running\n\n```shell\npip install coingecko_py\n```\n\n## Usage \n\nThis package exposes a single class called `CoingeckoApi`. To import and \ninitialize this class, do the following\n\n```python\nfrom coingecko_py import CoingeckoApi\ncg = CoingeckoApi()\n```\n\nCheck out the [API Reference](./docs/API.md) for more details on how\nto use this object. \n\n## Advanced Features  \n\nThis section includes usage examples for advanced features that have been added to the \nbase api client. \n\n### Advanced Features - Mitigate Rate Limiting  \n\n> Note: This functionality is available for **all** endpoints available on the base client.\n\nImagine you wanted to get price data for the last year on the top 1000 market cap coins. \n\nFirst, we get the data for the top 1000 market cap coins. Each page returns 100 results and pages are already sorted by market cap. \n```python\n# np.ravel flattens a list of lists\nimport numpy as np \ncoins = np.ravel([cg.coins_markets_get(\'usd\', page=i) for i in range(1, 11)])\n```\n\nNext, we iterate over `coins` and use each coin id to query for it\'s price data. \n\n```python\nndays = 365\nprices = dict()\nfor c in coins: \n    cid = c[\'id\']\n    prices[cid] = cg.coins_id_market_chart_get(cid, \'usd\', ndays)[\'prices\']\n```\nThe issue here is that the coingecko api performs server side rate limiting. If you are using the free tier, it\'s about 50 api calls per second. Paid tiers have higher limits, but there is still a limit. \n\nSince the above code block would be sending 1000 api requests synchronously, it is likely to fail at some point if you have a decent internet connection. In order to get around this, you would have to add error detection and call management logic. If you are writing a complex app with many api calls, this can be really annoying. \n\nThe **coingecko_py** client introduces a mechanism to queue api calls and execute a series of queued calls while performing **client side exponential backoff retries**. See [here](https://docs.aws.amazon.com/general/latest/gr/api-retries.html) for an explanation of this strategy. \n\nThis allows you to write code without worrying about rate limiting! Here is a block of code that is equivalent to the above code block that won\'t error out due to rate limiting. \n\n```python \nndays = 365\nfor c in coins: \n    cid = c[\'id\']\n    cg.coins_id_market_chart_get(cid, \'usd\', ndays, qid=cid)\nprices = cg.execute_queued()\nprices = {k: v[\'prices\'] for k, v in prices.items()}\n```\n\nThe key differences here are \n\n- The inclusion of the `qid` keyword argument in the api call signature. \n  - `qid` stands for queue id. \n  - Whenever `qid` is present as a keyword argument in an api call, the client will queue the call instead of executing it. \n  - `qid` can be used as a lookup key for the result of this api call once it is executed. \n  \n- The line containing the api call (`cg.coins_id_market_chart_get(...)`) does not return anything. \n  - Whenever `qid` is not a kwarg, an api call behaves exactly the same as the base api client. \n  - Whenever `qid` is a kwarg, an api call returns nothing, as it was queued. \n  \n- The function `execute_queued` must be invoked in order to execute all queued calls. \n  - It internally deals with rate limiting. \n  - It\'s return value is a dictionary where the keys are the `qid` values from queued calls and the values are the data parsed from responses of the corresponding api calls. \n  - If `execute_queued` is successful, the internal call queue is cleared. \n    - So if you called `execute_queued` on line 1 then again on line 2, the second call would return an empty dictionary. \n\nThese two blocks of code both produce a dictionary `prices` with the same exact structure (assuming the first code block doesn\'t error out because of rate limiting). \n\n```python \nprices = {\n    \'bitcoin\': {\n        \'prices\': [\n            [1610236800000, 40296.5290038294],\n            [1610323200000, 38397.895985418174],\n            [1610409600000, 35669.90668663349],\n            ...\n        ]\n    }, \n    \'ethereum\': {\n        \'prices\': [\n            [1610236800000, 1282.979575527323],\n            [1610323200000, 1267.7310031512136],\n            [1610409600000, 1092.9143378806064],\n            ...\n        ]\n    },\n    ...\n} \n```\n\nThis approach to API design was loosely inspired by [dask\'s][https://docs.dask.org/en/stable/] approach to lazy execution of a sequence of operations on dataframes.\n\n### Advanced Features - Page Range Queries \n\n> Note: This functionality is available for **all** endpoints the base client that support paging. \n\nThe coingecko api has a number of endpoints that support pagination. Pagination is a common api feature where you can request a specific page of data from an api. This is often necessary as some data objects are too large to return in a single api response. If you want all the data for a particular api call you are executing, you must request data from all pages. \n\nHere is an example that uses the client to query for a single page of data\n\n```python\ncg.coins_id_tickers_get(\'bitcoin\', page=2, per_page=50)\n```\n\nPage range queries allow you to request a range of pages in a **single** client call. The \napi client supports **bounded** and **unbounded** page range queries. \n\n- **Bounded** queries request pages over a fully defined range `[page_start, page_end]`. \n- **Unbounded** queries only require the specification of `page_start` and will return \n  data from all available pages from `page_start` onwards. \n\nFor the code blocks below, let\'s assume we magically know there are 100 data pages for the `coins_id_tickers_get` endpoint for the given set of parameters. In reality, if you wanted \nto determine the number of data pages for a client call, you would need to make the call, \ninspect the HTTP headers, perform a calculation to determine the total number of pages, \nthen loop over the page range and make a request per each page. \n\nHere is an example of doing pagination manually using the base api client functionality \n\n```python \ndata = []\nfor i in range(1, 101):\n    res = cg.coins_id_tickers_get(\'bitcoin\', page=i)\n    data.append(res)\n```\n\nHere is an example of doing a **bounded** page range query with the extended client.\n\n```python \ncg.coins_id_tickers_get(\'bitcoin\', qid="data", page_start=1, page_end=100)\ndata = cg.execute_queued()[\'data\']\n```\n\nHere is an example of doing an **unbounded** page range query with the extended client.\n\n```python \ncg.coins_id_tickers_get(\'bitcoin\', qid="data", page_start=1)\ndata = cg.execute_queued()[\'data\']\n```\n\nAll code blocks will produce equivalent output. The return value of a page range query is a list of response data from each individual api call. So `data[0]` contains the result for page 1, `data[49]` contains the result for page 50.\n\nIt\'s important to note that `qid` must be included as a keyword argument for page range queries. \nThus, page range queries will also automatically deal with rate limiting as detailed in the \n[rate limiting](#advanced-features---mitigate-rate-limiting) section. \n\n## Client Configuration\n\nThe extended client supports multiple configuration options which impact its behavior. \n\n| Kwarg | Default | Description | \n| --- | --- | --- |\n| exp_limit | `8` | Max exponent (2<sup>exp_limit</sup>) for exponential backoff retries |\n| progress_interval | `10` | Min percentage interval at which to log progress of queued api calls |\n| log_level | `logging.INFO` | python [logging](https://docs.python.org/3/library/logging.html) log level for client log messages |\n\nThe API client doesn\'t print any messages, but has logs at the following levels. \n- 10 (`logging.DEBUG`) will provide logs about internal state of client. \n- 20 (`logging.INFO`) progress logs and other useful info exists at this level. \n- 30 (`logging.WARNING`) useful warnings. I don\'t recommend any level higher than this. \nSee [here](https://docs.python.org/3/library/logging.html#levels) for more info on log levels. \n\nHere\'s an example of how to configure the client with non-default values. \n```python \ncg = CoingeckoApi(log_level=10, exp_limit=6, progress_interval=5)\n```\n\n## Summary \n\nA quick summary of the functionality offered by this package\n\n- It\'s base api client is automatically generated, ensuring correctness. \n  - It\'s functionality is described in the [API Reference](./docs/API.md). \n- It\'s extra features are accessible in the following ways \n  - `cg.execute_queued` is the only public method added to the client. It takes no input arguments and returns a dictionary that maps `qid` values to the corresponding queued api call. \n  - You can queue api calls by include the keyword argument `qid` in a client call. When you include the kwarg `qid` the function call does not return anything (as it was queued for later execution). \n  - Queued calls benefit from the clients internal strategy for mitigating server side rate limiting. \n  - Page range queries allow you to request a range of data pages in a single client call. \n    - If `page_start` and `page_end` are both defined, it will return all data pages in range. \n    - If `page_start` is defined and `page_end` is not, it will return all data pages from `page_start` onwards.\n    - Page range queries must be queued (include `qid` in their call signature). \n\n## Development and Testing \n\nThis package is packaged with [poetry](https://python-poetry.org/)\n\nIf you have poetry installed, you can perform the following steps to set up the development environment. \n\n```shell \ngit clone https://github.com/brycemorrow4564/coingecko_py.git\ncd coingecko_py\npoetry shell \npoetry update \npoetry install \n```\n\nIf you want to run the tests (within the dev environment), do the following \n```shell\npoetry run test \n```\n\n## License\n[MIT](https://choosealicense.com/licenses/mit/)',
    'author': 'Bryce Morrow',
    'author_email': 'brycemorrow4564@gmail.com',
    'maintainer': 'Bryce Morrow',
    'maintainer_email': 'brycemorrow4564@gmail.com',
    'url': 'https://github.com/brycemorrow4564/coingecko_py',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'entry_points': entry_points,
    'python_requires': '>=3.7,<4.0',
}


setup(**setup_kwargs)
