from simulation.avatar.avatar_wrapper import AvatarWrapper
from simulation.avatar.avatar_appearance import AvatarAppearance


class AvatarManager(object):
    """
    Stores all game avatars.
    """

    def __init__(self):
        self.avatars_by_id = {}

    def add_avatar(self, player_id, worker_url, location):
        avatar = AvatarWrapper(player_id, location, worker_url,
                               AvatarAppearance("#000", "#ddd", "#777", "#fff"))
        self.avatars_by_id[player_id] = avatar
        return avatar

    def get_avatar(self, user_id):
        return self.avatars_by_id[user_id]

    def remove_avatar(self, user_id):
        del self.avatars_by_id[user_id]

    @property
    def avatars(self):
        return list(self.avatars_by_id.viewvalues())

    @property
    def active_avatars(self):
        return [player for player in self.avatars]

    def players_update(self):
        """
        To be called on a required update of players in the front-end of the game. This
        is a wrapper for player_list.

        :return: A dictionary that holds the list generated by player_list.
        """

        def _generate_player_list():
            """
            Used in the serialising process to provide updates of avatars to the front end

            :return A list with each element containing a dictionary of each player data.
            """
            players = []

            for player in self.active_avatars:
                players.append({
                    'id': player.player_id,
                    'score': player.score,
                    'health': player.health,
                    'location': {
                        'x': player.location.x,
                        'y': player.location.y,
                    },
                    'orientation': player.orientation
                })

            return players

        return {
            'players': _generate_player_list()
        }
