#!/usr/bin/env ts-node
"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.invokeDotWasm = void 0;
/*! Copyright [Amazon.com](http://amazon.com/), Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0 */
const path = require("node:path");
const execa = require("execa"); // eslint-disable-line @typescript-eslint/no-require-imports
const utils_1 = require("./utils");
const fonts_1 = require("../../fonts");
/**
 * Invoke `dot-wasm` in shell with image support.
 * @param inputFile - The input dot file
 * @param absoluteAssetPaths - List of absolute asset paths to provide to dot-wasm
 * @returns The generated svg string
 *
 * @internal
 */
async function invokeDotWasm(inputFile, absoluteAssetPaths) {
    const invoker = path.resolve(__dirname, "dot-wasm-invoker.mjs");
    const options = {
        images: await Promise.all(absoluteAssetPaths.map((v) => (0, utils_1.resolveDotWasmImage)(v))),
    };
    const encodedOptions = Buffer.from(JSON.stringify(options), "utf-8").toString("base64");
    const response = await execa.node(invoker, [inputFile, encodedOptions], {
        shell: true,
    });
    logDotWasmErrors(response.stderr);
    return (0, utils_1.resolveSvg)(response.stdout);
}
exports.invokeDotWasm = invokeDotWasm;
/** Filtered logging of dot-wasm stderr */
function logDotWasmErrors(stderr) {
    stderr.split("\n").forEach((line) => {
        // Ignore font mapping issues - they are resolved by svg rendering itself
        if (!line.startsWith(`Warning: no hard-coded metrics for '${fonts_1.FONT_FAMILY}'`)) {
            console.warn("[dot-wasm]", line);
        }
    });
}
//# sourceMappingURL=data:application/json;base64,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