#!/usr/bin/env ts-node
"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.resolveSvg = exports.resolveDotWasmImage = void 0;
/*! Copyright [Amazon.com](http://amazon.com/), Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0 */
const fs = require("node:fs");
const node_util_1 = require("node:util");
const aws_arch_1 = require("../../../../aws-arch");
const svgson = require("svgson");
const svg_1 = require("../../utils/svg");
const readFile = (0, node_util_1.promisify)(fs.readFile);
/**
 * Resolve image path to [dot-wasm](https://hpcc-systems.github.io/hpcc-js-wasm/classes/graphviz.Graphviz.html) image
 * struct which requires width and height dimensions.
 */
async function resolveDotWasmImage(relativePath) {
    const absolutePath = aws_arch_1.AwsArchitecture.resolveAssetPath(relativePath);
    const svgString = await readFile(absolutePath, { encoding: "utf-8" });
    const dimensions = await (0, svg_1.extractSvgDimensions)(svgString);
    return {
        path: absolutePath,
        width: dimensions.width,
        height: dimensions.height,
    };
}
exports.resolveDotWasmImage = resolveDotWasmImage;
/**
 * Resolve [dot-wasm](https://hpcc-systems.github.io/hpcc-js-wasm/classes/graphviz.Graphviz.html) generated svg.
 *
 * **dot-wasm** (as well as Graphviz) generate non-desirable svg state which this function resolves:
 * 1. Unescape/Decode all html entities (eg: `&#45;` => `-`)
 * 2. Reconcile the `viewBox` size to match graph content
 * 3. Remove width and height from root svg, to ensure pure viewBox is utilized without side effects
 * 4. Resolve font **config** values to proper html/svg font-family and styles
 * 5. Inline all svg images via definitions
 */
async function resolveSvg(svgString) {
    const svg = await svgson.parse(svgString);
    (0, svg_1.unescapeSvgTextValues)(svg);
    // The resulting svg from graphviz (both dot and dot-wasm) have incorrect viewBox and width/height attributes
    // viewBox="0.00 0.00 494.00 508.00" => viewBox="0 0 2058.33498 2116.66836"
    // from container with transform="scale(4.16667 4.16667) rotate(0) translate(4 504)"
    (0, svg_1.reconcileViewBox)(svg);
    (0, svg_1.addGraphFontCssStyles)(svg);
    await (0, svg_1.resolveSvgAwsArchAssetImagesInline)(svg);
    return svgson.stringify(svg);
}
exports.resolveSvg = resolveSvg;
//# sourceMappingURL=data:application/json;base64,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