"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.CustomResourceNode = exports.ResourceNode = exports.CfnResourceNode = exports.ImageNode = exports.Node = void 0;
const startCase = require("lodash.startcase"); // eslint-disable-line @typescript-eslint/no-require-imports
const words = require("lodash.words"); // eslint-disable-line @typescript-eslint/no-require-imports
const Dot = require("ts-graphviz");
const wordWrap = require("word-wrap"); // eslint-disable-line @typescript-eslint/no-require-imports
const resource_images_1 = require("../../utils/resource-images");
const theme_1 = require("../theme");
/** Diagram label line height */
const LABEL_LINE_HEIGHT = 0.23;
/** Diagram label line max chars */
const LABEL_LINE_MAX_CHARS = 15;
/** Diagram label max number of lines */
const LABEL_MAX_LINES = 5;
/** Marshalls a label to contain length, output multi-line, etc for better rendering */
function marshallLabelForRendering(original) {
    let label = words(original).join(" ");
    label = wordWrap(label, {
        width: LABEL_LINE_MAX_CHARS,
        trim: true,
        indent: "",
    });
    const splitLabel = label.split("\n");
    const lines = splitLabel.slice(0, LABEL_MAX_LINES);
    // Ellipse last line if dropped lines
    if (splitLabel.length > lines.length) {
        lines[lines.length - 1] = lines[lines.length - 1] + "...";
    }
    label = lines
        .map((line) => {
        line = startCase(line).replace(/ /g, "");
        if (line.length > LABEL_LINE_MAX_CHARS) {
            return line.substring(0, LABEL_LINE_MAX_CHARS) + "...";
        }
        return line;
    })
        .join("\n");
    return { original, label, lines: lines.length };
}
/**
 * Node class defines a {@link Graph.Node} based diagram {@link Dot.Node}
 * @internal
 */
class Node extends Dot.Node {
    /** Get the label attribute for this node */
    get label() {
        return this.attributes.get("label");
    }
    /** @internal */
    constructor(node) {
        super(`node_${node.uuid}`);
        this.graphNode = node;
        this.attributes.set("label", marshallLabelForRendering(node.id).label);
        this.attributes.set("comment", `nodeType:${node.nodeType}` + (node.cfnType ? `(${node.cfnType})` : ""));
    }
}
exports.Node = Node;
/**
 * ImageNode class extends {@link Node} with support for rendering diagram images.
 * @internal
 */
class ImageNode extends Node {
    /** @internal */
    constructor(node, image) {
        super(node);
        // If image not defined, treat as regular node
        if (image) {
            this.attributes.apply(theme_1.GraphTheme.instance.imageNode);
            this.attributes.set("image", image);
            this.resize();
        }
    }
    /** Get `image` attribute */
    get image() {
        return this.attributes.get("image");
    }
    /** Resizes the node based on image and label dimensions */
    resize(baseHeight) {
        if (baseHeight == null) {
            baseHeight = (this.attributes.get("height") || 1);
        }
        const image = this.image;
        if (image) {
            const labelLines = this.label.split("\n").length;
            this.attributes.set("labelloc", "b");
            this.attributes.set("height", baseHeight + labelLines * LABEL_LINE_HEIGHT);
        }
        else {
            this.attributes.set("labelloc", "c");
            this.attributes.set("height", baseHeight);
        }
    }
}
exports.ImageNode = ImageNode;
/**
 * CfnResourceNode class defines a {@link Dot.Node} based on a {@link Graph.CfnResourceNode}
 * @internal
 */
class CfnResourceNode extends ImageNode {
    /** @internal */
    constructor(node) {
        super(node, (0, resource_images_1.resolveCfnResourceImage)(node));
        this.attributes.apply(theme_1.GraphTheme.instance.cfnResourceNode);
        this.resize(theme_1.GraphTheme.instance.cfnResourceNode.height === ""
            ? undefined
            : theme_1.GraphTheme.instance.cfnResourceNode.height);
        if (node.isImport) {
            this.attributes.apply({
                style: "filled,dotted",
                penwidth: 1,
                fontcolor: (theme_1.GraphTheme.instance.awsTheme?.text.tertiary ||
                    "#55555"),
                color: ((theme_1.GraphTheme.instance.awsTheme?.text.tertiary || "#55555") +
                    "33"),
                fillcolor: ((theme_1.GraphTheme.instance.awsTheme?.text.tertiary || "#55555") +
                    "1A"), // 10%
            });
        }
    }
}
exports.CfnResourceNode = CfnResourceNode;
/**
 * ResourceNode class defines a {@link Dot.Node} based on a {@link Graph.ResourceNode}
 * @internal
 */
class ResourceNode extends ImageNode {
    /** @internal */
    constructor(node) {
        const image = (0, resource_images_1.resolveResourceImage)(node);
        super(node, image);
        this.attributes.apply(theme_1.GraphTheme.instance.resourceNode);
        this.resize(theme_1.GraphTheme.instance.resourceNode.height === ""
            ? undefined
            : theme_1.GraphTheme.instance.resourceNode.height);
    }
}
exports.ResourceNode = ResourceNode;
/**
 * CustomResourceNode class defines a {@link Dot.Node} based on a {@link Graph.Node} for a *custom resource*
 * @internal
 */
class CustomResourceNode extends ImageNode {
    /** @internal */
    constructor(node) {
        super(node, (0, resource_images_1.resolveCustomResourceImage)(node));
    }
}
exports.CustomResourceNode = CustomResourceNode;
//# sourceMappingURL=data:application/json;base64,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