"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.getBaseTheme = exports.GraphTheme = void 0;
/*! Copyright [Amazon.com](http://amazon.com/), Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0 */
const aws_arch_1 = require("../../../../aws-arch");
const cloneDeep = require("lodash.clonedeep"); // eslint-disable-line @typescript-eslint/no-require-imports
const theme_1 = require("./theme");
const fonts_1 = require("../../fonts");
const DEFAULT_RENDERING = {
    resourceIconMin: theme_1.GraphThemeRenderingIconTarget.DATA,
    resourceIconMax: theme_1.GraphThemeRenderingIconTarget.CATEGORY,
    cfnResourceIconMin: theme_1.GraphThemeRenderingIconTarget.DATA,
    cfnResourceIconMax: theme_1.GraphThemeRenderingIconTarget.SERVICE,
};
/** GraphTheme class is the implementation of diagram graph based theme */
class GraphTheme {
    /** Initializes the theme based on {@link aws_arch.Theme} definition */
    static init(config) {
        if (config == null || typeof config === "string") {
            this._instance = new GraphTheme(generateGraphThemeFromAwsTheme(aws_arch_1.aws_arch.resolveTheme(config)));
        }
        else if (Object.keys(config).length <= 2) {
            const { theme, rendering } = config;
            this._instance = new GraphTheme(generateGraphThemeFromAwsTheme(aws_arch_1.aws_arch.resolveTheme(theme), rendering));
        }
        else {
            this._instance = new GraphTheme(config);
        }
        return this.instance;
    }
    /** Get the current singleton instance for the theme being utilized for the diagram */
    static get instance() {
        if (this._instance == null) {
            throw new Error("Must init GraphTheme before requesting instance");
        }
        return this._instance;
    }
    /** @internal */
    constructor(theme) {
        this.rendering = theme.rendering;
        this.graph = theme.graph;
        this.subgraph = theme.subgraph;
        this.cluster = theme.cluster;
        this.node = theme.node;
        this.edge = theme.edge;
        this.cloud = theme.cloud;
        this.stage = theme.stage;
        this.stack = theme.stack;
        this.nestedStack = theme.nestedStack;
        this.imageNode = theme.imageNode;
        this.cfnResourceNode = theme.cfnResourceNode;
        this.resourceNode = theme.resourceNode;
        this.childLink = theme.childLink;
        this.referenceLink = theme.referenceLink;
        this.dependencyLink = theme.dependencyLink;
        this.awsTheme = theme.awsTheme;
    }
}
exports.GraphTheme = GraphTheme;
/** Get the base theme */
function getBaseTheme(rendering) {
    return cloneDeep({
        rendering: {
            ...DEFAULT_RENDERING,
            ...rendering,
        },
        graph: GRAPH_ATTRIBUTES,
        subgraph: SUBGRAPH_ATTRIBUTES,
        cluster: CLUSTER_ATTRIBUTES,
        node: NODE_ATTRIBUTES,
        edge: EDGE_ATTRIBUTES,
        cloud: CLOUD_ATTRIBUTES,
        stage: STAGE_ATTRIBUTES,
        stack: STACK_ATTRIBUTES,
        nestedStack: NESTED_STACK_ATTRIBUTES,
        imageNode: IMAGE_NODE_ATTRIBUTES,
        cfnResourceNode: CFN_RESOURCE_NODE_ATTRIBUTES,
        resourceNode: RESOURCE_NODE_ATTRIBUTES,
        childLink: CHILD_LINK_ATTRIBUTES,
        referenceLink: REFERENCE_LINK_ATTRIBUTES,
        dependencyLink: DEPENDENCY_LINK_ATTRIBUTES,
    });
}
exports.getBaseTheme = getBaseTheme;
/** Generate {@link IGraphTheme} from {@link aws_arch.Theme} */
function generateGraphThemeFromAwsTheme(awsTheme, rendering) {
    const theme = getBaseTheme(rendering);
    Object.assign(theme, {
        awsTheme,
    });
    Object.assign(theme.graph, {
        bgcolor: awsTheme.backgrounds.base,
        fontcolor: awsTheme.text.default,
    });
    Object.assign(theme.node, {
        fontcolor: awsTheme.text.primary,
    });
    Object.assign(theme.edge, awsArrowToEdgeAtts(awsTheme.arrows.default));
    // containers
    Object.assign(theme.cloud, awsGroupToSubgraphAtts(awsTheme.groups.cloud));
    Object.assign(theme.stage, awsGroupToSubgraphAtts(awsTheme.groups.awsAccount, true));
    Object.assign(theme.stack, awsGroupToSubgraphAtts(awsTheme.groups.genericAlt, true));
    Object.assign(theme.nestedStack, awsGroupToSubgraphAtts(awsTheme.groups.genericAlt, true));
    // nodes
    Object.assign(theme.cfnResourceNode, {
        color: awsTheme.text.secondary,
        fontcolor: awsTheme.text.secondary,
    });
    Object.assign(theme.resourceNode, {
        color: awsTheme.text.primary,
        fontcolor: awsTheme.text.primary,
    });
    Object.assign(theme.childLink, awsArrowToEdgeAtts(awsTheme.arrows.child));
    Object.assign(theme.referenceLink, awsArrowToEdgeAtts(awsTheme.arrows.reference));
    Object.assign(theme.dependencyLink, awsArrowToEdgeAtts(awsTheme.arrows.dependency));
    return theme;
}
/** Base graph attributes */
const GRAPH_ATTRIBUTES = {
    ...fonts_1.GraphFonts.REGULAR,
    center: true,
    compound: true,
    concentrate: true,
    dpi: 300,
    fontcolor: "#222222",
    fontnames: "ps",
    fontsize: 14,
    forcelabels: true,
    labelloc: "tc",
    nodesep: 0.6,
    pad: "%2,%1",
    // @ts-ignore
    rankdir: "TB",
    ranksep: 0.75,
    ratio: "compress",
    remincross: true,
    size: "%1024,%1024!",
    splines: "ortho",
    stylesheet: fonts_1.FONT_STYLESHEET,
};
/** Base subgraph attributes */
const SUBGRAPH_ATTRIBUTES = {
    ...fonts_1.GraphFonts.LIGHT_ITALIC,
    labelloc: "tc",
    fontsize: 12,
    style: "rounded,solid",
};
/** Base cluster attributes */
const CLUSTER_ATTRIBUTES = {
    ...fonts_1.GraphFonts.LIGHT,
    labelloc: "tc",
    fontsize: 12,
    style: "rounded,dashed",
};
/** Base cloud container attributes */
const CLOUD_ATTRIBUTES = {
    ...fonts_1.GraphFonts.BOLD,
    style: "solid",
};
/** Base stage attributes */
const STAGE_ATTRIBUTES = {
    ...fonts_1.GraphFonts.BOLD_ITALIC,
    style: "dashed",
    margin: 6,
    rank: "same",
};
/** Base stack attributes */
const STACK_ATTRIBUTES = {
    ...fonts_1.GraphFonts.LIGHT,
    style: "solid,bold,filled",
    fillcolor: "#5A6B861A",
    margin: 10,
};
/** Base nested stack attributes */
const NESTED_STACK_ATTRIBUTES = {
    ...fonts_1.GraphFonts.LIGHT_ITALIC,
    style: "solid,filled",
    fillcolor: "#5A6B861A",
    margin: 6,
};
/** Base node attributes */
const NODE_ATTRIBUTES = {
    ...fonts_1.GraphFonts.REGULAR,
    shape: "box",
    style: "solid",
    fixedsize: false,
    width: 0.25,
    height: 0.25,
    labelloc: "c",
    imagescale: true,
    fontsize: 11,
    penwidth: 0.25,
};
/** Base image based node attributes */
const IMAGE_NODE_ATTRIBUTES = {
    shape: "box",
    style: "solid,rounded",
    fixedsize: true,
    width: 1,
    height: 1,
    labelloc: "b",
    imagescale: true,
    imagepos: "tc",
    penwidth: 0,
    fillcolor: "transparent",
};
/** Base cfn resource node attributes */
const CFN_RESOURCE_NODE_ATTRIBUTES = {
    width: 1,
    height: 1,
    fixedsize: true,
    style: "solid,rounded",
    color: "#999999",
    fontcolor: "#999999",
};
/** Base resource node attributes */
const RESOURCE_NODE_ATTRIBUTES = {
    width: 1,
    height: 1,
    fixedsize: true,
    color: "#666666",
    fontcolor: "#666666",
};
/** Base edge attributes */
const EDGE_ATTRIBUTES = {
    ...fonts_1.GraphFonts.LIGHT_ITALIC,
    dir: "both",
    color: "#545B64",
    penwidth: 0.75,
    arrowhead: "dot",
    arrowtail: "dot",
    arrowsize: 0.5,
    fontsize: 9,
    style: "solid",
};
/** Base child link attributes */
const CHILD_LINK_ATTRIBUTES = {
    penwidth: 1,
    arrowhead: "none",
    arrowtail: "normal",
    arrowsize: 1,
    style: "solid",
};
/** Base reference link attributes */
const REFERENCE_LINK_ATTRIBUTES = {
    penwidth: 1,
    arrowhead: "none",
    arrowtail: "normal",
    arrowsize: 0.75,
    style: "solid",
};
/** Base dependency link attributes */
const DEPENDENCY_LINK_ATTRIBUTES = {
    penwidth: 0.75,
    arrowhead: "dot",
    arrowtail: "odot",
    arrowsize: 0.75,
    style: "dotted",
};
/** Convert {@link aws_arch.GroupFormat} to {@link Dot.SubgraphAttributesObject} */
function awsGroupToSubgraphAtts(group, ignoreStyle = false) {
    const attributes = {};
    if (!ignoreStyle) {
        if (group.borderStyle === "none") {
            attributes.style = "solid";
            attributes.penwidth = 0;
        }
        else {
            attributes.style = group.borderStyle;
        }
    }
    attributes.color = group.color || "transparent";
    attributes.bgcolor = group.bgcolor || "transparent";
    attributes.fillcolor = group.bgcolor || "transparent";
    attributes.pencolor = (group.borderColor || "transparent");
    if (group.color && group.color !== "transparent") {
        attributes.fontcolor = group.color;
    }
    return attributes;
}
/** Convert {@link aws_arch.ArrowFormat} to {@link Dot.EdgeAttributesObject} */
function awsArrowToEdgeAtts(arrow) {
    return {
        color: arrow.color,
        fontcolor: arrow.color,
        arrowtail: arrow.tail,
        arrowhead: arrow.head,
        penwidth: arrow.width,
        style: arrow.style,
    };
}
//# sourceMappingURL=data:application/json;base64,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