"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.unescapeSvgTextValues = exports.parseSvgTransformScale = exports.stringifySvgViewBox = exports.parseSvgViewBox = exports.getSvgRootContainer = exports.reconcileViewBox = exports.decodeSvgDataUrl = exports.encodeSvgFileDataUrl = exports.encodeHtmlDataUrl = exports.encodeDataUrl = exports.resolveSvgAwsArchAssetImagesInline = exports.addGraphFontCssStyles = exports.extractSvgDimensions = exports.convertSvg = exports.convertSvgImageDefsFromSvgToPng = void 0;
/*! Copyright [Amazon.com](http://amazon.com/), Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0 */
const aws_arch_1 = require("../../../aws-arch");
const fs = require("fs-extra");
const he = require("he"); // eslint-disable-line @typescript-eslint/no-require-imports
const sharp = require("sharp"); // eslint-disable-line @typescript-eslint/no-require-imports
const svgson = require("svgson");
const traverse = require("traverse"); // eslint-disable-line @typescript-eslint/no-require-imports
const fonts_1 = require("../fonts");
const XLINK_HREF = "xlink:href";
const DATAURL_SVG_BASE64 = "data:image/svg+xml;base64,";
const DATAURL_PNG_BASE64 = "data:image/png;base64,";
/**
 * Convert svg image definition from svg to png.
 *
 * This is necessary before able to convert full svg to png, as without this
 * step the nested svg images are not rendered in final png.
 * @param svgString - SVG string to convert
 * @returns Returns the converted SVG string
 * @internal
 */
async function convertSvgImageDefsFromSvgToPng(svgString) {
    // https://github.com/lovell/sharp/issues/2844
    const svg = await svgson.parse(svgString);
    const defs = (svg.children.find((child) => child.name === "defs")?.children || []).filter((def) => {
        return (def.name === "image" &&
            def.attributes[XLINK_HREF]?.startsWith(DATAURL_SVG_BASE64));
    });
    for (const def of defs) {
        const assetKey = def.attributes.id;
        const png = sharp(await fs.readFile(aws_arch_1.AwsArchitecture.resolveAssetPath(aws_arch_1.AwsArchitecture.formatAssetPath(assetKey, "png"))), { limitInputPixels: false });
        const pngBuffer = await png
            .resize({
            width: 128,
        })
            .toBuffer();
        def.attributes[XLINK_HREF] = encodeDataUrl(pngBuffer, DATAURL_PNG_BASE64);
    }
    return svgson.stringify(svg);
}
exports.convertSvgImageDefsFromSvgToPng = convertSvgImageDefsFromSvgToPng;
/**
 * Convert svg value to a given output file
 * @param svgString - SVG to convert
 * @param outputFile - The output file to generate from svg, the format is inferred from file extension
 * @internal
 */
async function convertSvg(svgString, outputFile) {
    const resolvedSvg = await convertSvgImageDefsFromSvgToPng(svgString);
    await sharp(Buffer.from(resolvedSvg), { limitInputPixels: false })
        .trim({ background: "transparent" })
        .toFile(outputFile);
}
exports.convertSvg = convertSvg;
/**
 * Extract SVG dimensions (width / height)
 * @internal
 */
async function extractSvgDimensions(svgString) {
    const svg = await svgson.parse(svgString);
    return {
        width: svg.attributes.width,
        height: svg.attributes.height,
    };
}
exports.extractSvgDimensions = extractSvgDimensions;
/**
 * Add graph font css styles to svg
 * @internal
 */
function addGraphFontCssStyles(svg) {
    svg.children.unshift({
        name: "style",
        type: "element",
        value: "",
        attributes: {},
        children: [
            {
                name: "",
                type: "text",
                attributes: {},
                children: [],
                value: fonts_1.FONT_CSS_CLASSES,
            },
        ],
    });
}
exports.addGraphFontCssStyles = addGraphFontCssStyles;
/**
 * Resolve SVG image paths to inline base64 **Data URLs**.
 * @internal
 */
async function resolveSvgAwsArchAssetImagesInline(svg) {
    const imageDefs = new Map();
    svg = traverse(svg).forEach(function (x) {
        if (typeof x === "object" && x?.type === "element") {
            const node = x;
            if (node.name !== "image") {
                return;
            }
            const xlinkHref = node.attributes[XLINK_HREF];
            const isAssetPath = xlinkHref &&
                xlinkHref.length &&
                !(xlinkHref.startsWith("http") ||
                    (xlinkHref.startsWith("/") &&
                        !xlinkHref.startsWith(aws_arch_1.AwsArchitecture.assetDirectory)));
            if (isAssetPath) {
                const { width, height, value, [XLINK_HREF]: assetPath, ...attributes } = node.attributes;
                const id = aws_arch_1.AwsArchitecture.parseAssetPath(assetPath).assetKey;
                if (!imageDefs.has(id)) {
                    imageDefs.set(id, {
                        type: "element",
                        name: "image",
                        value,
                        children: [],
                        attributes: {
                            id,
                            width,
                            height,
                            [XLINK_HREF]: aws_arch_1.AwsArchitecture.resolveAssetPath(assetPath),
                        },
                    });
                }
                const useDefNode = {
                    type: "element",
                    name: "use",
                    value,
                    children: [],
                    attributes: {
                        ...attributes,
                        [XLINK_HREF]: `#${id}`,
                    },
                };
                this.update(useDefNode, true);
            }
        }
    });
    for (const [, imageDef] of imageDefs.entries()) {
        const href = imageDef.attributes[XLINK_HREF];
        imageDef.attributes[XLINK_HREF] = await encodeSvgFileDataUrl(href);
    }
    svg.children.unshift({
        type: "element",
        name: "defs",
        value: "",
        attributes: {},
        children: Array.from(imageDefs.values()),
    });
}
exports.resolveSvgAwsArchAssetImagesInline = resolveSvgAwsArchAssetImagesInline;
/**
 * Encode buffer as base64 encoded **Data URL**
 * @internal
 */
function encodeDataUrl(buffer, prefix) {
    return prefix + buffer.toString("base64");
}
exports.encodeDataUrl = encodeDataUrl;
/**
 * Encode string as html and base64 encoded **Data URL**
 * @internal
 */
function encodeHtmlDataUrl(data, prefix) {
    return encodeDataUrl(Buffer.from(unescape(encodeURIComponent(data)), "utf-8"), prefix);
}
exports.encodeHtmlDataUrl = encodeHtmlDataUrl;
/**
 * Encode SVG file as base64 encoded **Data URL**
 * @internal
 */
async function encodeSvgFileDataUrl(svgFile) {
    const svgXml = await fs.readFile(svgFile, { encoding: "utf-8" });
    return encodeHtmlDataUrl(svgXml, DATAURL_SVG_BASE64);
}
exports.encodeSvgFileDataUrl = encodeSvgFileDataUrl;
/**
 * Decode SVG base64 encoded **Data URL** to string
 * @internal
 */
function decodeSvgDataUrl(svgDataUrl) {
    svgDataUrl = svgDataUrl.replace(DATAURL_SVG_BASE64, "");
    return decodeURIComponent(escape(Buffer.from(svgDataUrl, "base64").toString("utf-8")));
}
exports.decodeSvgDataUrl = decodeSvgDataUrl;
const CSS_TRANSFORM_SCALE = /scale\((?<scale>[^)]+)\)/i;
/**
 * Reconcile svg viewBox attribute based on root container ([g](https://developer.mozilla.org/en-US/docs/Web/SVG/Element/g)) scale.
 *
 * This will modify the viewbox and root container in place.
 * @param svg Svg to reconcile
 * @throws Error if svg does not define `viewBox` attribute
 * @internal
 */
function reconcileViewBox(svg) {
    const viewBox = parseSvgViewBox(svg);
    if (viewBox == null || !viewBox.width || !viewBox.height) {
        throw new Error("Svg `viewBox` undefined or does not define dimensions");
    }
    // drop width/height to ensure only reconciled viewbox is used
    delete svg.attributes.width;
    delete svg.attributes.height;
    const container = getSvgRootContainer(svg);
    if (container == null) {
        return;
    }
    // let [x, y, width, height] = viewBox.split(" ").map((v) => parseFloat(v)) as [number, number, number, number];
    let scale = parseSvgTransformScale(container);
    if (scale == null) {
        return;
    }
    let scaledViewBox = {
        ...viewBox,
        width: viewBox.width * (scale[0] || 1),
        height: viewBox.height * (scale[1] || scale[0] || 1),
    };
    // Max allowed by sharp: https://github.com/lovell/sharp/blob/2c465282699432299c478ba00ab825e07d9bdab0/src/pipeline.cc#L288
    const MAX_SVG = 10000; // 32767 is max width & height in sharp, but capping at 10k for perf and downscale huge diagrams
    if (scaledViewBox.width && scaledViewBox.width > MAX_SVG) {
        const downscale = MAX_SVG / scaledViewBox.width;
        scaledViewBox = {
            ...scaledViewBox,
            width: scaledViewBox.width * downscale,
            height: scaledViewBox.height
                ? scaledViewBox.height * downscale
                : undefined,
        };
        if (scale[0])
            scale[0] *= downscale;
        if (scale[1])
            scale[1] *= downscale;
    }
    if (scaledViewBox.height && scaledViewBox.height > MAX_SVG) {
        const downscale = MAX_SVG / scaledViewBox.height;
        scaledViewBox = {
            ...scaledViewBox,
            height: scaledViewBox.height * downscale,
            width: scaledViewBox.width ? scaledViewBox.width * downscale : undefined,
        };
        if (scale[0])
            scale[0] *= downscale;
        if (scale[1])
            scale[1] *= downscale;
    }
    svg.attributes.viewBox = stringifySvgViewBox(scaledViewBox);
    if (scale[0] || scale[1]) {
        const _scale = scale.filter((v) => v != null && !isNaN(v)).join(" ");
        container.attributes.transform = container.attributes.transform.replace(CSS_TRANSFORM_SCALE, `scale(${_scale})`);
    }
}
exports.reconcileViewBox = reconcileViewBox;
/**
 * Get SVG root container - the first "g" element
 * @internal
 */
function getSvgRootContainer(svg) {
    return svg.children.find((child) => child.name === "g");
}
exports.getSvgRootContainer = getSvgRootContainer;
/** Parse SVG viewBox */
function parseSvgViewBox(svgOrViewBox) {
    let viewBox;
    if (typeof svgOrViewBox === "object") {
        viewBox = svgOrViewBox.attributes.viewBox;
    }
    else {
        viewBox = svgOrViewBox;
    }
    if (viewBox == null || viewBox === "") {
        return undefined;
    }
    let [x, y, width, height] = viewBox.split(" ").map((v) => parseFloat(v));
    return {
        x: x || 0,
        y: y || 0,
        width,
        height,
    };
}
exports.parseSvgViewBox = parseSvgViewBox;
/** Stringify SVG viewBox attribute */
function stringifySvgViewBox(viewBox) {
    return [viewBox.x, viewBox.y, viewBox.width, viewBox.height]
        .filter((v) => v != null)
        .join(" ");
}
exports.stringifySvgViewBox = stringifySvgViewBox;
/** Parse SVG transform attribute scale property */
function parseSvgTransformScale(elementOrTransform) {
    let transform;
    if (typeof elementOrTransform === "object") {
        transform = elementOrTransform.attributes.transform;
    }
    else {
        transform = elementOrTransform;
    }
    if (transform == null || transform === "") {
        return undefined;
    }
    const transformScale = transform?.match(CSS_TRANSFORM_SCALE)?.groups?.scale;
    if (transformScale == null) {
        return undefined;
    }
    return transformScale.split(" ").map((v) => parseFloat(v));
}
exports.parseSvgTransformScale = parseSvgTransformScale;
/**
 * Unescape SVG **text** values.
 *
 * *dot-wasm* escapes svg text ("-" -> "&#45;") and [svgson](https://github.com/elrumordelaluz/svgson/blob/e7234b645b4e344f525d4d2fde2d3f2911d3a75a/src/stringify.js#L20)
 * escapes strings that contain *&...* by wrapping in `<![CDATA[...]]>` tag which causes the
 * resulting text value in SVG and PNG files to show raw escaped value (`&#45;`).
 *
 * We expect to have the original text values rendered rather than escaped version rendered.
 *
 * @example `Diagram (hyphenated&#45;value)` => `Diagram (hyphenated-value)`
 * @internal
 */
function unescapeSvgTextValues(svg) {
    traverse(svg).forEach(function (x) {
        if (this.key === "value" && typeof x === "string" && x !== "") {
            if (x.includes("&")) {
                this.update(he.decode(x), true);
            }
        }
    });
}
exports.unescapeSvgTextValues = unescapeSvgTextValues;
//# sourceMappingURL=data:application/json;base64,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