"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.CdkGraphDiagramPlugin = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/*! Copyright [Amazon.com](http://amazon.com/), Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0 */
const path = require("path");
const cdk_graph_1 = require("../cdk-graph");
const fs = require("fs-extra");
const config_1 = require("./config");
const debug_1 = require("./internal/debug");
const diagram_1 = require("./internal/graphviz/diagram");
const dot_wasm_1 = require("./internal/graphviz/dot-wasm");
const svg_1 = require("./internal/utils/svg");
/**
 * CdkGraphDiagramPlugin is a {@link ICdkGraphPlugin CdkGraph Plugin} implementation for generating
 * diagram artifacts from the {@link CdkGraph} framework.
 */
class CdkGraphDiagramPlugin {
    /** Get standardized artifact id for diagram artifacts */
    static artifactId(name, format) {
        if (name === config_1.DEFAULT_DIAGRAM_NAME) {
            return `${this.ARTIFACT_NS}_${format.toUpperCase()}`;
        }
        return `${this.ARTIFACT_NS}_${name.toUpperCase()}_${format.toUpperCase()}`;
    }
    /** Get standardized artifact file name for diagram artifacts */
    static artifactFilename(name, format) {
        if (name === config_1.DEFAULT_DIAGRAM_NAME) {
            return `${this.ARTIFACT_NS.toLowerCase()}.${format}`;
        }
        return `${this.ARTIFACT_NS.toLowerCase()}.${name}.${format}`;
    }
    /** @inheritdoc */
    get id() {
        return CdkGraphDiagramPlugin.ID;
    }
    /** @inheritdoc */
    get version() {
        return CdkGraphDiagramPlugin.VERSION;
    }
    /** Get default dot artifact */
    get defaultDotArtifact() {
        try {
            return this.getDiagramArtifact(config_1.DEFAULT_DIAGRAM_NAME, config_1.DiagramFormat.DOT);
        }
        catch {
            return;
        }
    }
    /** Get default PNG artifact */
    get defaultPngArtifact() {
        try {
            return this.getDiagramArtifact(config_1.DEFAULT_DIAGRAM_NAME, config_1.DiagramFormat.PNG);
        }
        catch {
            return;
        }
    }
    /** Get diagram plugin config */
    get config() {
        if (this._config != null) {
            return this._config;
        }
        throw new Error("Plugin config has not been instantiated, ensure bind is called");
    }
    constructor(config) {
        /** @inheritdoc */
        this.dependencies = [];
        /** @inheritdoc */
        this.bind = (graph) => {
            this._graph = graph;
            const rc = (graph.config[CdkGraphDiagramPlugin.ID] ||
                {});
            let diagrams = [
                ...(rc.diagrams || []),
                ...(this._config?.diagrams || []),
            ];
            if (diagrams.length === 0) {
                diagrams = [config_1.DEFAULT_DIAGRAM];
            }
            this._config = {
                defaults: {
                    ...config_1.CONFIG_DEFAULTS,
                    ...rc.defaults,
                    ...this._config?.defaults,
                },
                diagrams,
            };
        };
        /** @inheritdoc */
        this.report = async (context) => {
            const pluginConfig = this.config;
            for (const diagramConfig of pluginConfig.diagrams) {
                const config = {
                    ...(diagramConfig.ignoreDefaults ? {} : pluginConfig.defaults),
                    ...diagramConfig,
                };
                let formats = Array.isArray(config.format)
                    ? config.format
                    : [config.format || config_1.DiagramFormat.PNG];
                if (!formats.length) {
                    throw new Error(`Diagram config specifies empty list of formats; must provide at least 1 or undefined to use default.`);
                }
                // each diagram is destructive, so we need a clone
                const store = context.store.clone();
                if (config.filterPlan) {
                    (0, cdk_graph_1.performGraphFilterPlan)(store, config.filterPlan);
                }
                const generatePng = formats.includes(config_1.DiagramFormat.PNG);
                const generateSvg = generatePng || formats.includes(config_1.DiagramFormat.SVG);
                const generateDot = generateSvg || formats.includes(config_1.DiagramFormat.DOT);
                debug_1.IS_DEBUG &&
                    context.writeArtifact(this, "filtered.graph." + config.name + "", "debug/filtered-graph/" + config.name + ".json", JSON.stringify(store.serialize(), null, 2), "DEBUG");
                if (generateDot) {
                    // Graphviz provider
                    const diagram = (0, diagram_1.buildDiagram)(store, {
                        title: config.title,
                        preset: config.filterPlan?.preset,
                        theme: config.theme,
                    });
                    const dot = diagram.toDot();
                    const dotArtifact = context.writeArtifact(this, CdkGraphDiagramPlugin.artifactId(config.name, config_1.DiagramFormat.DOT), CdkGraphDiagramPlugin.artifactFilename(config.name, config_1.DiagramFormat.DOT), dot, `Diagram generated "dot" file for ${config.name} - "${config.title}"`);
                    if (generateSvg) {
                        // const svg = await convertDotToSvg(dotArtifact.filepath);
                        const svg = await (0, dot_wasm_1.invokeDotWasm)(dotArtifact.filepath, diagram.getTrackedImages());
                        context.writeArtifact(this, CdkGraphDiagramPlugin.artifactId(config.name, config_1.DiagramFormat.SVG), CdkGraphDiagramPlugin.artifactFilename(config.name, config_1.DiagramFormat.SVG), svg, `Diagram generated "svg" file for ${config.name} - "${config.title}"`);
                        if (generatePng) {
                            const pngFile = path.join(context.outdir, CdkGraphDiagramPlugin.artifactFilename(config.name, config_1.DiagramFormat.PNG));
                            try {
                                await fs.ensureDir(path.dirname(pngFile));
                                await (0, svg_1.convertSvg)(svg, pngFile);
                                context.logArtifact(this, CdkGraphDiagramPlugin.artifactId(config.name, config_1.DiagramFormat.PNG), pngFile, `Diagram generated "png" file for ${config.name} - "${config.title}"`);
                            }
                            catch (error) {
                                console.error(error);
                                throw new Error(`Failed to generate PNG diagram for ${config.name} at "${pngFile}" - ${String(error)}`);
                            }
                        }
                    }
                }
                // NB: add drawio provider support here
            }
        };
        this._config = config;
    }
    /** Get diagram artifact for a given name and format */
    getDiagramArtifact(name, format) {
        return this._graph?.graphContext?.getArtifact(CdkGraphDiagramPlugin.artifactId(name, format));
    }
}
exports.CdkGraphDiagramPlugin = CdkGraphDiagramPlugin;
_a = JSII_RTTI_SYMBOL_1;
CdkGraphDiagramPlugin[_a] = { fqn: "@aws/pdk.cdk_graph_plugin_diagram.CdkGraphDiagramPlugin", version: "0.22.14" };
/** Namespace for artifacts of the diagram plugin */
CdkGraphDiagramPlugin.ARTIFACT_NS = "DIAGRAM";
/** Fixed id of the diagram plugin */
CdkGraphDiagramPlugin.ID = "diagram";
/** Current semantic version of the diagram plugin */
CdkGraphDiagramPlugin.VERSION = "0.0.0";
//# sourceMappingURL=data:application/json;base64,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