import { ConstructOrder } from "constructs";
import { Counter, ICounterRecord } from "./counter";
import * as SerializedGraph from "./serialized-graph";
import { NodeTypeEnum, UUID, LOGICAL_ID, LOGICAL_UNIVERSAL_ID, EdgeTypeEnum, ReferenceTypeEnum, FlagEnum, EdgeDirectionEnum } from "./types";
import { ConstructInfo } from "../cdk-internals";
/** Public cdk-graph interface */
/**
 * Interface for store counters
 * @internal
 */
export interface IStoreCounters {
    /** Counts total number of each *node type* ({@link NodeTypeEnum}) */
    readonly nodeTypes: Counter<NodeTypeEnum>;
    /** Counts total number of each *edge type* ({@link EdgeTypeEnum}) */
    readonly edgeTypes: Counter<EdgeTypeEnum>;
    /** Counts total number of each *cfnResourceType* */
    readonly cfnResources: Counter;
}
/** Interface for store counts */
export interface IStoreCounts {
    /** Counts total number of nodes in the store */
    readonly nodes: number;
    /** Counts total number of edges in the store */
    readonly edges: number;
    /** Counts total number of stacks in the store */
    readonly stacks: number;
    /** Counts total number of stages in the store */
    readonly stages: number;
    /** Returns {@link ICounterRecord} containing total number of each *node type* ({@link NodeTypeEnum}) */
    readonly nodeTypes: ICounterRecord;
    /** Returns {@link ICounterRecord} containing total number of each *edge type* ({@link EdgeTypeEnum}) */
    readonly edgeTypes: ICounterRecord;
    /** Returns {@link ICounterRecord} containing total number of each *cfnResourceType* */
    readonly cfnResources: ICounterRecord;
}
/** Store class provides the in-memory database-like interface for managing all entities in the graph */
export declare class Store implements SerializedGraph.ISerializableGraphStore {
    /** Builds store from serialized store data */
    static fromSerializedStore(serializedStore: SerializedGraph.SGGraphStore): Store;
    /** Current SemVer version of the store */
    readonly version = "0.0.0";
    /** @internal */
    private _root;
    /** @internal */
    private _edges;
    /** @internal */
    private _nodes;
    /** @internal */
    private _stacks;
    /** @internal */
    private _stages;
    /** @internal */
    private _logicalIdLookup;
    /** @internal */
    private _importArnTokenLookup;
    /** @internal */
    private _counters;
    /**
     * Indicates if the store allows destructive mutations.
     *
     * Destructive mutations are only allowed on clones of the store to prevent plugins and filters from
     * mutating the store for downstream plugins.
     *
     * All `mutate*` methods are only allowed on stores that allow destructive mutations.
     *
     * This behavior may change in the future if the need arises for plugins to pass mutated stores
     * to downstream plugins. But it will be done cautiously with ensuring the intent of
     * downstream plugin is to receive the mutated store.
     */
    readonly allowDestructiveMutations: boolean;
    constructor(allowDestructiveMutations?: boolean);
    /**
     * Root node in the store. The **root** node is not the computed root, but the graph root
     * which is auto-generated and can not be mutated.
     */
    get root(): RootNode;
    /**
     * Gets all stored **edges**
     * @type ReadonlyArray<Edge>
     */
    get edges(): Edge[];
    /**
     * Gets all stored **nodes**
     * @type ReadonlyArray<Node>
     */
    get nodes(): Node[];
    /**
     * Gets all stored **stack** nodes
     * @type ReadonlyArray<StackNode>
     */
    get stacks(): StackNode[];
    /**
     * Gets all stored **stage** nodes
     * @type ReadonlyArray<StageNode>
     */
    get stages(): StageNode[];
    /**
     * Gets all stored **root stack** nodes
     * @type ReadonlyArray<StackNode>
     */
    get rootStacks(): StackNode[];
    /** Get record of all store counters */
    get counts(): IStoreCounts;
    /** Add **edge** to the store */
    addEdge(edge: Edge): void;
    /** Get stored **edge** by UUID */
    getEdge(uuid: UUID): Edge;
    /** Add **node** to the store */
    addNode(node: Node): void;
    /** Get stored **node** by UUID */
    getNode(uuid: UUID): Node;
    /** Add **stack** node to the store */
    addStack(stack: StackNode): void;
    /** Get stored **stack** node by UUID */
    getStack(uuid: UUID): StackNode;
    /** Add **stage** to the store */
    addStage(stage: StageNode): void;
    /** Get stored **stage** node by UUID */
    getStage(uuid: UUID): StageNode;
    /**
     * Compute **universal** *logicalId* based on parent stack and construct *logicalId* (`<stack>:<logicalId>`).
     *
     * Construct *logicalIds are only unique within their containing stack, so to use *logicalId*
     * lookups universally (like resolving references) we need a universal key.
     */
    computeLogicalUniversalId(stack: StackNode, logicalId: string): LOGICAL_UNIVERSAL_ID;
    /** Find node by **universal** *logicalId* (`<stack>:<logicalId>`) */
    findNodeByLogicalUniversalId(uid: LOGICAL_UNIVERSAL_ID): Node;
    /** Find node within given **stack** with given *logicalId* */
    findNodeByLogicalId(stack: StackNode, logicalId: string): Node;
    /** Record a **universal** *logicalId* to node mapping in the store */
    recordLogicalId(stack: StackNode, logicalId: string, resource: Node): void;
    /**
     * Records arn tokens from imported resources (eg: `s3.Bucket.fromBucketArn()`)
     * that are used for resolving references.
     */
    recordImportArn(arnToken: string, resource: Node): void;
    /**
     * Attempts to lookup the {@link Node} associated with a given *import arn token*.
     * @param value Import arn value, which is either object to tokenize or already tokenized string.
     * @returns Returns matching {@link Node} if found, otherwise undefined.
     */
    findNodeByImportArn(value: any): Node | undefined;
    /** Serialize the store */
    serialize(): SerializedGraph.SGGraphStore;
    /**
     * Clone the store to allow destructive mutations.
     * @param allowDestructiveMutations Indicates if destructive mutations are allowed; defaults to `true`
     * @returns {Store} Returns a clone of the store that allows destructive mutations
     */
    clone(allowDestructiveMutations?: boolean): Store;
    /**
     * Verifies that the store allows destructive mutations.
     * @throws Error is store does **not** allow mutations
     */
    verifyDestructiveMutationAllowed(): void;
    /**
     * Remove **edge** from the store
     * @destructive
     */
    mutateRemoveEdge(edge: Edge): boolean;
    /**
     * Remove **node** from the store
     * @destructive
     */
    mutateRemoveNode(node: Node): boolean;
}
/** Base interface for all store entities **data** props */
export interface IBaseEntityDataProps {
    /** Attributes */
    readonly attributes?: SerializedGraph.Attributes;
    /** Metadata entries */
    readonly metadata?: SerializedGraph.Metadata;
    /** Tags */
    readonly tags?: SerializedGraph.Tags;
    /** Flags */
    readonly flags?: FlagEnum[];
}
/** Base interface for all store entities props */
export interface IBaseEntityProps extends IBaseEntityDataProps {
    /** Store */
    readonly store: Store;
    /** UUID */
    readonly uuid: UUID;
}
/** Base class for all store entities (Node and Edges) */
export declare abstract class BaseEntity implements SerializedGraph.ISerializableEntity {
    /** Reference to the store */
    readonly store: Store;
    /** Universally unique identifier  */
    readonly uuid: UUID;
    /** @internal */
    private readonly _attributes;
    /** @internal */
    private readonly _metadata;
    /** @internal */
    private readonly _tags;
    /** @internal */
    private readonly _flags;
    /** @internal */
    protected _destroyed: boolean;
    constructor(props: IBaseEntityProps);
    /**
     * Get *readonly* record of all attributes
     * @type Readonly<SerializedGraph.Attributes>
     */
    get attributes(): SerializedGraph.Attributes;
    /**
     * Get *readonly* list of all metadata entries
     * @type Readonly<SerializedGraph.Metadata>
     */
    get metadata(): SerializedGraph.Metadata;
    /**
     * Get *readonly* record of all tags
     * @type Readonly<SerializedGraph.Tags>
     */
    get tags(): SerializedGraph.Tags;
    /**
     * Get *readonly* list of all flags
     * @type ReadonlyArray<FlagEnum>
     */
    get flags(): FlagEnum[];
    /** Indicates if the entity has been destroyed (eg: removed from store) */
    get isDestroyed(): boolean;
    /** Indicates if the entity has had destructive mutations applied */
    get isMutated(): boolean;
    /** Indicates if entity has a given attribute defined, and optionally with a specific value */
    hasAttribute(key: string, value?: any): boolean;
    /**
     * Add attribute.
     *
     * @throws Error if attribute for key already exists
     */
    addAttribute(key: string, value: any): void;
    /** Set attribute. This will overwrite existing attribute. */
    setAttribute(key: string, value: any): void;
    /** Get attribute by key */
    getAttribute(key: string): any;
    /** Add metadata entry */
    addMetadata(metadataType: string, data: any): void;
    /** Indicates if entity has matching metadata entry */
    hasMetadata(metadataType: string, data: any): boolean;
    /**
     * Retrieves all metadata entries of a given type
     * @type Readonly<SerializedGraph.Metadata>
     */
    findMetadata(metadataType: string): SerializedGraph.Metadata;
    /**
     * Add tag.
     * @throws Throws Error is tag for key already exists
     */
    addTag(key: string, value: string): void;
    /** Set tag. Will overwrite existing tag. */
    setTag(key: string, value: string): void;
    /** Indicates if entity has tag, optionally verifying tag value */
    hasTag(key: string, value?: string): boolean;
    /** Get tag by key */
    getTag(key: string): string | undefined;
    /** Add flag */
    addFlag(flag: FlagEnum): void;
    /** Indicates if entity has a given flag */
    hasFlag(flag: FlagEnum): boolean;
    /**
     * Applies data (attributes, metadata, tags, flag) to entity.
     *
     * Generally used only for mutations such as collapse and consume to retain data.
     * @param data - The data to apply
     * @param {boolean} [applyFlags=false] - Indicates if data is overwritten
     * @param {boolean} [applyFlags=false] - Indicates if flags should be applied
     */
    applyData(data: IBaseEntityDataProps, overwrite?: boolean, applyFlags?: boolean): void;
    /**
     * Performs pre-mutate operations on entity and store
     * @internal
     */
    protected _preMutate(): void;
    /**
     * Destroy the entity be removing all references and removing from store.
     * @param {boolean} [strict=false] - If `strict`, then entity must not have any references remaining when attempting to destroy
     * @destructive
     */
    abstract mutateDestroy(strict?: boolean): void;
    /**
     * Serialize entity
     * @internal
     */
    _serialize(): SerializedGraph.SGEntity;
}
/** Base edge props agnostic to edge type. Used for extending per edge class with type specifics. */
export interface ITypedEdgeProps extends IBaseEntityProps {
    /** Edge **source** is the node that defines the edge (tail) */
    readonly source: Node;
    /** Edge **target** is the node being referenced by the **source** (head) */
    readonly target: Node;
}
/** Edge props interface */
export interface IEdgeProps extends ITypedEdgeProps {
    /** Indicates the direction in which the edge is directed */
    readonly direction: EdgeDirectionEnum;
    /** Type of edge */
    readonly edgeType: EdgeTypeEnum;
}
/**
 * EdgeCain defines an edge traversal path to terminal point.
 *
 * An edge may reference an *OutputNode* which is just a proxy to actual *ResourceNode*, in which case
 * the target node expected is generally the *ResourceNode*. An EdgeChain defines this resolution path
 * to the expected target.
 *
 * Note: Due to JSII not supporting tuple types in interfaces, this has to be any[]. If using this type and need to cast to somethign specific,
 * the ts type is: export type EdgeChain = [Edge, ...EdgeChain[]];
 */
export type EdgeChain = any[];
/** Edge class defines a link (relationship) between nodes, as in standard [graph theory](https://en.wikipedia.org/wiki/Graph_theory) */
export declare class Edge extends BaseEntity implements SerializedGraph.ISerializableEdge {
    /** Find first edge matching predicate within an EdgeChain */
    static findInChain(chain: EdgeChain, predicate: IEdgePredicate): Edge | undefined;
    /** Find all matching edges based on predicate within an EdgeChain */
    static findAllInChain(chain: EdgeChain, predicate: IEdgePredicate): Edge[];
    /** Type of edge */
    readonly edgeType: EdgeTypeEnum;
    /** @internal */
    private _source;
    /** @internal */
    private _target;
    /** @internal */
    private _direction;
    /** Edge **source** is the node that defines the edge (tail) */
    get source(): Node;
    /** Edge **target** is the node being referenced by the **source** (head) */
    get target(): Node;
    /** Indicates the direction in which the edge is directed */
    get direction(): EdgeDirectionEnum;
    /** Indicates if **source** and **target** nodes reside in different *root* stacks */
    get isCrossStack(): boolean;
    /**
     * Indicates if the Edge's **source** and **target** are the same, or were the same
     * when it was created (prior to mutations).
     *
     * To check whether it was originally closed, use `hasFlag(FlagEnum.CLOSED_EDGE)` instead.
     */
    get isClosed(): boolean;
    /**
     * Indicates if edge is extraneous which is determined by explicitly having *EXTRANEOUS* flag
     * added and/or being a closed loop (source===target).
     */
    get isExtraneous(): boolean;
    constructor(props: IEdgeProps);
    /**
     * Indicates if this edge is equivalent to another edge.
     *
     * Edges are considered equivalent if they share same type, source, and target.
     */
    isEquivalent(edge: Edge): boolean;
    /** Indicates if edge allows destructive mutations */
    get allowDestructiveMutations(): boolean;
    /**
     * Change the edge **direction**
     * @destructive
     */
    mutateDirection(direction: EdgeDirectionEnum): void;
    /**
     * Change the edge **source**
     * @destructive
     */
    mutateSource(node: Node): void;
    /**
     * Change the edge **target**
     * @destructive
     */
    mutateTarget(node: Node): void;
    /**
     * Destroy the edge. Remove all references and remove from store.
     * @destructive
     */
    mutateDestroy(_strict?: boolean): void;
    /**
     * Merge an equivalent edge's data into this edge and destroy the other edge.
     *
     * Used during filtering operations to consolidate equivalent edges.
     * @param edge - The edge to consume
     * @throws Error is edge is not *equivalent*
     * @destructive
     */
    mutateConsume(edge: Edge): void;
    /** Get string representation of this edge */
    toString(): string;
    /** @internal */
    _serialize(): SerializedGraph.SGEdge;
}
/** Dependency edge class defines CloudFormation dependency between resources */
export declare class Dependency extends Edge {
    /** Edge prefix to denote dependency edge  */
    static readonly PREFIX = "DEP:";
    /** Indicates if given edge is a {@link Dependency} edge */
    static isDependency(edge: Edge): edge is Reference;
    constructor(props: ITypedEdgeProps);
}
/** Reference edge props */
export interface IReferenceProps extends ITypedEdgeProps {
    /** Type of reference */
    referenceType?: ReferenceTypeEnum;
}
/** Reference edge class defines a directed relationship between nodes  */
export declare class Reference extends Edge {
    /** Edge prefix to denote **Ref** type reference edge  */
    static readonly PREFIX: string;
    /** Attribute defining the type of reference */
    static readonly ATT_TYPE = "graph:reference:type";
    /** Indicates if edge is a {@link Reference} */
    static isReference(edge: Edge): edge is Reference;
    /** Indicates if edge is a **Ref** based {@link Reference} edge */
    static isRef(edge: Edge): edge is Reference;
    constructor(props: IReferenceProps);
    /** Get type of reference */
    get referenceType(): ReferenceTypeEnum;
    /** Resolve reference chain */
    resolveChain(): EdgeChain;
    /**
     * Resolve targets by following potential edge chain.
     *
     * @see {@link EdgeChain}
     */
    resolveTargets(): Node[];
}
/** Attribute type reference props */
export interface IAttributeReferenceProps extends ITypedEdgeProps {
    /** Resolved attribute value */
    value: SerializedGraph.Value;
}
/** Attribute type reference edge */
export declare class AttributeReference extends Reference {
    /** Edge prefix to denote **Fn::GetAtt** type reference edge  */
    static readonly PREFIX: string;
    /** Attribute key for resolved value of attribute reference */
    static readonly ATT_VALUE = "graph:reference:attribute:value";
    /** Indicates if edge in an **Fn::GetAtt** {@link Reference} */
    static isAtt(edge: Edge): edge is AttributeReference;
    constructor(props: IAttributeReferenceProps);
    /** Get the resolved attribute value */
    get value(): string;
}
/** Import reference defines **Fn::ImportValue** type reference edge. */
export declare class ImportReference extends Reference {
    /** Edge prefix to denote **Fn::ImportValue** type reference edge */
    static readonly PREFIX: string;
    /** Indicates if edge is **Fn::ImportValue** based {@link Reference} */
    static isImport(edge: Edge): edge is ImportReference;
    constructor(props: ITypedEdgeProps);
}
/** Base node props agnostic to node type. Used for extending per node class with type specifics. */
export interface ITypedNodeProps extends IBaseEntityProps {
    /** Node id, which is unique within parent scope */
    readonly id: string;
    /** Path of the node */
    readonly path: string;
    /** Stack the node is contained */
    readonly stack?: StackNode;
    /** Parent node */
    readonly parent?: Node;
    /** Synthesized construct information defining jii resolution data */
    readonly constructInfo?: ConstructInfo;
    /** Logical id of the node, which is only unique within containing stack */
    readonly logicalId?: string;
    /** Type of CloudFormation resource */
    readonly cfnType?: string;
}
/** Node props */
export interface INodeProps extends ITypedNodeProps {
    /** Type of node */
    readonly nodeType: NodeTypeEnum;
}
/** Predicate to match node */
export interface INodePredicate {
    filter(node: Node): boolean;
}
/** Predicate to match edge */
export interface IEdgePredicate {
    filter(edge: Edge): boolean;
}
/** Options for node based search operations */
export interface IFindNodeOptions {
    /** The predicate to match node(s) */
    predicate?: INodePredicate;
    /** The order of traversal during search path */
    order?: ConstructOrder;
}
/** Options for edge based search operations */
export interface IFindEdgeOptions {
    /** The predicate to match edges(s) */
    predicate?: IEdgePredicate;
    /** The order of traversal during search path */
    order?: ConstructOrder;
    /** Indicates reverse order */
    reverse?: boolean;
}
/** Node class is the base definition of **node** entities in the graph, as in standard [graph theory](https://en.wikipedia.org/wiki/Graph_theory) */
export declare class Node extends BaseEntity implements SerializedGraph.ISerializableNode {
    /** Type of node */
    readonly nodeType: NodeTypeEnum;
    /** Node id, which is only unique within parent scope */
    readonly id: string;
    /** Path of the node */
    readonly path: string;
    /** @internal */
    private _stack?;
    /** @internal */
    private _parent?;
    /** Stack the node is contained in */
    get stack(): StackNode | undefined;
    /** Parent node. Only the root node should not have parent. */
    get parent(): Node | undefined;
    /** Synthesized construct information defining jii resolution data */
    readonly constructInfo?: ConstructInfo;
    /** Logical id of the node, which is only unique within containing stack */
    readonly logicalId?: LOGICAL_ID;
    /** @internal */
    protected _cfnType?: string;
    /** @internal */
    protected readonly _children: Map<string, Node>;
    /** @internal */
    protected readonly _links: Map<UUID, Edge>;
    /** @internal */
    protected readonly _reverseLinks: Map<UUID, Edge>;
    /** Indicates the depth of the node relative to root (0) */
    readonly depth: number;
    constructor(props: INodeProps);
    /** Gets descending ordered list of ancestors from the root */
    get scopes(): Node[];
    /** Indicates if node is direct child of the graph root node */
    get isTopLevel(): boolean;
    /** Get **root** stack */
    get rootStack(): StackNode | undefined;
    /** Get all direct child nodes */
    get children(): Node[];
    /** Indicates if this node is a *leaf* node, which means it does not have children */
    get isLeaf(): boolean;
    /** Gets all links (edges) in which this node is the **source** */
    get links(): Edge[];
    /** Gets all links (edges) in which this node is the **target** */
    get reverseLinks(): Edge[];
    /** Synthesized construct information defining jii resolution data */
    get constructInfoFqn(): string | undefined;
    /** Indicates if node is a *Custom Resource* */
    get isCustomResource(): boolean;
    /**
     * Indicates if node ConstructInfoFqn denotes a `aws-cdk-lib.*.Cfn*` construct.
     * @see {@link FlagEnum.CFN_FQN}
     */
    get isCfnFqn(): boolean;
    /** Gets CloudFormation properties for this node */
    get cfnProps(): SerializedGraph.PlainObject | undefined;
    /** Get the CloudFormation resource type for this node */
    get cfnType(): string | undefined;
    /** Gets list of {@link Dependency} links (edges) where this node is the **source** */
    get dependencyLinks(): Dependency[];
    /** Gets list of {@link Dependency} links (edges) where this node is the **target** */
    get reverseDependencyLinks(): Dependency[];
    /** Gets list of {@link Reference} links (edges) where this node is the **source** */
    get referenceLinks(): Reference[];
    /** Gets list of {@link Reference} links (edges) where this node is the **target** */
    get reverseReferenceLinks(): Reference[];
    /**
     * Get list of **Nodes** that *this node references*
     * @see {@link Node.referenceLinks}
     */
    get references(): Node[];
    /**
     * Get list of **Nodes** that *reference this node*
     * @see {@link Node.reverseReferenceLinks}
     */
    get referencedBy(): Node[];
    /**
     * Get list of **Nodes** that *this node depends on*
     * @see {@link Node.dependencyLinks}
     */
    get dependencies(): Node[];
    /**
     * Get list of **Nodes** that *depend on this node*
     * @see {@link Node.reverseDependencyLinks}
     */
    get dependedOnBy(): Node[];
    /** Indicates if this node is considered a {@link FlagEnum.GRAPH_CONTAINER} */
    get isGraphContainer(): boolean;
    /** Indicates if this node is considered a {@link FlagEnum.CLUSTER} */
    get isCluster(): boolean;
    /**
     * Indicates if this node is considered a {@link FlagEnum.EXTRANEOUS} node
     * or determined to be extraneous:
     * - Clusters that contain no children
     */
    get isExtraneous(): boolean;
    /** Indicates if this node is considered a {@link FlagEnum.ASSET} */
    get isAsset(): boolean;
    /** Get list of *siblings* of this node. */
    get siblings(): Node[];
    /** Get specific CloudFormation property */
    getCfnProp(key: string): SerializedGraph.Value | undefined;
    /** Add *link* to another node */
    addLink(edge: Edge): void;
    /** Add *link* from another node */
    addReverseLink(edge: Edge): void;
    /** Add *child* node */
    addChild(node: Node): void;
    /** Indicates if specific *node* is a *child* of *this node* */
    isChild(node: Node): boolean;
    /** Indicates if a specific *node* is an *ancestor* of *this node* */
    isAncestor(ancestor: Node): boolean;
    /**
     * Find nearest *ancestor* of *this node* matching given predicate.
     * @param predicate - Predicate to match ancestor
     * @max {number} [max] - Optional maximum levels to ascend
     */
    findAncestor(predicate: INodePredicate, max?: number): Node | undefined;
    /**
     * Gets the nearest **common** *ancestor* shared between *this node* and another *node*.
     * @throws Error if *node* does not share a **common** *ancestor*
     */
    getNearestAncestor(node: Node): Node;
    /**
     * Return this construct and all of its sub-nodes in the given order.
     *
     * Optionally filter nodes based on predicate.
     */
    findAll(options?: IFindNodeOptions): Node[];
    /** Recursively find the nearest sub-node matching predicate */
    find(predicate: INodePredicate): Node | undefined;
    /**
     * Get *child* node with given *id*.
     *
     * @throws Error if no child with given id
     */
    getChild(id: string): Node;
    /** Find child with given *id*. Similar to `find` but does not throw error if no child found. */
    findChild(id: string): Node | undefined;
    /**
     * Return all direct links of this node and that of all sub-nodes.
     *
     * Optionally filter links based on predicate.
     */
    findAllLinks(options?: IFindEdgeOptions): Edge[];
    /**
     * Resolve all link chains
     * @see {@link EdgeChain}
     */
    getLinkChains(reverse?: boolean): EdgeChain[];
    /**
     * Find link of this node based on predicate. By default this will follow link
     * chains to evaluate the predicate against and return the matching direct link
     * of this node.
     *
     * @param predicate Edge predicate function to match edge
     * @param reverse Indicates if links are search in reverse order
     * @param follow Indicates if link chain is followed
     * @param direct Indicates that only *direct* links should be searched
     * @returns
     */
    findLink(predicate: IEdgePredicate, reverse?: boolean, follow?: boolean, direct?: boolean): Edge | undefined;
    /**
     * Find all links of this node based on predicate. By default this will follow link
     * chains to evaluate the predicate against and return the matching direct links
     * of this node.
     *
     * @param predicate Edge predicate function to match edge
     * @param reverse Indicates if links are search in reverse order
     * @param follow Indicates if link chain is followed
     * @param direct Indicates that only *direct* links should be searched
     * @returns
     */
    findLinks(predicate: IEdgePredicate, reverse?: boolean, follow?: boolean, direct?: boolean): Edge[];
    /** Indicates if *this node* references *another node* */
    doesReference(node: Node): boolean;
    /** Indicates if *this node* depends on *another node* */
    doesDependOn(node: Node): boolean;
    /**
     * Indicates if this node allows destructive mutations
     * @see {@link Store.allowDestructiveMutations}
     */
    get allowDestructiveMutations(): boolean;
    /**
     * Collapses all sub-nodes of *this node* into *this node*.
     * @destructive
     */
    mutateCollapse(): void;
    /**
     * Collapses *this node* into *it's parent node*
     * @destructive
     */
    mutateCollapseToParent(): Node;
    /**
     * Collapses *this node* into *an ancestor*
     * @destructive
     */
    mutateCollapseTo(ancestor: Node): Node;
    /**
     * Destroys this node by removing all references and removing this node from the store.
     * @param {boolean} [strict=false] - Indicates that this node must not have references
     * @destructive
     */
    mutateDestroy(strict?: boolean): void;
    /**
     * Reconciles links defined by this node. During mutations, multiple *equivalent* links may exist and should be
     * consolidated into a single link. This operation should be called after collapsing children to remove duplicates.
     * @internal
     * @destructive
     */
    protected _mutateReconcileLinks(): void;
    /**
     * Remove a *child* node from *this node*
     * @destructive
     */
    mutateRemoveChild(node: Node): boolean;
    /**
     * Remove a *link* from *this node*
     * @destructive
     */
    mutateRemoveLink(link: Edge): boolean;
    /**
     * Remove a *link* to *this node*
     * @destructive
     */
    mutateRemoveReverseLink(link: Edge): boolean;
    /**
     * Hoist *this node* to an *ancestor* by removing it from its current parent node and
     * in turn moving it to the ancestor.
     * @destructive
     */
    mutateHoist(newParent: Node): void;
    /**
     * Hoist all children to parent and collapse node to parent.
     * @destructive
     */
    mutateUncluster(): void;
    /**
     * Move this node into a new parent node.
     * @param {Node} newParent - The parent to move this node to.
     * @destructive
     */
    mutateMove(newParent: Node): void;
    /** Get string representation of this node */
    toString(): string;
    /**
     * Serialize this node
     * @internal
     */
    _serialize(): SerializedGraph.SGNode;
}
/** ResourceNode props */
export interface IResourceNodeProps extends ITypedNodeProps {
    /** Type of node */
    nodeType?: NodeTypeEnum;
    /** Indicates if this resource is owned by cdk (defined in cdk library) */
    cdkOwned: boolean;
}
/** ResourceNode class defines a L2 cdk resource construct */
export declare class ResourceNode extends Node {
    /** Attribute key for cfn resource type */
    static readonly ATT_WRAPPED_CFN_TYPE = "graph:resource:cfn-type";
    /** Attribute key for cfn properties */
    static readonly ATT_WRAPPED_CFN_PROPS = "graph:resource:cfn-props";
    /** Indicates if node is a {@link ResourceNode} */
    static isResourceNode(node: Node): node is ResourceNode;
    /** @internal */
    private _cfnResource;
    constructor(props: IResourceNodeProps);
    /** Get the CloudFormation resource type for this L2 resource or for the L1 resource is wraps. */
    get cfnType(): string | undefined;
    /** Indicates if this resource is owned by cdk (defined in cdk library) */
    get isCdkOwned(): boolean;
    /** Indicates if Resource wraps a single CfnResource */
    get isWrapper(): boolean;
    /** Get the default/primary CfnResource that this Resource wraps */
    get cfnResource(): CfnResourceNode | undefined;
    /** Get the cfn properties from the L1 resource that this L2 resource wraps */
    get cfnProps(): SerializedGraph.PlainObject | undefined;
    /**
     * Modifies the L1 resource wrapped by this L2 resource
     * @param cfnResource
     * @destructive
     */
    mutateCfnResource(cfnResource?: CfnResourceNode): void;
    /** @inheritdoc */
    mutateRemoveChild(node: Node): boolean;
}
/** CfnResourceNode props */
export interface ICfnResourceNodeProps extends ITypedNodeProps {
    nodeType?: NodeTypeEnum;
    importArnToken?: string;
}
/** CfnResourceNode defines an L1 cdk resource */
export declare class CfnResourceNode extends Node {
    /** Normalized CfnReference attribute */
    static readonly ATT_IMPORT_ARN_TOKEN = "graph:cfn-resource:import-arn-token";
    /** Indicates if a node is a {@link CfnResourceNode} */
    static isCfnResourceNode(node: Node): node is CfnResourceNode;
    /** @internal */
    private _resource?;
    constructor(props: ICfnResourceNodeProps);
    /** @inheritdoc */
    get isExtraneous(): boolean;
    /** Indicates if this CfnResource is imported (eg: `s3.Bucket.fromBucketArn`) */
    get isImport(): boolean;
    /** Reference to the L2 Resource that wraps this L1 CfnResource if it is wrapped. */
    get resource(): ResourceNode | undefined;
    /**
     * Evaluates if CfnResourceNode fqn is equivalent to ResourceNode fqn.
     * @example `aws-cdk-lib.aws_lambda.Function` => `aws-cdk-lib.aws_lambda.CfnFunction`
     * @param resource - {@link Graph.ResourceNode} to compare
     * @returns Returns `true` if equivalent, otherwise `false`
     */
    isEquivalentFqn(resource: ResourceNode): boolean;
    /**
     * Finds the near *ancestor* that is a {@link ResourceNode}
     * @internal
     */
    private _findNearestResource;
    /**
     * @inheritdoc
     */
    mutateDestroy(strict?: boolean): void;
}
/** OutputNode props */
export interface IOutputNodeProps extends ITypedNodeProps {
    /** Resolved output value */
    readonly value: any;
    /** Export name */
    readonly exportName?: string;
    /** Description */
    readonly description?: string;
}
/** OutputNode defines a cdk CfnOutput resources */
export declare class OutputNode extends Node {
    /** Attribute key where output value is stored */
    static readonly ATTR_VALUE = "graph:output:value";
    /** Attribute key where output export name is stored */
    static readonly ATTR_EXPORT_NAME = "graph:output:export-name";
    /** Indicates if node is an {@link OutputNode} */
    static isOutputNode(node: Node): node is OutputNode;
    /** Indicates if {@link OutputNode} is **exported** */
    readonly isExport: boolean;
    constructor(props: IOutputNodeProps);
    /** Get the *value** attribute */
    get value(): any;
    /** Get the export name attribute */
    get exportName(): string | undefined;
    /** @inheritdoc */
    mutateDestroy(strict?: boolean): void;
}
/** {@link ParameterNode} props */
export interface IParameterNodeProps extends ITypedNodeProps {
    /** Resolved value */
    readonly value: any;
    /** Parameter type */
    readonly parameterType: string;
    /** Description */
    readonly description?: string;
}
/** ParameterNode defines a CfnParameter node */
export declare class ParameterNode extends Node {
    /** Attribute key where parameter value is store */
    static readonly ATTR_VALUE = "graph:parameter:value";
    /** Attribute key where parameter type is stored */
    static readonly ATTR_TYPE = "graph:parameter:type";
    /** Indicates if node is a {@link ParameterNode} */
    static isParameterNode(node: Node): node is ParameterNode;
    /** Indicates if parameter is a reference to a stack */
    readonly isStackReference: boolean;
    constructor(props: IParameterNodeProps);
    /** Get the value attribute */
    get value(): any;
    /** Get the parameter type attribute */
    get parameterType(): any;
    /** @inheritdoc */
    mutateDestroy(strict?: boolean): void;
}
/** {@link StackNode} props */
export interface IStackNodeProps extends ITypedNodeProps {
    /** Type of node */
    nodeType?: NodeTypeEnum.NESTED_STACK;
}
/** StackNode defines a cdk Stack */
export declare class StackNode extends Node {
    /** Indicates if node is a {@link StackNode} */
    static isStackNode(node: Node): node is StackNode;
    /**
     * Gets the {@link StackNode} containing a given resource
     * @throws Error is node is not contained in a stack
     */
    static of(node: Node): StackNode;
    /** @internal */
    private readonly _outputs;
    /** @internal */
    private readonly _parameters;
    /** @internal */
    private _stage?;
    /** Get {@link StageNode} containing this stack */
    get stage(): StageNode | undefined;
    /** Get all {@link OutputNode}s defined by this stack */
    get outputs(): OutputNode[];
    /** Get all {@link ParameterNode}s defined by this stack */
    get parameters(): ParameterNode[];
    constructor(props: IStackNodeProps);
    /** Get all **exported** {@link OutputNode}s defined by this stack */
    get exports(): OutputNode[];
    /** Associate {@link OutputNode} with this stack */
    addOutput(node: OutputNode): void;
    /**
     * Find {@link OutputNode} with *logicalId* defined by this stack
     * @throws Error is no output found matching *logicalId*
     */
    findOutput(logicalId: string): OutputNode;
    /** Associate {@link ParameterNode} with this stack */
    addParameter(node: ParameterNode): void;
    /**
     * Find {@link ParameterNode} with *parameterId* defined by this stack
     * @throws Error is no parameter found matching *parameterId*
     */
    findParameter(parameterId: string): ParameterNode;
    /**
     * Disassociate {@link OutputNode} from this stack
     * @destructive
     */
    mutateRemoveOutput(node: OutputNode): boolean;
    /**
     * Disassociate {@link ParameterNode} from this stack
     * @destructive
     */
    mutateRemoveParameter(node: ParameterNode): boolean;
    /** @inheritdoc */
    mutateDestroy(strict?: boolean): void;
    /** @inheritdoc */
    mutateHoist(newParent: Node): void;
}
/** {@link NestedStackNode} props */
export interface INestedStackNodeProps extends IStackNodeProps {
    /** Parent stack */
    readonly parentStack: StackNode;
}
/** NestedStackNode defines a cdk NestedStack */
export declare class NestedStackNode extends StackNode {
    /** Indicates if node is a {@link NestedStackNode} */
    static isNestedStackNode(node: Node): node is NestedStackNode;
    /** @internal */
    private _parentStack?;
    /** Get parent stack of this nested stack */
    get parentStack(): StackNode | undefined;
    constructor(props: INestedStackNodeProps);
    /** @inheritdoc */
    mutateHoist(newParent: Node): void;
}
/** StageNode defines a cdk Stage */
export declare class StageNode extends Node {
    /** Indicates if node is a {@link StageNode} */
    static isStageNode(node: Node): node is StageNode;
    /**
     * Gets the {@link StageNode} containing a given resource
     * @throws Error is node is not contained in a stage
     */
    static of(node: Node): StageNode;
    /** @internal */
    private readonly _stacks;
    /** Gets all stacks contained by this stage */
    get stacks(): StackNode[];
    constructor(props: ITypedNodeProps);
    /** Associate a {@link StackNode} with this stage */
    addStack(stack: StackNode): void;
    /**
     * Disassociate {@link StackNode} from this stage
     * @destructive
     */
    mutateRemoveStack(stack: StackNode): boolean;
}
/** {@link AppNode} props */
export interface IAppNodeProps extends IBaseEntityDataProps {
    /** Store */
    readonly store: Store;
    /** Parent node */
    readonly parent?: Node;
    /** Synthesized construct information defining jii resolution data */
    readonly constructInfo?: ConstructInfo;
    /** Logical id of the node, which is only unique within containing stack */
    readonly logicalId?: string;
    /** Type of CloudFormation resource */
    readonly cfnType?: string;
}
/** AppNode defines a cdk App */
export declare class AppNode extends Node {
    /** Fixed UUID for App node */
    static readonly UUID = "App";
    /** Fixed path of the App  */
    static readonly PATH = "/";
    /** Indicates if node is a {@link AppNode} */
    static isAppNode(node: Node): node is AppNode;
    constructor(props: IAppNodeProps);
}
/** RootNode represents the root of the store tree */
export declare class RootNode extends Node {
    /** Fixed UUID of root */
    static readonly UUID = "Root";
    /** Fixed path of root */
    static readonly PATH = "";
    /** Indicates if node is a {@link RootNode} */
    static isRootNode(node: Node): node is RootNode;
    constructor(store: Store);
    /**
     * @inheritdoc **The root not is excluded from list**
     */
    findAll(options?: IFindNodeOptions | undefined): Node[];
    /**
     * > {@link RootNode} does not support this mutation
     * @throws Error does not support
     * @inheritdoc
     */
    mutateCollapse(): void;
    /**
     * > {@link RootNode} does not support this mutation
     * @throws Error does not support
     * @inheritdoc
     */
    mutateCollapseToParent(): Node;
    /**
     * > {@link RootNode} does not support this mutation
     * @throws Error does not support
     * @inheritdoc
     */
    mutateCollapseTo(_ancestor: Node): Node;
    /**
     * > {@link RootNode} does not support this mutation
     * @throws Error does not support
     * @inheritdoc
     */
    mutateDestroy(_strict?: boolean): void;
    /**
     * > {@link RootNode} does not support this mutation
     * @throws Error does not support
     * @inheritdoc
     */
    mutateHoist(_newParent: Node): void;
}
export declare function isResourceLike(node: Node): boolean;
/**
 * Deserializes a *serialized store object* into an *in-memory store instance**.
 * @param serializedStore - The serialized store to deserialize
 * @param allowDestructiveMutations - Indicates if the store instance allows destructive mutations.
 * @throws Error if a serialized node's parent does not match visitor parent
 * @throws Error if a serialized node type deserialization mapping is not defined
 * @throws Error if edge type deserialization mapping is not defined
 */
export declare function deserializeStore(serializedStore: SerializedGraph.SGGraphStore, allowDestructiveMutations?: boolean): Store;
