"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.CloudscapeReactTsWebsiteProject = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/*! Copyright [Amazon.com](http://amazon.com/), Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0 */
const fs = require("fs");
const path = require("path");
const Mustache = require("mustache");
const projen_1 = require("projen");
const javascript_1 = require("projen/lib/javascript");
const web_1 = require("projen/lib/web");
/**
 * Synthesizes a Cloudscape React Typescript Website Project.
 *
 * @pjid cloudscape-react-ts-website
 */
class CloudscapeReactTsWebsiteProject extends web_1.ReactTypeScriptProject {
    constructor(options) {
        const hasApi = !!options.typeSafeApi;
        super({
            ...options,
            defaultReleaseBranch: options.defaultReleaseBranch ?? "main",
            name: options.name,
            sampleCode: false,
            prettier: options.prettier || true,
            packageManager: options.parent && options.parent instanceof javascript_1.NodeProject
                ? options.parent.package.packageManager
                : options.packageManager,
            readme: {
                contents: fs
                    .readFileSync(path.resolve(__dirname, "../samples/cloudscape-react-ts-website/README.md"))
                    .toString(),
            },
            gitignore: [
                "public/runtime-config.json",
                "public/api.json",
                ...(options.gitignore || []),
            ],
        });
        this.addDeps("@aws-northstar/ui", "@cloudscape-design/components", "@cloudscape-design/board-components", "react-router-dom");
        this.testTask.reset();
        const lintTask = this.tasks.tryFind("eslint");
        lintTask && this.testTask.spawn(lintTask);
        this.testTask.exec("react-scripts test --watchAll=false --passWithNoTests");
        this.applicationName = options.applicationName ?? "Sample App";
        this.publicDir = options.publicDir ?? "public";
        const srcDir = path.resolve(__dirname, "../samples/cloudscape-react-ts-website/src");
        const publicDir = path.resolve(__dirname, "../samples/cloudscape-react-ts-website/public");
        if (options.typeSafeApi) {
            const hooks = options.typeSafeApi.library?.typescriptReactQueryHooks;
            const libraryHooksPackage = hooks?.package?.packageName;
            if (!libraryHooksPackage) {
                throw new Error("Cannot pass in a Type Safe Api without React Hooks Library configured!");
            }
            this.addDeps(libraryHooksPackage);
            this.setupSwaggerUi(options.typeSafeApi);
        }
        const mustacheConfig = {
            applicationName: this.applicationName,
            hasApi,
            apiHooksPackage: options.typeSafeApi?.library?.typescriptReactQueryHooks?.package
                ?.packageName,
        };
        new projen_1.SampleDir(this, this.srcdir, {
            files: {
                ...Object.fromEntries(this.buildSampleDirEntries(srcDir, [], mustacheConfig)),
            },
        });
        new projen_1.SampleDir(this, this.publicDir, {
            files: {
                ...Object.fromEntries(this.buildSampleDirEntries(publicDir, [], mustacheConfig)),
            },
        });
        // Linting is managed as part of the test task already, so disable react-scripts running eslint again
        this.tasks.addEnvironment("DISABLE_ESLINT_PLUGIN", "true");
        // Relax EsLint and TSC for dev
        this.tasks.tryFind("dev")?.env("ESLINT_NO_DEV_ERRORS", "true");
        this.tasks.tryFind("dev")?.env("TSC_COMPILE_ON_ERROR", "true");
    }
    setupSwaggerUi(tsApi) {
        this.addDevDeps("@types/swagger-ui-react");
        this.addDeps("swagger-ui-react@5.5.0", "aws4fetch");
        const targetApiSpecPath = `${path.relative(tsApi.model.outdir, this.outdir)}/public/api.json`;
        tsApi.model.postCompileTask.exec(`rm -f ${targetApiSpecPath}`);
        tsApi.model.postCompileTask.exec(`cp .api.json ${targetApiSpecPath}`);
    }
    buildSampleDirEntries(dir, pathPrefixes = [], mustacheConfig) {
        return fs
            .readdirSync(dir, { withFileTypes: true })
            .filter((f) => mustacheConfig.hasApi ||
            (!`${pathPrefixes.join("/")}${f.name}`.includes("DefaultApi") &&
                !`${pathPrefixes.join("/")}${f.name}`.includes("ApiExplorer")))
            .flatMap((f) => f.isDirectory()
            ? this.buildSampleDirEntries(`${dir}/${f.name}`, [...pathPrefixes, f.name], mustacheConfig)
            : [
                [
                    `${path.join(...pathPrefixes, f.name.replace(".mustache", ""))}`,
                    Mustache.render(fs.readFileSync(`${dir}/${f.name}`).toString(), mustacheConfig),
                ],
            ]);
    }
}
exports.CloudscapeReactTsWebsiteProject = CloudscapeReactTsWebsiteProject;
_a = JSII_RTTI_SYMBOL_1;
CloudscapeReactTsWebsiteProject[_a] = { fqn: "@aws/pdk.cloudscape_react_ts_website.CloudscapeReactTsWebsiteProject", version: "0.22.14" };
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiY2xvdWRzY2FwZS1yZWFjdC10cy13ZWJzaXRlLXByb2plY3QuanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyJjbG91ZHNjYXBlLXJlYWN0LXRzLXdlYnNpdGUtcHJvamVjdC50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiOzs7OztBQUFBO3NDQUNzQztBQUN0Qyx5QkFBeUI7QUFDekIsNkJBQTZCO0FBRTdCLHFDQUFxQztBQUNyQyxtQ0FBbUM7QUFDbkMsc0RBQW9EO0FBQ3BELHdDQUF3RDtBQTJCeEQ7Ozs7R0FJRztBQUNILE1BQWEsK0JBQWdDLFNBQVEsNEJBQXNCO0lBSXpFLFlBQVksT0FBK0M7UUFDekQsTUFBTSxNQUFNLEdBQUcsQ0FBQyxDQUFDLE9BQU8sQ0FBQyxXQUFXLENBQUM7UUFFckMsS0FBSyxDQUFDO1lBQ0osR0FBRyxPQUFPO1lBQ1Ysb0JBQW9CLEVBQUUsT0FBTyxDQUFDLG9CQUFvQixJQUFJLE1BQU07WUFDNUQsSUFBSSxFQUFFLE9BQU8sQ0FBQyxJQUFJO1lBQ2xCLFVBQVUsRUFBRSxLQUFLO1lBQ2pCLFFBQVEsRUFBRSxPQUFPLENBQUMsUUFBUSxJQUFJLElBQUk7WUFDbEMsY0FBYyxFQUNaLE9BQU8sQ0FBQyxNQUFNLElBQUksT0FBTyxDQUFDLE1BQU0sWUFBWSx3QkFBVztnQkFDckQsQ0FBQyxDQUFDLE9BQU8sQ0FBQyxNQUFNLENBQUMsT0FBTyxDQUFDLGNBQWM7Z0JBQ3ZDLENBQUMsQ0FBQyxPQUFPLENBQUMsY0FBYztZQUM1QixNQUFNLEVBQUU7Z0JBQ04sUUFBUSxFQUFFLEVBQUU7cUJBQ1QsWUFBWSxDQUNYLElBQUksQ0FBQyxPQUFPLENBQ1YsU0FBUyxFQUNULGtEQUFrRCxDQUNuRCxDQUNGO3FCQUNBLFFBQVEsRUFBRTthQUNkO1lBQ0QsU0FBUyxFQUFFO2dCQUNULDRCQUE0QjtnQkFDNUIsaUJBQWlCO2dCQUNqQixHQUFHLENBQUMsT0FBTyxDQUFDLFNBQVMsSUFBSSxFQUFFLENBQUM7YUFDN0I7U0FDRixDQUFDLENBQUM7UUFFSCxJQUFJLENBQUMsT0FBTyxDQUNWLG1CQUFtQixFQUNuQiwrQkFBK0IsRUFDL0IscUNBQXFDLEVBQ3JDLGtCQUFrQixDQUNuQixDQUFDO1FBRUYsSUFBSSxDQUFDLFFBQVEsQ0FBQyxLQUFLLEVBQUUsQ0FBQztRQUN0QixNQUFNLFFBQVEsR0FBRyxJQUFJLENBQUMsS0FBSyxDQUFDLE9BQU8sQ0FBQyxRQUFRLENBQUMsQ0FBQztRQUM5QyxRQUFRLElBQUksSUFBSSxDQUFDLFFBQVEsQ0FBQyxLQUFLLENBQUMsUUFBUSxDQUFDLENBQUM7UUFDMUMsSUFBSSxDQUFDLFFBQVEsQ0FBQyxJQUFJLENBQUMsdURBQXVELENBQUMsQ0FBQztRQUU1RSxJQUFJLENBQUMsZUFBZSxHQUFHLE9BQU8sQ0FBQyxlQUFlLElBQUksWUFBWSxDQUFDO1FBQy9ELElBQUksQ0FBQyxTQUFTLEdBQUcsT0FBTyxDQUFDLFNBQVMsSUFBSSxRQUFRLENBQUM7UUFDL0MsTUFBTSxNQUFNLEdBQUcsSUFBSSxDQUFDLE9BQU8sQ0FDekIsU0FBUyxFQUNULDRDQUE0QyxDQUM3QyxDQUFDO1FBQ0YsTUFBTSxTQUFTLEdBQUcsSUFBSSxDQUFDLE9BQU8sQ0FDNUIsU0FBUyxFQUNULCtDQUErQyxDQUNoRCxDQUFDO1FBRUYsSUFBSSxPQUFPLENBQUMsV0FBVyxFQUFFO1lBQ3ZCLE1BQU0sS0FBSyxHQUFHLE9BQU8sQ0FBQyxXQUFXLENBQUMsT0FBTyxFQUFFLHlCQUF5QixDQUFDO1lBQ3JFLE1BQU0sbUJBQW1CLEdBQUcsS0FBSyxFQUFFLE9BQU8sRUFBRSxXQUFXLENBQUM7WUFDeEQsSUFBSSxDQUFDLG1CQUFtQixFQUFFO2dCQUN4QixNQUFNLElBQUksS0FBSyxDQUNiLHdFQUF3RSxDQUN6RSxDQUFDO2FBQ0g7WUFDRCxJQUFJLENBQUMsT0FBTyxDQUFDLG1CQUFtQixDQUFDLENBQUM7WUFFbEMsSUFBSSxDQUFDLGNBQWMsQ0FBQyxPQUFPLENBQUMsV0FBVyxDQUFDLENBQUM7U0FDMUM7UUFFRCxNQUFNLGNBQWMsR0FBRztZQUNyQixlQUFlLEVBQUUsSUFBSSxDQUFDLGVBQWU7WUFDckMsTUFBTTtZQUNOLGVBQWUsRUFDYixPQUFPLENBQUMsV0FBVyxFQUFFLE9BQU8sRUFBRSx5QkFBeUIsRUFBRSxPQUFPO2dCQUM5RCxFQUFFLFdBQVc7U0FDbEIsQ0FBQztRQUVGLElBQUksa0JBQVMsQ0FBQyxJQUFJLEVBQUUsSUFBSSxDQUFDLE1BQU0sRUFBRTtZQUMvQixLQUFLLEVBQUU7Z0JBQ0wsR0FBRyxNQUFNLENBQUMsV0FBVyxDQUNuQixJQUFJLENBQUMscUJBQXFCLENBQUMsTUFBTSxFQUFFLEVBQUUsRUFBRSxjQUFjLENBQUMsQ0FDdkQ7YUFDRjtTQUNGLENBQUMsQ0FBQztRQUVILElBQUksa0JBQVMsQ0FBQyxJQUFJLEVBQUUsSUFBSSxDQUFDLFNBQVMsRUFBRTtZQUNsQyxLQUFLLEVBQUU7Z0JBQ0wsR0FBRyxNQUFNLENBQUMsV0FBVyxDQUNuQixJQUFJLENBQUMscUJBQXFCLENBQUMsU0FBUyxFQUFFLEVBQUUsRUFBRSxjQUFjLENBQUMsQ0FDMUQ7YUFDRjtTQUNGLENBQUMsQ0FBQztRQUVILHFHQUFxRztRQUNyRyxJQUFJLENBQUMsS0FBSyxDQUFDLGNBQWMsQ0FBQyx1QkFBdUIsRUFBRSxNQUFNLENBQUMsQ0FBQztRQUUzRCwrQkFBK0I7UUFDL0IsSUFBSSxDQUFDLEtBQUssQ0FBQyxPQUFPLENBQUMsS0FBSyxDQUFDLEVBQUUsR0FBRyxDQUFDLHNCQUFzQixFQUFFLE1BQU0sQ0FBQyxDQUFDO1FBQy9ELElBQUksQ0FBQyxLQUFLLENBQUMsT0FBTyxDQUFDLEtBQUssQ0FBQyxFQUFFLEdBQUcsQ0FBQyxzQkFBc0IsRUFBRSxNQUFNLENBQUMsQ0FBQztJQUNqRSxDQUFDO0lBRU8sY0FBYyxDQUFDLEtBQXlCO1FBQzlDLElBQUksQ0FBQyxVQUFVLENBQUMseUJBQXlCLENBQUMsQ0FBQztRQUMzQyxJQUFJLENBQUMsT0FBTyxDQUFDLHdCQUF3QixFQUFFLFdBQVcsQ0FBQyxDQUFDO1FBRXBELE1BQU0saUJBQWlCLEdBQUcsR0FBRyxJQUFJLENBQUMsUUFBUSxDQUN4QyxLQUFLLENBQUMsS0FBSyxDQUFDLE1BQU0sRUFDbEIsSUFBSSxDQUFDLE1BQU0sQ0FDWixrQkFBa0IsQ0FBQztRQUNwQixLQUFLLENBQUMsS0FBSyxDQUFDLGVBQWUsQ0FBQyxJQUFJLENBQUMsU0FBUyxpQkFBaUIsRUFBRSxDQUFDLENBQUM7UUFDL0QsS0FBSyxDQUFDLEtBQUssQ0FBQyxlQUFlLENBQUMsSUFBSSxDQUFDLGdCQUFnQixpQkFBaUIsRUFBRSxDQUFDLENBQUM7SUFDeEUsQ0FBQztJQUVPLHFCQUFxQixDQUMzQixHQUFXLEVBQ1gsZUFBeUIsRUFBRSxFQUMzQixjQUFtQjtRQUVuQixPQUFPLEVBQUU7YUFDTixXQUFXLENBQUMsR0FBRyxFQUFFLEVBQUUsYUFBYSxFQUFFLElBQUksRUFBRSxDQUFDO2FBQ3pDLE1BQU0sQ0FDTCxDQUFDLENBQUMsRUFBRSxFQUFFLENBQ0osY0FBYyxDQUFDLE1BQU07WUFDckIsQ0FBQyxDQUFDLEdBQUcsWUFBWSxDQUFDLElBQUksQ0FBQyxHQUFHLENBQUMsR0FBRyxDQUFDLENBQUMsSUFBSSxFQUFFLENBQUMsUUFBUSxDQUFDLFlBQVksQ0FBQztnQkFDM0QsQ0FBQyxHQUFHLFlBQVksQ0FBQyxJQUFJLENBQUMsR0FBRyxDQUFDLEdBQUcsQ0FBQyxDQUFDLElBQUksRUFBRSxDQUFDLFFBQVEsQ0FBQyxhQUFhLENBQUMsQ0FBQyxDQUNuRTthQUNBLE9BQU8sQ0FBQyxDQUFDLENBQUMsRUFBRSxFQUFFLENBQ2IsQ0FBQyxDQUFDLFdBQVcsRUFBRTtZQUNiLENBQUMsQ0FBQyxJQUFJLENBQUMscUJBQXFCLENBQ3hCLEdBQUcsR0FBRyxJQUFJLENBQUMsQ0FBQyxJQUFJLEVBQUUsRUFDbEIsQ0FBQyxHQUFHLFlBQVksRUFBRSxDQUFDLENBQUMsSUFBSSxDQUFDLEVBQ3pCLGNBQWMsQ0FDZjtZQUNILENBQUMsQ0FBQztnQkFDRTtvQkFDRSxHQUFHLElBQUksQ0FBQyxJQUFJLENBQ1YsR0FBRyxZQUFZLEVBQ2YsQ0FBQyxDQUFDLElBQUksQ0FBQyxPQUFPLENBQUMsV0FBVyxFQUFFLEVBQUUsQ0FBQyxDQUNoQyxFQUFFO29CQUNILFFBQVEsQ0FBQyxNQUFNLENBQ2IsRUFBRSxDQUFDLFlBQVksQ0FBQyxHQUFHLEdBQUcsSUFBSSxDQUFDLENBQUMsSUFBSSxFQUFFLENBQUMsQ0FBQyxRQUFRLEVBQUUsRUFDOUMsY0FBYyxDQUNmO2lCQUNGO2FBQ0YsQ0FDTixDQUFDO0lBQ04sQ0FBQzs7QUFuSkgsMEVBb0pDIiwic291cmNlc0NvbnRlbnQiOlsiLyohIENvcHlyaWdodCBbQW1hem9uLmNvbV0oaHR0cDovL2FtYXpvbi5jb20vKSwgSW5jLiBvciBpdHMgYWZmaWxpYXRlcy4gQWxsIFJpZ2h0cyBSZXNlcnZlZC5cblNQRFgtTGljZW5zZS1JZGVudGlmaWVyOiBBcGFjaGUtMi4wICovXG5pbXBvcnQgKiBhcyBmcyBmcm9tIFwiZnNcIjtcbmltcG9ydCAqIGFzIHBhdGggZnJvbSBcInBhdGhcIjtcbmltcG9ydCB7IFR5cGVTYWZlQXBpUHJvamVjdCB9IGZyb20gXCJAYXdzL3R5cGUtc2FmZS1hcGlcIjtcbmltcG9ydCAqIGFzIE11c3RhY2hlIGZyb20gXCJtdXN0YWNoZVwiO1xuaW1wb3J0IHsgU2FtcGxlRGlyIH0gZnJvbSBcInByb2plblwiO1xuaW1wb3J0IHsgTm9kZVByb2plY3QgfSBmcm9tIFwicHJvamVuL2xpYi9qYXZhc2NyaXB0XCI7XG5pbXBvcnQgeyBSZWFjdFR5cGVTY3JpcHRQcm9qZWN0IH0gZnJvbSBcInByb2plbi9saWIvd2ViXCI7XG5pbXBvcnQgeyBSZWFjdFR5cGVTY3JpcHRQcm9qZWN0T3B0aW9ucyB9IGZyb20gXCIuL3JlYWN0LXRzLXByb2plY3Qtb3B0aW9uc1wiO1xuXG4vKipcbiAqIENvbmZpZ3VyYXRpb24gb3B0aW9ucyBmb3IgdGhlIENsb3Vkc2NhcGVSZWFjdFRzV2Vic2l0ZVByb2plY3QuXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgQ2xvdWRzY2FwZVJlYWN0VHNXZWJzaXRlUHJvamVjdE9wdGlvbnNcbiAgZXh0ZW5kcyBSZWFjdFR5cGVTY3JpcHRQcm9qZWN0T3B0aW9ucyB7XG4gIC8qKlxuICAgKiBOYW1lIG9mIHRoZSBhcHBsaWNhdGlvbiBuYW1lLlxuICAgKlxuICAgKiBAZGVmYXVsdCBcIlNhbXBsZSBBcHBcIlxuICAgKi9cbiAgcmVhZG9ubHkgYXBwbGljYXRpb25OYW1lPzogc3RyaW5nO1xuICAvKipcbiAgICogUHVibGljIGRpcmVjdG9yeS5cbiAgICpcbiAgICogQGRlZmF1bHQgXCJwdWJsaWNcIlxuICAgKi9cbiAgcmVhZG9ubHkgcHVibGljRGlyPzogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBUeXBlU2FmZUFwaSBpbnN0YW5jZSB0byB1c2Ugd2hlbiBzZXR0aW5nIHVwIHRoZSBpbml0aWFsIHByb2plY3Qgc2FtcGxlIGNvZGUuXG4gICAqL1xuICByZWFkb25seSB0eXBlU2FmZUFwaT86IFR5cGVTYWZlQXBpUHJvamVjdDtcbn1cblxuLyoqXG4gKiBTeW50aGVzaXplcyBhIENsb3Vkc2NhcGUgUmVhY3QgVHlwZXNjcmlwdCBXZWJzaXRlIFByb2plY3QuXG4gKlxuICogQHBqaWQgY2xvdWRzY2FwZS1yZWFjdC10cy13ZWJzaXRlXG4gKi9cbmV4cG9ydCBjbGFzcyBDbG91ZHNjYXBlUmVhY3RUc1dlYnNpdGVQcm9qZWN0IGV4dGVuZHMgUmVhY3RUeXBlU2NyaXB0UHJvamVjdCB7XG4gIHB1YmxpYyByZWFkb25seSBhcHBsaWNhdGlvbk5hbWU6IHN0cmluZztcbiAgcHVibGljIHJlYWRvbmx5IHB1YmxpY0Rpcjogc3RyaW5nO1xuXG4gIGNvbnN0cnVjdG9yKG9wdGlvbnM6IENsb3Vkc2NhcGVSZWFjdFRzV2Vic2l0ZVByb2plY3RPcHRpb25zKSB7XG4gICAgY29uc3QgaGFzQXBpID0gISFvcHRpb25zLnR5cGVTYWZlQXBpO1xuXG4gICAgc3VwZXIoe1xuICAgICAgLi4ub3B0aW9ucyxcbiAgICAgIGRlZmF1bHRSZWxlYXNlQnJhbmNoOiBvcHRpb25zLmRlZmF1bHRSZWxlYXNlQnJhbmNoID8/IFwibWFpblwiLFxuICAgICAgbmFtZTogb3B0aW9ucy5uYW1lLFxuICAgICAgc2FtcGxlQ29kZTogZmFsc2UsXG4gICAgICBwcmV0dGllcjogb3B0aW9ucy5wcmV0dGllciB8fCB0cnVlLFxuICAgICAgcGFja2FnZU1hbmFnZXI6XG4gICAgICAgIG9wdGlvbnMucGFyZW50ICYmIG9wdGlvbnMucGFyZW50IGluc3RhbmNlb2YgTm9kZVByb2plY3RcbiAgICAgICAgICA/IG9wdGlvbnMucGFyZW50LnBhY2thZ2UucGFja2FnZU1hbmFnZXJcbiAgICAgICAgICA6IG9wdGlvbnMucGFja2FnZU1hbmFnZXIsXG4gICAgICByZWFkbWU6IHtcbiAgICAgICAgY29udGVudHM6IGZzXG4gICAgICAgICAgLnJlYWRGaWxlU3luYyhcbiAgICAgICAgICAgIHBhdGgucmVzb2x2ZShcbiAgICAgICAgICAgICAgX19kaXJuYW1lLFxuICAgICAgICAgICAgICBcIi4uL3NhbXBsZXMvY2xvdWRzY2FwZS1yZWFjdC10cy13ZWJzaXRlL1JFQURNRS5tZFwiXG4gICAgICAgICAgICApXG4gICAgICAgICAgKVxuICAgICAgICAgIC50b1N0cmluZygpLFxuICAgICAgfSxcbiAgICAgIGdpdGlnbm9yZTogW1xuICAgICAgICBcInB1YmxpYy9ydW50aW1lLWNvbmZpZy5qc29uXCIsXG4gICAgICAgIFwicHVibGljL2FwaS5qc29uXCIsXG4gICAgICAgIC4uLihvcHRpb25zLmdpdGlnbm9yZSB8fCBbXSksXG4gICAgICBdLFxuICAgIH0pO1xuXG4gICAgdGhpcy5hZGREZXBzKFxuICAgICAgXCJAYXdzLW5vcnRoc3Rhci91aVwiLFxuICAgICAgXCJAY2xvdWRzY2FwZS1kZXNpZ24vY29tcG9uZW50c1wiLFxuICAgICAgXCJAY2xvdWRzY2FwZS1kZXNpZ24vYm9hcmQtY29tcG9uZW50c1wiLFxuICAgICAgXCJyZWFjdC1yb3V0ZXItZG9tXCJcbiAgICApO1xuXG4gICAgdGhpcy50ZXN0VGFzay5yZXNldCgpO1xuICAgIGNvbnN0IGxpbnRUYXNrID0gdGhpcy50YXNrcy50cnlGaW5kKFwiZXNsaW50XCIpO1xuICAgIGxpbnRUYXNrICYmIHRoaXMudGVzdFRhc2suc3Bhd24obGludFRhc2spO1xuICAgIHRoaXMudGVzdFRhc2suZXhlYyhcInJlYWN0LXNjcmlwdHMgdGVzdCAtLXdhdGNoQWxsPWZhbHNlIC0tcGFzc1dpdGhOb1Rlc3RzXCIpO1xuXG4gICAgdGhpcy5hcHBsaWNhdGlvbk5hbWUgPSBvcHRpb25zLmFwcGxpY2F0aW9uTmFtZSA/PyBcIlNhbXBsZSBBcHBcIjtcbiAgICB0aGlzLnB1YmxpY0RpciA9IG9wdGlvbnMucHVibGljRGlyID8/IFwicHVibGljXCI7XG4gICAgY29uc3Qgc3JjRGlyID0gcGF0aC5yZXNvbHZlKFxuICAgICAgX19kaXJuYW1lLFxuICAgICAgXCIuLi9zYW1wbGVzL2Nsb3Vkc2NhcGUtcmVhY3QtdHMtd2Vic2l0ZS9zcmNcIlxuICAgICk7XG4gICAgY29uc3QgcHVibGljRGlyID0gcGF0aC5yZXNvbHZlKFxuICAgICAgX19kaXJuYW1lLFxuICAgICAgXCIuLi9zYW1wbGVzL2Nsb3Vkc2NhcGUtcmVhY3QtdHMtd2Vic2l0ZS9wdWJsaWNcIlxuICAgICk7XG5cbiAgICBpZiAob3B0aW9ucy50eXBlU2FmZUFwaSkge1xuICAgICAgY29uc3QgaG9va3MgPSBvcHRpb25zLnR5cGVTYWZlQXBpLmxpYnJhcnk/LnR5cGVzY3JpcHRSZWFjdFF1ZXJ5SG9va3M7XG4gICAgICBjb25zdCBsaWJyYXJ5SG9va3NQYWNrYWdlID0gaG9va3M/LnBhY2thZ2U/LnBhY2thZ2VOYW1lO1xuICAgICAgaWYgKCFsaWJyYXJ5SG9va3NQYWNrYWdlKSB7XG4gICAgICAgIHRocm93IG5ldyBFcnJvcihcbiAgICAgICAgICBcIkNhbm5vdCBwYXNzIGluIGEgVHlwZSBTYWZlIEFwaSB3aXRob3V0IFJlYWN0IEhvb2tzIExpYnJhcnkgY29uZmlndXJlZCFcIlxuICAgICAgICApO1xuICAgICAgfVxuICAgICAgdGhpcy5hZGREZXBzKGxpYnJhcnlIb29rc1BhY2thZ2UpO1xuXG4gICAgICB0aGlzLnNldHVwU3dhZ2dlclVpKG9wdGlvbnMudHlwZVNhZmVBcGkpO1xuICAgIH1cblxuICAgIGNvbnN0IG11c3RhY2hlQ29uZmlnID0ge1xuICAgICAgYXBwbGljYXRpb25OYW1lOiB0aGlzLmFwcGxpY2F0aW9uTmFtZSxcbiAgICAgIGhhc0FwaSxcbiAgICAgIGFwaUhvb2tzUGFja2FnZTpcbiAgICAgICAgb3B0aW9ucy50eXBlU2FmZUFwaT8ubGlicmFyeT8udHlwZXNjcmlwdFJlYWN0UXVlcnlIb29rcz8ucGFja2FnZVxuICAgICAgICAgID8ucGFja2FnZU5hbWUsXG4gICAgfTtcblxuICAgIG5ldyBTYW1wbGVEaXIodGhpcywgdGhpcy5zcmNkaXIsIHtcbiAgICAgIGZpbGVzOiB7XG4gICAgICAgIC4uLk9iamVjdC5mcm9tRW50cmllcyhcbiAgICAgICAgICB0aGlzLmJ1aWxkU2FtcGxlRGlyRW50cmllcyhzcmNEaXIsIFtdLCBtdXN0YWNoZUNvbmZpZylcbiAgICAgICAgKSxcbiAgICAgIH0sXG4gICAgfSk7XG5cbiAgICBuZXcgU2FtcGxlRGlyKHRoaXMsIHRoaXMucHVibGljRGlyLCB7XG4gICAgICBmaWxlczoge1xuICAgICAgICAuLi5PYmplY3QuZnJvbUVudHJpZXMoXG4gICAgICAgICAgdGhpcy5idWlsZFNhbXBsZURpckVudHJpZXMocHVibGljRGlyLCBbXSwgbXVzdGFjaGVDb25maWcpXG4gICAgICAgICksXG4gICAgICB9LFxuICAgIH0pO1xuXG4gICAgLy8gTGludGluZyBpcyBtYW5hZ2VkIGFzIHBhcnQgb2YgdGhlIHRlc3QgdGFzayBhbHJlYWR5LCBzbyBkaXNhYmxlIHJlYWN0LXNjcmlwdHMgcnVubmluZyBlc2xpbnQgYWdhaW5cbiAgICB0aGlzLnRhc2tzLmFkZEVudmlyb25tZW50KFwiRElTQUJMRV9FU0xJTlRfUExVR0lOXCIsIFwidHJ1ZVwiKTtcblxuICAgIC8vIFJlbGF4IEVzTGludCBhbmQgVFNDIGZvciBkZXZcbiAgICB0aGlzLnRhc2tzLnRyeUZpbmQoXCJkZXZcIik/LmVudihcIkVTTElOVF9OT19ERVZfRVJST1JTXCIsIFwidHJ1ZVwiKTtcbiAgICB0aGlzLnRhc2tzLnRyeUZpbmQoXCJkZXZcIik/LmVudihcIlRTQ19DT01QSUxFX09OX0VSUk9SXCIsIFwidHJ1ZVwiKTtcbiAgfVxuXG4gIHByaXZhdGUgc2V0dXBTd2FnZ2VyVWkodHNBcGk6IFR5cGVTYWZlQXBpUHJvamVjdCkge1xuICAgIHRoaXMuYWRkRGV2RGVwcyhcIkB0eXBlcy9zd2FnZ2VyLXVpLXJlYWN0XCIpO1xuICAgIHRoaXMuYWRkRGVwcyhcInN3YWdnZXItdWktcmVhY3RANS41LjBcIiwgXCJhd3M0ZmV0Y2hcIik7XG5cbiAgICBjb25zdCB0YXJnZXRBcGlTcGVjUGF0aCA9IGAke3BhdGgucmVsYXRpdmUoXG4gICAgICB0c0FwaS5tb2RlbC5vdXRkaXIsXG4gICAgICB0aGlzLm91dGRpclxuICAgICl9L3B1YmxpYy9hcGkuanNvbmA7XG4gICAgdHNBcGkubW9kZWwucG9zdENvbXBpbGVUYXNrLmV4ZWMoYHJtIC1mICR7dGFyZ2V0QXBpU3BlY1BhdGh9YCk7XG4gICAgdHNBcGkubW9kZWwucG9zdENvbXBpbGVUYXNrLmV4ZWMoYGNwIC5hcGkuanNvbiAke3RhcmdldEFwaVNwZWNQYXRofWApO1xuICB9XG5cbiAgcHJpdmF0ZSBidWlsZFNhbXBsZURpckVudHJpZXMoXG4gICAgZGlyOiBzdHJpbmcsXG4gICAgcGF0aFByZWZpeGVzOiBzdHJpbmdbXSA9IFtdLFxuICAgIG11c3RhY2hlQ29uZmlnOiBhbnlcbiAgKTogW3N0cmluZywgc3RyaW5nXVtdIHtcbiAgICByZXR1cm4gZnNcbiAgICAgIC5yZWFkZGlyU3luYyhkaXIsIHsgd2l0aEZpbGVUeXBlczogdHJ1ZSB9KVxuICAgICAgLmZpbHRlcihcbiAgICAgICAgKGYpID0+XG4gICAgICAgICAgbXVzdGFjaGVDb25maWcuaGFzQXBpIHx8XG4gICAgICAgICAgKCFgJHtwYXRoUHJlZml4ZXMuam9pbihcIi9cIil9JHtmLm5hbWV9YC5pbmNsdWRlcyhcIkRlZmF1bHRBcGlcIikgJiZcbiAgICAgICAgICAgICFgJHtwYXRoUHJlZml4ZXMuam9pbihcIi9cIil9JHtmLm5hbWV9YC5pbmNsdWRlcyhcIkFwaUV4cGxvcmVyXCIpKVxuICAgICAgKVxuICAgICAgLmZsYXRNYXAoKGYpID0+XG4gICAgICAgIGYuaXNEaXJlY3RvcnkoKVxuICAgICAgICAgID8gdGhpcy5idWlsZFNhbXBsZURpckVudHJpZXMoXG4gICAgICAgICAgICAgIGAke2Rpcn0vJHtmLm5hbWV9YCxcbiAgICAgICAgICAgICAgWy4uLnBhdGhQcmVmaXhlcywgZi5uYW1lXSxcbiAgICAgICAgICAgICAgbXVzdGFjaGVDb25maWdcbiAgICAgICAgICAgIClcbiAgICAgICAgICA6IFtcbiAgICAgICAgICAgICAgW1xuICAgICAgICAgICAgICAgIGAke3BhdGguam9pbihcbiAgICAgICAgICAgICAgICAgIC4uLnBhdGhQcmVmaXhlcyxcbiAgICAgICAgICAgICAgICAgIGYubmFtZS5yZXBsYWNlKFwiLm11c3RhY2hlXCIsIFwiXCIpXG4gICAgICAgICAgICAgICAgKX1gLFxuICAgICAgICAgICAgICAgIE11c3RhY2hlLnJlbmRlcihcbiAgICAgICAgICAgICAgICAgIGZzLnJlYWRGaWxlU3luYyhgJHtkaXJ9LyR7Zi5uYW1lfWApLnRvU3RyaW5nKCksXG4gICAgICAgICAgICAgICAgICBtdXN0YWNoZUNvbmZpZ1xuICAgICAgICAgICAgICAgICksXG4gICAgICAgICAgICAgIF0sXG4gICAgICAgICAgICBdXG4gICAgICApO1xuICB9XG59XG4iXX0=