"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.InfrastructureJavaProject = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/*! Copyright [Amazon.com](http://amazon.com/), Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0 */
const fs = require("fs");
const path = require("path");
const monorepo_1 = require("../../../monorepo");
const Mustache = require("mustache");
const projen_1 = require("projen");
const awscdk_1 = require("projen/lib/awscdk");
/**
 * Synthesizes a Infrastructure Typescript Project.
 */
class InfrastructureJavaProject extends awscdk_1.AwsCdkJavaApp {
    constructor(options) {
        const hasApi = !!options.typeSafeApi;
        const hasWebsite = !!options.cloudscapeReactTsWebsite;
        const groupId = options.groupId ?? "software.aws.infra";
        const artifactId = options.artifactId ?? "infra";
        super({
            ...options,
            cdkVersion: options.cdkVersion ?? "2.1.0",
            sample: false,
            junit: false,
            groupId,
            artifactId,
            mainClass: `${groupId}.Main`,
            version: options.version ?? "0.0.0",
            name: options.name,
            readme: {
                contents: fs
                    .readFileSync(path.resolve(__dirname, "../../../samples/infrastructure/java/README.md"))
                    .toString(),
            },
        });
        this.pom.addPlugin("org.apache.maven.plugins/maven-surefire-plugin@3.1.2");
        this.pom.addPlugin("org.apache.maven.plugins/maven-compiler-plugin@3.8.1", {
            configuration: {
                release: "11",
            },
        });
        if (options.junit !== false) {
            [
                "org.junit.jupiter/junit-jupiter-api@^5",
                "org.junit.jupiter/junit-jupiter-engine@^5",
                "io.github.origin-energy/java-snapshot-testing-junit5@^4.0.6",
                "io.github.origin-energy/java-snapshot-testing-plugin-jackson@^4.0.6",
                "org.slf4j/slf4j-simple@2.0.0-alpha0",
            ].forEach((d) => this.addTestDependency(d));
            this.testTask.exec("mvn test");
        }
        this.addDependency("software.aws/pdk@^0");
        const srcDir = path.resolve(__dirname, "../../../samples/infrastructure/java/src");
        const testDir = path.resolve(__dirname, "../../../samples/infrastructure/java/test");
        if (hasApi) {
            if (!options.typeSafeApi.infrastructure.java) {
                throw new Error("Cannot pass in a Type Safe Api without Java Infrastructure configured!");
            }
            monorepo_1.NxProject.ensure(this).addJavaDependency(options.typeSafeApi.infrastructure.java);
            // Ensure handlers are built before infra
            options.typeSafeApi.all.handlers?.forEach((handler) => {
                monorepo_1.NxProject.ensure(this).addImplicitDependency(handler);
            });
        }
        if (hasWebsite) {
            // Ensure website is built before infra
            monorepo_1.NxProject.ensure(this).addImplicitDependency(options.cloudscapeReactTsWebsite);
        }
        const mustacheConfig = {
            hasApi,
            hasWebsite,
            infraPackage: `${options.typeSafeApi?.infrastructure.java?.pom.groupId}.${options.typeSafeApi?.infrastructure.java?.pom.name}.infra`,
            groupId,
            websiteDistRelativePath: hasWebsite &&
                path.relative(this.outdir, `${options.cloudscapeReactTsWebsite?.outdir}/build`),
        };
        options.sample !== false &&
            this.emitSampleFiles(srcDir, ["src", "main"], mustacheConfig);
        options.sample !== false &&
            this.emitSampleFiles(testDir, ["src", "test"], mustacheConfig);
    }
    emitSampleFiles(dir, pathPrefixes = [], mustacheConfig) {
        fs.readdirSync(dir, { withFileTypes: true })
            .filter((f) => {
            if (!mustacheConfig.hasApi) {
                return !f.name.endsWith("api.ts.mustache");
            }
            else if (!mustacheConfig.hasWebsite) {
                return !f.name.endsWith("website.ts.mustache");
            }
            else {
                return true;
            }
        })
            .forEach((f) => {
            if (f.isDirectory()) {
                return this.emitSampleFiles(`${dir}/${f.name}`, [
                    ...pathPrefixes,
                    ...(f.name === "groupId"
                        ? mustacheConfig.groupId.split(".")
                        : [f.name]),
                ], mustacheConfig);
            }
            else {
                const contents = Mustache.render(fs.readFileSync(`${dir}/${f.name}`).toString(), mustacheConfig);
                return new projen_1.SampleFile(this, `${path.join(...pathPrefixes, f.name.replace(".mustache", ""))}`, {
                    contents,
                    sourcePath: (!contents && `${dir}/${f.name}`) || undefined,
                });
            }
        });
    }
}
exports.InfrastructureJavaProject = InfrastructureJavaProject;
_a = JSII_RTTI_SYMBOL_1;
InfrastructureJavaProject[_a] = { fqn: "@aws/pdk.infrastructure.InfrastructureJavaProject", version: "0.22.14" };
//# sourceMappingURL=data:application/json;base64,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