"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.InfrastructurePyProject = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/*! Copyright [Amazon.com](http://amazon.com/), Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0 */
const fs = require("fs");
const path = require("path");
const monorepo_1 = require("../../../monorepo");
const Mustache = require("mustache");
const projen_1 = require("projen");
const awscdk_1 = require("projen/lib/awscdk");
/**
 * Synthesizes a Infrastructure Python Project.
 */
class InfrastructurePyProject extends awscdk_1.AwsCdkPythonApp {
    constructor(options) {
        const hasApi = !!options.typeSafeApi;
        const hasWebsite = !!options.cloudscapeReactTsWebsite;
        const moduleName = options.moduleName ?? "infra";
        super({
            ...options,
            cdkVersion: options.cdkVersion ?? "2.1.0",
            sample: false,
            poetry: true,
            moduleName,
            appEntrypoint: "main.py",
            pytest: options.pytest ?? false,
            version: options.version ?? "0.0.0",
            authorName: options.authorName ?? "pdkuser",
            authorEmail: options.authorEmail ?? "user@pdk.com",
            name: options.name,
            readme: {
                contents: fs
                    .readFileSync(path.resolve(__dirname, "../../../samples/infrastructure/python/README.md"))
                    .toString(),
            },
        });
        ["pytest@^7", "syrupy@^4"].forEach((devDep) => this.addDevDependency(devDep));
        ["aws_pdk@^0", "python@^3.9"].forEach((dep) => this.addDependency(dep));
        const srcDir = path.resolve(__dirname, "../../../samples/infrastructure/python/src");
        const testDir = path.resolve(__dirname, "../../../samples/infrastructure/python/test");
        if (hasApi) {
            if (!options.typeSafeApi.infrastructure.python) {
                throw new Error("Cannot pass in a Type Safe Api without Python Infrastructure configured!");
            }
            monorepo_1.NxProject.ensure(this).addPythonPoetryDependency(options.typeSafeApi.infrastructure.python);
            // Ensure handlers are built before infra
            options.typeSafeApi.all.handlers?.forEach((handler) => {
                monorepo_1.NxProject.ensure(this).addImplicitDependency(handler);
            });
        }
        if (hasWebsite) {
            // Ensure website is built before infra
            monorepo_1.NxProject.ensure(this).addImplicitDependency(options.cloudscapeReactTsWebsite);
        }
        const mustacheConfig = {
            hasApi,
            hasWebsite,
            infraPackage: options.typeSafeApi?.infrastructure.python?.moduleName,
            moduleName,
            websiteDistRelativePath: hasWebsite &&
                path.relative(this.outdir, `${options.cloudscapeReactTsWebsite?.outdir}/build`),
        };
        options.sample !== false &&
            this.emitSampleFiles(srcDir, [this.moduleName], mustacheConfig);
        options.sample !== false &&
            this.emitSampleFiles(testDir, ["tests"], mustacheConfig);
        this.testTask.reset("poetry run pytest ${CI:-'--snapshot-update'}");
    }
    emitSampleFiles(dir, pathPrefixes = [], mustacheConfig) {
        fs.readdirSync(dir, { withFileTypes: true })
            .filter((f) => {
            if (!mustacheConfig.hasApi) {
                return !f.name.endsWith("api.ts.mustache");
            }
            else if (!mustacheConfig.hasWebsite) {
                return !f.name.endsWith("website.ts.mustache");
            }
            else {
                return true;
            }
        })
            .forEach((f) => {
            if (f.isDirectory()) {
                return this.emitSampleFiles(`${dir}/${f.name}`, [...pathPrefixes, f.name], mustacheConfig);
            }
            else {
                const contents = Mustache.render(fs.readFileSync(`${dir}/${f.name}`).toString(), mustacheConfig);
                return new projen_1.SampleFile(this, `${path.join(...(f.name !== "main.py.mustache" ? pathPrefixes : []), // emit at the root so package imports work correctly :(
                f.name.replace(".mustache", ""))}`, {
                    contents,
                    sourcePath: (!contents && `${dir}/${f.name}`) || undefined,
                });
            }
        });
    }
}
exports.InfrastructurePyProject = InfrastructurePyProject;
_a = JSII_RTTI_SYMBOL_1;
InfrastructurePyProject[_a] = { fqn: "@aws/pdk.infrastructure.InfrastructurePyProject", version: "0.22.14" };
//# sourceMappingURL=data:application/json;base64,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