"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.InfrastructureTsProject = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/*! Copyright [Amazon.com](http://amazon.com/), Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0 */
const fs = require("fs");
const path = require("path");
const monorepo_1 = require("../../../monorepo");
const Mustache = require("mustache");
const projen_1 = require("projen");
const awscdk_1 = require("projen/lib/awscdk");
const javascript_1 = require("projen/lib/javascript");
/**
 * Synthesizes a Infrastructure Typescript Project.
 */
class InfrastructureTsProject extends awscdk_1.AwsCdkTypeScriptApp {
    constructor(options) {
        const hasApi = !!options.typeSafeApi;
        const hasWebsite = !!options.cloudscapeReactTsWebsite;
        super({
            ...options,
            defaultReleaseBranch: options.defaultReleaseBranch ?? "main",
            prettier: options.prettier || true,
            packageManager: options.parent && options.parent instanceof javascript_1.NodeProject
                ? options.parent.package.packageManager
                : options.packageManager,
            cdkVersion: options.cdkVersion ?? "2.1.0",
            name: options.name,
            sampleCode: false,
            readme: {
                contents: fs
                    .readFileSync(path.resolve(__dirname, "../../../samples/infrastructure/typescript/README.md"))
                    .toString(),
            },
        });
        this.addDeps("@aws/pdk");
        const srcDir = path.resolve(__dirname, "../../../samples/infrastructure/typescript/src");
        const testDir = path.resolve(__dirname, "../../../samples/infrastructure/typescript/test");
        if (hasApi) {
            if (!options.typeSafeApi.infrastructure.typescript) {
                throw new Error("Cannot pass in a Type Safe Api without Typescript Infrastructure configured!");
            }
            this.addDeps(options.typeSafeApi.infrastructure.typescript?.package.packageName);
            // Ensure handlers are built before infra
            options.typeSafeApi.all.handlers?.forEach((handler) => {
                monorepo_1.NxProject.ensure(this).addImplicitDependency(handler);
            });
        }
        if (hasWebsite) {
            // Ensure website is built before infra
            this.addDevDeps(options.cloudscapeReactTsWebsite.package.packageName);
        }
        const mustacheConfig = {
            hasApi,
            hasWebsite,
            infraPackage: options.typeSafeApi?.infrastructure.typescript?.package.packageName,
            websiteDistRelativePath: hasWebsite &&
                path.relative(this.outdir, `${options.cloudscapeReactTsWebsite?.outdir}/build`),
        };
        options.sampleCode !== false &&
            this.emitSampleFiles(srcDir, ["src"], mustacheConfig);
        options.sampleCode !== false &&
            this.emitSampleFiles(testDir, ["test"], mustacheConfig);
        const eslintTask = this.tasks.tryFind("eslint");
        this.testTask.reset();
        this.testTask.exec("jest --passWithNoTests ${CI:-'--updateSnapshot'}");
        eslintTask && this.testTask.spawn(eslintTask);
    }
    emitSampleFiles(dir, pathPrefixes = [], mustacheConfig) {
        fs.readdirSync(dir, { withFileTypes: true })
            .filter((f) => {
            if (!mustacheConfig.hasApi) {
                return !f.name.endsWith("api.ts.mustache");
            }
            else if (!mustacheConfig.hasWebsite) {
                return !f.name.endsWith("website.ts.mustache");
            }
            else {
                return true;
            }
        })
            .forEach((f) => f.isDirectory()
            ? this.emitSampleFiles(`${dir}/${f.name}`, [...pathPrefixes, f.name], mustacheConfig)
            : new projen_1.SampleFile(this, `${path.join(...pathPrefixes, f.name.replace(".mustache", ""))}`, {
                contents: Mustache.render(fs.readFileSync(`${dir}/${f.name}`).toString(), mustacheConfig),
            }));
    }
}
exports.InfrastructureTsProject = InfrastructureTsProject;
_a = JSII_RTTI_SYMBOL_1;
InfrastructureTsProject[_a] = { fqn: "@aws/pdk.infrastructure.InfrastructureTsProject", version: "0.22.14" };
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiaW5mcmFzdHJ1Y3R1cmUtdHMtcHJvamVjdC5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbImluZnJhc3RydWN0dXJlLXRzLXByb2plY3QudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6Ijs7Ozs7QUFBQTtzQ0FDc0M7QUFDdEMseUJBQXlCO0FBQ3pCLDZCQUE2QjtBQUU3Qiw0Q0FBMEM7QUFFMUMscUNBQXFDO0FBQ3JDLG1DQUFvQztBQUNwQyw4Q0FBd0Q7QUFDeEQsc0RBQW9EO0FBbUJwRDs7R0FFRztBQUNILE1BQWEsdUJBQXdCLFNBQVEsNEJBQW1CO0lBQzlELFlBQVksT0FBdUM7UUFDakQsTUFBTSxNQUFNLEdBQUcsQ0FBQyxDQUFDLE9BQU8sQ0FBQyxXQUFXLENBQUM7UUFDckMsTUFBTSxVQUFVLEdBQUcsQ0FBQyxDQUFDLE9BQU8sQ0FBQyx3QkFBd0IsQ0FBQztRQUV0RCxLQUFLLENBQUM7WUFDSixHQUFHLE9BQU87WUFDVixvQkFBb0IsRUFBRSxPQUFPLENBQUMsb0JBQW9CLElBQUksTUFBTTtZQUM1RCxRQUFRLEVBQUUsT0FBTyxDQUFDLFFBQVEsSUFBSSxJQUFJO1lBQ2xDLGNBQWMsRUFDWixPQUFPLENBQUMsTUFBTSxJQUFJLE9BQU8sQ0FBQyxNQUFNLFlBQVksd0JBQVc7Z0JBQ3JELENBQUMsQ0FBQyxPQUFPLENBQUMsTUFBTSxDQUFDLE9BQU8sQ0FBQyxjQUFjO2dCQUN2QyxDQUFDLENBQUMsT0FBTyxDQUFDLGNBQWM7WUFDNUIsVUFBVSxFQUFFLE9BQU8sQ0FBQyxVQUFVLElBQUksT0FBTztZQUN6QyxJQUFJLEVBQUUsT0FBTyxDQUFDLElBQUk7WUFDbEIsVUFBVSxFQUFFLEtBQUs7WUFDakIsTUFBTSxFQUFFO2dCQUNOLFFBQVEsRUFBRSxFQUFFO3FCQUNULFlBQVksQ0FDWCxJQUFJLENBQUMsT0FBTyxDQUNWLFNBQVMsRUFDVCxzREFBc0QsQ0FDdkQsQ0FDRjtxQkFDQSxRQUFRLEVBQUU7YUFDZDtTQUNGLENBQUMsQ0FBQztRQUVILElBQUksQ0FBQyxPQUFPLENBQUMsVUFBVSxDQUFDLENBQUM7UUFFekIsTUFBTSxNQUFNLEdBQUcsSUFBSSxDQUFDLE9BQU8sQ0FDekIsU0FBUyxFQUNULGdEQUFnRCxDQUNqRCxDQUFDO1FBQ0YsTUFBTSxPQUFPLEdBQUcsSUFBSSxDQUFDLE9BQU8sQ0FDMUIsU0FBUyxFQUNULGlEQUFpRCxDQUNsRCxDQUFDO1FBRUYsSUFBSSxNQUFNLEVBQUU7WUFDVixJQUFJLENBQUMsT0FBTyxDQUFDLFdBQVcsQ0FBQyxjQUFjLENBQUMsVUFBVSxFQUFFO2dCQUNsRCxNQUFNLElBQUksS0FBSyxDQUNiLDhFQUE4RSxDQUMvRSxDQUFDO2FBQ0g7WUFDRCxJQUFJLENBQUMsT0FBTyxDQUNWLE9BQU8sQ0FBQyxXQUFXLENBQUMsY0FBYyxDQUFDLFVBQVUsRUFBRSxPQUFPLENBQUMsV0FBWSxDQUNwRSxDQUFDO1lBQ0YseUNBQXlDO1lBQ3pDLE9BQU8sQ0FBQyxXQUFXLENBQUMsR0FBRyxDQUFDLFFBQVEsRUFBRSxPQUFPLENBQUMsQ0FBQyxPQUFPLEVBQUUsRUFBRTtnQkFDcEQsb0JBQVMsQ0FBQyxNQUFNLENBQUMsSUFBSSxDQUFDLENBQUMscUJBQXFCLENBQUMsT0FBTyxDQUFDLENBQUM7WUFDeEQsQ0FBQyxDQUFDLENBQUM7U0FDSjtRQUNELElBQUksVUFBVSxFQUFFO1lBQ2QsdUNBQXVDO1lBQ3ZDLElBQUksQ0FBQyxVQUFVLENBQUMsT0FBTyxDQUFDLHdCQUF3QixDQUFDLE9BQU8sQ0FBQyxXQUFXLENBQUMsQ0FBQztTQUN2RTtRQUVELE1BQU0sY0FBYyxHQUFHO1lBQ3JCLE1BQU07WUFDTixVQUFVO1lBQ1YsWUFBWSxFQUNWLE9BQU8sQ0FBQyxXQUFXLEVBQUUsY0FBYyxDQUFDLFVBQVUsRUFBRSxPQUFPLENBQUMsV0FBVztZQUNyRSx1QkFBdUIsRUFDckIsVUFBVTtnQkFDVixJQUFJLENBQUMsUUFBUSxDQUNYLElBQUksQ0FBQyxNQUFNLEVBQ1gsR0FBRyxPQUFPLENBQUMsd0JBQXdCLEVBQUUsTUFBTSxRQUFRLENBQ3BEO1NBQ0osQ0FBQztRQUVGLE9BQU8sQ0FBQyxVQUFVLEtBQUssS0FBSztZQUMxQixJQUFJLENBQUMsZUFBZSxDQUFDLE1BQU0sRUFBRSxDQUFDLEtBQUssQ0FBQyxFQUFFLGNBQWMsQ0FBQyxDQUFDO1FBQ3hELE9BQU8sQ0FBQyxVQUFVLEtBQUssS0FBSztZQUMxQixJQUFJLENBQUMsZUFBZSxDQUFDLE9BQU8sRUFBRSxDQUFDLE1BQU0sQ0FBQyxFQUFFLGNBQWMsQ0FBQyxDQUFDO1FBRTFELE1BQU0sVUFBVSxHQUFHLElBQUksQ0FBQyxLQUFLLENBQUMsT0FBTyxDQUFDLFFBQVEsQ0FBQyxDQUFDO1FBQ2hELElBQUksQ0FBQyxRQUFRLENBQUMsS0FBSyxFQUFFLENBQUM7UUFDdEIsSUFBSSxDQUFDLFFBQVEsQ0FBQyxJQUFJLENBQUMsa0RBQWtELENBQUMsQ0FBQztRQUN2RSxVQUFVLElBQUksSUFBSSxDQUFDLFFBQVEsQ0FBQyxLQUFLLENBQUMsVUFBVSxDQUFDLENBQUM7SUFDaEQsQ0FBQztJQUVPLGVBQWUsQ0FDckIsR0FBVyxFQUNYLGVBQXlCLEVBQUUsRUFDM0IsY0FBbUI7UUFFbkIsRUFBRSxDQUFDLFdBQVcsQ0FBQyxHQUFHLEVBQUUsRUFBRSxhQUFhLEVBQUUsSUFBSSxFQUFFLENBQUM7YUFDekMsTUFBTSxDQUFDLENBQUMsQ0FBQyxFQUFFLEVBQUU7WUFDWixJQUFJLENBQUMsY0FBYyxDQUFDLE1BQU0sRUFBRTtnQkFDMUIsT0FBTyxDQUFDLENBQUMsQ0FBQyxJQUFJLENBQUMsUUFBUSxDQUFDLGlCQUFpQixDQUFDLENBQUM7YUFDNUM7aUJBQU0sSUFBSSxDQUFDLGNBQWMsQ0FBQyxVQUFVLEVBQUU7Z0JBQ3JDLE9BQU8sQ0FBQyxDQUFDLENBQUMsSUFBSSxDQUFDLFFBQVEsQ0FBQyxxQkFBcUIsQ0FBQyxDQUFDO2FBQ2hEO2lCQUFNO2dCQUNMLE9BQU8sSUFBSSxDQUFDO2FBQ2I7UUFDSCxDQUFDLENBQUM7YUFDRCxPQUFPLENBQUMsQ0FBQyxDQUFDLEVBQUUsRUFBRSxDQUNiLENBQUMsQ0FBQyxXQUFXLEVBQUU7WUFDYixDQUFDLENBQUMsSUFBSSxDQUFDLGVBQWUsQ0FDbEIsR0FBRyxHQUFHLElBQUksQ0FBQyxDQUFDLElBQUksRUFBRSxFQUNsQixDQUFDLEdBQUcsWUFBWSxFQUFFLENBQUMsQ0FBQyxJQUFJLENBQUMsRUFDekIsY0FBYyxDQUNmO1lBQ0gsQ0FBQyxDQUFDLElBQUksbUJBQVUsQ0FDWixJQUFJLEVBQ0osR0FBRyxJQUFJLENBQUMsSUFBSSxDQUFDLEdBQUcsWUFBWSxFQUFFLENBQUMsQ0FBQyxJQUFJLENBQUMsT0FBTyxDQUFDLFdBQVcsRUFBRSxFQUFFLENBQUMsQ0FBQyxFQUFFLEVBQ2hFO2dCQUNFLFFBQVEsRUFBRSxRQUFRLENBQUMsTUFBTSxDQUN2QixFQUFFLENBQUMsWUFBWSxDQUFDLEdBQUcsR0FBRyxJQUFJLENBQUMsQ0FBQyxJQUFJLEVBQUUsQ0FBQyxDQUFDLFFBQVEsRUFBRSxFQUM5QyxjQUFjLENBQ2Y7YUFDRixDQUNGLENBQ04sQ0FBQztJQUNOLENBQUM7O0FBbkhILDBEQW9IQyIsInNvdXJjZXNDb250ZW50IjpbIi8qISBDb3B5cmlnaHQgW0FtYXpvbi5jb21dKGh0dHA6Ly9hbWF6b24uY29tLyksIEluYy4gb3IgaXRzIGFmZmlsaWF0ZXMuIEFsbCBSaWdodHMgUmVzZXJ2ZWQuXG5TUERYLUxpY2Vuc2UtSWRlbnRpZmllcjogQXBhY2hlLTIuMCAqL1xuaW1wb3J0ICogYXMgZnMgZnJvbSBcImZzXCI7XG5pbXBvcnQgKiBhcyBwYXRoIGZyb20gXCJwYXRoXCI7XG5pbXBvcnQgeyBDbG91ZHNjYXBlUmVhY3RUc1dlYnNpdGVQcm9qZWN0IH0gZnJvbSBcIkBhd3MvY2xvdWRzY2FwZS1yZWFjdC10cy13ZWJzaXRlXCI7XG5pbXBvcnQgeyBOeFByb2plY3QgfSBmcm9tIFwiQGF3cy9tb25vcmVwb1wiO1xuaW1wb3J0IHsgVHlwZVNhZmVBcGlQcm9qZWN0IH0gZnJvbSBcIkBhd3MvdHlwZS1zYWZlLWFwaVwiO1xuaW1wb3J0ICogYXMgTXVzdGFjaGUgZnJvbSBcIm11c3RhY2hlXCI7XG5pbXBvcnQgeyBTYW1wbGVGaWxlIH0gZnJvbSBcInByb2plblwiO1xuaW1wb3J0IHsgQXdzQ2RrVHlwZVNjcmlwdEFwcCB9IGZyb20gXCJwcm9qZW4vbGliL2F3c2Nka1wiO1xuaW1wb3J0IHsgTm9kZVByb2plY3QgfSBmcm9tIFwicHJvamVuL2xpYi9qYXZhc2NyaXB0XCI7XG5pbXBvcnQgeyBBd3NDZGtUeXBlU2NyaXB0QXBwT3B0aW9ucyB9IGZyb20gXCIuL2F3cy1jZGstdHMtYXBwLW9wdGlvbnNcIjtcblxuLyoqXG4gKiBDb25maWd1cmF0aW9uIG9wdGlvbnMgZm9yIHRoZSBJbmZyYXN0cnVjdHVyZVRzUHJvamVjdC5cbiAqL1xuZXhwb3J0IGludGVyZmFjZSBJbmZyYXN0cnVjdHVyZVRzUHJvamVjdE9wdGlvbnNcbiAgZXh0ZW5kcyBBd3NDZGtUeXBlU2NyaXB0QXBwT3B0aW9ucyB7XG4gIC8qKlxuICAgKiBUeXBlU2FmZUFwaSBpbnN0YW5jZSB0byB1c2Ugd2hlbiBzZXR0aW5nIHVwIHRoZSBpbml0aWFsIHByb2plY3Qgc2FtcGxlIGNvZGUuXG4gICAqL1xuICByZWFkb25seSB0eXBlU2FmZUFwaT86IFR5cGVTYWZlQXBpUHJvamVjdDtcblxuICAvKipcbiAgICogQ2xvdWRzY2FwZVJlYWN0VHNXZWJzaXRlUHJvamVjdCBpbnN0YW5jZSB0byB1c2Ugd2hlbiBzZXR0aW5nIHVwIHRoZSBpbml0aWFsIHByb2plY3Qgc2FtcGxlIGNvZGUuXG4gICAqL1xuICByZWFkb25seSBjbG91ZHNjYXBlUmVhY3RUc1dlYnNpdGU/OiBDbG91ZHNjYXBlUmVhY3RUc1dlYnNpdGVQcm9qZWN0O1xufVxuXG4vKipcbiAqIFN5bnRoZXNpemVzIGEgSW5mcmFzdHJ1Y3R1cmUgVHlwZXNjcmlwdCBQcm9qZWN0LlxuICovXG5leHBvcnQgY2xhc3MgSW5mcmFzdHJ1Y3R1cmVUc1Byb2plY3QgZXh0ZW5kcyBBd3NDZGtUeXBlU2NyaXB0QXBwIHtcbiAgY29uc3RydWN0b3Iob3B0aW9uczogSW5mcmFzdHJ1Y3R1cmVUc1Byb2plY3RPcHRpb25zKSB7XG4gICAgY29uc3QgaGFzQXBpID0gISFvcHRpb25zLnR5cGVTYWZlQXBpO1xuICAgIGNvbnN0IGhhc1dlYnNpdGUgPSAhIW9wdGlvbnMuY2xvdWRzY2FwZVJlYWN0VHNXZWJzaXRlO1xuXG4gICAgc3VwZXIoe1xuICAgICAgLi4ub3B0aW9ucyxcbiAgICAgIGRlZmF1bHRSZWxlYXNlQnJhbmNoOiBvcHRpb25zLmRlZmF1bHRSZWxlYXNlQnJhbmNoID8/IFwibWFpblwiLFxuICAgICAgcHJldHRpZXI6IG9wdGlvbnMucHJldHRpZXIgfHwgdHJ1ZSxcbiAgICAgIHBhY2thZ2VNYW5hZ2VyOlxuICAgICAgICBvcHRpb25zLnBhcmVudCAmJiBvcHRpb25zLnBhcmVudCBpbnN0YW5jZW9mIE5vZGVQcm9qZWN0XG4gICAgICAgICAgPyBvcHRpb25zLnBhcmVudC5wYWNrYWdlLnBhY2thZ2VNYW5hZ2VyXG4gICAgICAgICAgOiBvcHRpb25zLnBhY2thZ2VNYW5hZ2VyLFxuICAgICAgY2RrVmVyc2lvbjogb3B0aW9ucy5jZGtWZXJzaW9uID8/IFwiMi4xLjBcIixcbiAgICAgIG5hbWU6IG9wdGlvbnMubmFtZSxcbiAgICAgIHNhbXBsZUNvZGU6IGZhbHNlLFxuICAgICAgcmVhZG1lOiB7XG4gICAgICAgIGNvbnRlbnRzOiBmc1xuICAgICAgICAgIC5yZWFkRmlsZVN5bmMoXG4gICAgICAgICAgICBwYXRoLnJlc29sdmUoXG4gICAgICAgICAgICAgIF9fZGlybmFtZSxcbiAgICAgICAgICAgICAgXCIuLi8uLi8uLi9zYW1wbGVzL2luZnJhc3RydWN0dXJlL3R5cGVzY3JpcHQvUkVBRE1FLm1kXCJcbiAgICAgICAgICAgIClcbiAgICAgICAgICApXG4gICAgICAgICAgLnRvU3RyaW5nKCksXG4gICAgICB9LFxuICAgIH0pO1xuXG4gICAgdGhpcy5hZGREZXBzKFwiQGF3cy9wZGtcIik7XG5cbiAgICBjb25zdCBzcmNEaXIgPSBwYXRoLnJlc29sdmUoXG4gICAgICBfX2Rpcm5hbWUsXG4gICAgICBcIi4uLy4uLy4uL3NhbXBsZXMvaW5mcmFzdHJ1Y3R1cmUvdHlwZXNjcmlwdC9zcmNcIlxuICAgICk7XG4gICAgY29uc3QgdGVzdERpciA9IHBhdGgucmVzb2x2ZShcbiAgICAgIF9fZGlybmFtZSxcbiAgICAgIFwiLi4vLi4vLi4vc2FtcGxlcy9pbmZyYXN0cnVjdHVyZS90eXBlc2NyaXB0L3Rlc3RcIlxuICAgICk7XG5cbiAgICBpZiAoaGFzQXBpKSB7XG4gICAgICBpZiAoIW9wdGlvbnMudHlwZVNhZmVBcGkuaW5mcmFzdHJ1Y3R1cmUudHlwZXNjcmlwdCkge1xuICAgICAgICB0aHJvdyBuZXcgRXJyb3IoXG4gICAgICAgICAgXCJDYW5ub3QgcGFzcyBpbiBhIFR5cGUgU2FmZSBBcGkgd2l0aG91dCBUeXBlc2NyaXB0IEluZnJhc3RydWN0dXJlIGNvbmZpZ3VyZWQhXCJcbiAgICAgICAgKTtcbiAgICAgIH1cbiAgICAgIHRoaXMuYWRkRGVwcyhcbiAgICAgICAgb3B0aW9ucy50eXBlU2FmZUFwaS5pbmZyYXN0cnVjdHVyZS50eXBlc2NyaXB0Py5wYWNrYWdlLnBhY2thZ2VOYW1lIVxuICAgICAgKTtcbiAgICAgIC8vIEVuc3VyZSBoYW5kbGVycyBhcmUgYnVpbHQgYmVmb3JlIGluZnJhXG4gICAgICBvcHRpb25zLnR5cGVTYWZlQXBpLmFsbC5oYW5kbGVycz8uZm9yRWFjaCgoaGFuZGxlcikgPT4ge1xuICAgICAgICBOeFByb2plY3QuZW5zdXJlKHRoaXMpLmFkZEltcGxpY2l0RGVwZW5kZW5jeShoYW5kbGVyKTtcbiAgICAgIH0pO1xuICAgIH1cbiAgICBpZiAoaGFzV2Vic2l0ZSkge1xuICAgICAgLy8gRW5zdXJlIHdlYnNpdGUgaXMgYnVpbHQgYmVmb3JlIGluZnJhXG4gICAgICB0aGlzLmFkZERldkRlcHMob3B0aW9ucy5jbG91ZHNjYXBlUmVhY3RUc1dlYnNpdGUucGFja2FnZS5wYWNrYWdlTmFtZSk7XG4gICAgfVxuXG4gICAgY29uc3QgbXVzdGFjaGVDb25maWcgPSB7XG4gICAgICBoYXNBcGksXG4gICAgICBoYXNXZWJzaXRlLFxuICAgICAgaW5mcmFQYWNrYWdlOlxuICAgICAgICBvcHRpb25zLnR5cGVTYWZlQXBpPy5pbmZyYXN0cnVjdHVyZS50eXBlc2NyaXB0Py5wYWNrYWdlLnBhY2thZ2VOYW1lLFxuICAgICAgd2Vic2l0ZURpc3RSZWxhdGl2ZVBhdGg6XG4gICAgICAgIGhhc1dlYnNpdGUgJiZcbiAgICAgICAgcGF0aC5yZWxhdGl2ZShcbiAgICAgICAgICB0aGlzLm91dGRpcixcbiAgICAgICAgICBgJHtvcHRpb25zLmNsb3Vkc2NhcGVSZWFjdFRzV2Vic2l0ZT8ub3V0ZGlyfS9idWlsZGBcbiAgICAgICAgKSxcbiAgICB9O1xuXG4gICAgb3B0aW9ucy5zYW1wbGVDb2RlICE9PSBmYWxzZSAmJlxuICAgICAgdGhpcy5lbWl0U2FtcGxlRmlsZXMoc3JjRGlyLCBbXCJzcmNcIl0sIG11c3RhY2hlQ29uZmlnKTtcbiAgICBvcHRpb25zLnNhbXBsZUNvZGUgIT09IGZhbHNlICYmXG4gICAgICB0aGlzLmVtaXRTYW1wbGVGaWxlcyh0ZXN0RGlyLCBbXCJ0ZXN0XCJdLCBtdXN0YWNoZUNvbmZpZyk7XG5cbiAgICBjb25zdCBlc2xpbnRUYXNrID0gdGhpcy50YXNrcy50cnlGaW5kKFwiZXNsaW50XCIpO1xuICAgIHRoaXMudGVzdFRhc2sucmVzZXQoKTtcbiAgICB0aGlzLnRlc3RUYXNrLmV4ZWMoXCJqZXN0IC0tcGFzc1dpdGhOb1Rlc3RzICR7Q0k6LSctLXVwZGF0ZVNuYXBzaG90J31cIik7XG4gICAgZXNsaW50VGFzayAmJiB0aGlzLnRlc3RUYXNrLnNwYXduKGVzbGludFRhc2spO1xuICB9XG5cbiAgcHJpdmF0ZSBlbWl0U2FtcGxlRmlsZXMoXG4gICAgZGlyOiBzdHJpbmcsXG4gICAgcGF0aFByZWZpeGVzOiBzdHJpbmdbXSA9IFtdLFxuICAgIG11c3RhY2hlQ29uZmlnOiBhbnlcbiAgKSB7XG4gICAgZnMucmVhZGRpclN5bmMoZGlyLCB7IHdpdGhGaWxlVHlwZXM6IHRydWUgfSlcbiAgICAgIC5maWx0ZXIoKGYpID0+IHtcbiAgICAgICAgaWYgKCFtdXN0YWNoZUNvbmZpZy5oYXNBcGkpIHtcbiAgICAgICAgICByZXR1cm4gIWYubmFtZS5lbmRzV2l0aChcImFwaS50cy5tdXN0YWNoZVwiKTtcbiAgICAgICAgfSBlbHNlIGlmICghbXVzdGFjaGVDb25maWcuaGFzV2Vic2l0ZSkge1xuICAgICAgICAgIHJldHVybiAhZi5uYW1lLmVuZHNXaXRoKFwid2Vic2l0ZS50cy5tdXN0YWNoZVwiKTtcbiAgICAgICAgfSBlbHNlIHtcbiAgICAgICAgICByZXR1cm4gdHJ1ZTtcbiAgICAgICAgfVxuICAgICAgfSlcbiAgICAgIC5mb3JFYWNoKChmKSA9PlxuICAgICAgICBmLmlzRGlyZWN0b3J5KClcbiAgICAgICAgICA/IHRoaXMuZW1pdFNhbXBsZUZpbGVzKFxuICAgICAgICAgICAgICBgJHtkaXJ9LyR7Zi5uYW1lfWAsXG4gICAgICAgICAgICAgIFsuLi5wYXRoUHJlZml4ZXMsIGYubmFtZV0sXG4gICAgICAgICAgICAgIG11c3RhY2hlQ29uZmlnXG4gICAgICAgICAgICApXG4gICAgICAgICAgOiBuZXcgU2FtcGxlRmlsZShcbiAgICAgICAgICAgICAgdGhpcyxcbiAgICAgICAgICAgICAgYCR7cGF0aC5qb2luKC4uLnBhdGhQcmVmaXhlcywgZi5uYW1lLnJlcGxhY2UoXCIubXVzdGFjaGVcIiwgXCJcIikpfWAsXG4gICAgICAgICAgICAgIHtcbiAgICAgICAgICAgICAgICBjb250ZW50czogTXVzdGFjaGUucmVuZGVyKFxuICAgICAgICAgICAgICAgICAgZnMucmVhZEZpbGVTeW5jKGAke2Rpcn0vJHtmLm5hbWV9YCkudG9TdHJpbmcoKSxcbiAgICAgICAgICAgICAgICAgIG11c3RhY2hlQ29uZmlnXG4gICAgICAgICAgICAgICAgKSxcbiAgICAgICAgICAgICAgfVxuICAgICAgICAgICAgKVxuICAgICAgKTtcbiAgfVxufVxuIl19