"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.NxConfigurator = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/*! Copyright [Amazon.com](http://amazon.com/), Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0 */
const path = require("path");
const projen_1 = require("projen");
const javascript_1 = require("projen/lib/javascript");
const python_1 = require("projen/lib/python");
const nx_project_1 = require("./nx-project");
const nx_workspace_1 = require("./nx-workspace");
const utils_1 = require("../utils");
/**
 * Configues common NX related tasks and methods.
 */
class NxConfigurator extends projen_1.Component {
    constructor(project, options) {
        super(project);
        this.nxPlugins = {};
        project.addGitIgnore(".nx/cache");
        project.addTask("run-many", {
            receiveArgs: true,
            exec: utils_1.NodePackageUtils.command.exec(utils_1.NodePackageUtils.getNodePackageManager(this.project, javascript_1.NodePackageManager.NPM), "nx", "run-many"),
            description: "Run task against multiple workspace projects",
        });
        project.addTask("graph", {
            receiveArgs: true,
            exec: utils_1.NodePackageUtils.command.exec(utils_1.NodePackageUtils.getNodePackageManager(this.project, javascript_1.NodePackageManager.NPM), "nx", "graph"),
            description: "Generate dependency graph for monorepo",
        });
        this.nx = nx_workspace_1.NxWorkspace.of(project) || new nx_workspace_1.NxWorkspace(project);
        this.nx.affected.defaultBase = options?.defaultReleaseBranch ?? "mainline";
    }
    patchPoetryEnv(project) {
        // Since the root monorepo is a poetry project and sets the VIRTUAL_ENV, and poetry env info -p will print
        // the virtual env set in the VIRTUAL_ENV variable if set, we need to unset it to ensure the local project's
        // env is used.
        if (utils_1.ProjectUtils.isNamedInstanceOf(project.depsManager, python_1.Poetry)) {
            project.tasks.addEnvironment("VIRTUAL_ENV", "$(env -u VIRTUAL_ENV poetry env info -p || echo '')");
            project.tasks.addEnvironment("PATH", "$(echo $(env -u VIRTUAL_ENV poetry env info -p || echo '')/bin:$PATH)");
        }
    }
    patchPythonProjects(projects) {
        projects.forEach((p) => {
            if (utils_1.ProjectUtils.isNamedInstanceOf(p, python_1.PythonProject)) {
                this.patchPoetryEnv(p);
            }
            this.patchPythonProjects(p.subprojects);
        });
    }
    /**
     * Overrides "build" related project tasks (build, compile, test, etc.) with `npx nx run-many` format.
     * @param task - The task or task name to override
     * @param options - Nx run-many options
     * @param overrideOptions - Options for overriding the task
     * @returns - The task that was overridden
     * @internal
     */
    _overrideNxBuildTask(task, options, overrideOptions) {
        if (typeof task === "string") {
            task = this.project.tasks.tryFind(task);
        }
        if (task == null) {
            return;
        }
        if (overrideOptions?.force) {
            // @ts-ignore - private property
            task._locked = false;
        }
        task.reset(this.execNxRunManyCommand(options), {
            receiveArgs: true,
        });
        task.description += " for all affected projects";
        if (overrideOptions?.disableReset) {
            // Prevent any further resets of the task to force it to remain as the overridden nx build task
            task.reset = () => { };
        }
        return task;
    }
    /**
     * Returns the install task or creates one with nx installation command added.
     *
     * Note: this should only be called from non-node projects
     *
     * @param nxPlugins additional plugins to install
     * @returns install task
     */
    ensureNxInstallTask(nxPlugins) {
        this.nxPlugins = nxPlugins;
        const installTask = this.project.tasks.tryFind("install") ?? this.project.addTask("install");
        installTask.exec("yarn install --check-files");
        (this.project.tasks.tryFind("install:ci") ??
            this.project.addTask("install:ci")).exec("yarn install --check-files --frozen-lockfile");
        return installTask;
    }
    /**
     * Helper to format `npx nx run-many ...` style command execution in package manager.
     * @param options
     */
    execNxRunManyCommand(options) {
        return utils_1.NodePackageUtils.command.exec(utils_1.NodePackageUtils.getNodePackageManager(this.project, javascript_1.NodePackageManager.NPM), ...this.composeNxRunManyCommand(options));
    }
    /**
     * Helper to format `npx nx run-many ...` style command
     * @param options
     */
    composeNxRunManyCommand(options) {
        const args = [];
        if (options.configuration) {
            args.push(`--configuration=${options.configuration}`);
        }
        if (options.runner) {
            args.push(`--runner=${options.runner}`);
        }
        if (options.parallel) {
            args.push(`--parallel=${options.parallel}`);
        }
        if (options.skipCache) {
            args.push("--skip-nx-cache");
        }
        if (options.ignoreCycles) {
            args.push("--nx-ignore-cycles");
        }
        if (options.noBail !== true) {
            args.push("--nx-bail");
        }
        if (options.projects && options.projects.length) {
            args.push(`--projects=${options.projects.join(",")}`);
        }
        if (options.exclude) {
            args.push(`--exclude=${options.exclude}`);
        }
        if (options.verbose) {
            args.push("--verbose");
        }
        return [
            "nx",
            "run-many",
            `--target=${options.target}`,
            `--output-style=${options.outputStyle || "stream"}`,
            ...args,
        ];
    }
    /**
     * Add project task that executes `npx nx run-many ...` style command.
     */
    addNxRunManyTask(name, options) {
        return this.project.addTask(name, {
            receiveArgs: true,
            exec: this.execNxRunManyCommand(options),
        });
    }
    /**
     * Create an implicit dependency between two Projects. This is typically
     * used in polygot repos where a Typescript project wants a build dependency
     * on a Python project as an example.
     *
     * @param dependent project you want to have the dependency.
     * @param dependee project you wish to depend on.
     * @throws error if this is called on a dependent which does not have a NXProject component attached.
     */
    addImplicitDependency(dependent, dependee) {
        nx_project_1.NxProject.ensure(dependent).addImplicitDependency(dependee);
    }
    /**
     * Adds a dependency between two Java Projects in the monorepo.
     * @param dependent project you want to have the dependency
     * @param dependee project you wish to depend on
     */
    addJavaDependency(dependent, dependee) {
        nx_project_1.NxProject.ensure(dependent).addJavaDependency(dependee);
    }
    /**
     * Adds a dependency between two Python Projects in the monorepo. The dependent must have Poetry enabled.
     * @param dependent project you want to have the dependency (must be a Poetry Python Project)
     * @param dependee project you wish to depend on
     * @throws error if the dependent does not have Poetry enabled
     */
    addPythonPoetryDependency(dependent, dependee) {
        nx_project_1.NxProject.ensure(dependent).addPythonPoetryDependency(dependee);
    }
    /**
     * Ensures that all non-root projects have NxProject applied.
     * @internal
     */
    _ensureNxProjectGraph() {
        function _ensure(_project) {
            if (_project.root === _project)
                return;
            nx_project_1.NxProject.ensure(_project);
            _project.subprojects.forEach((p) => {
                _ensure(p);
            });
        }
        this.project.subprojects.forEach(_ensure);
    }
    /**
     * Emits package.json for non-node NX monorepos.
     * @internal
     */
    _emitPackageJson() {
        if (!utils_1.ProjectUtils.isNamedInstanceOf(this.project, javascript_1.NodeProject) &&
            !this.project.tryFindFile("package.json")) {
            new projen_1.JsonFile(this.project, "package.json", {
                obj: {
                    devDependencies: {
                        ...this.nxPlugins,
                        nx: "^16",
                        "@nx/devkit": "^16",
                    },
                    private: true,
                    workspaces: this.project.subprojects
                        .filter((p) => utils_1.ProjectUtils.isNamedInstanceOf(p, javascript_1.NodeProject))
                        .map((p) => path.relative(this.project.outdir, p.outdir)),
                },
            }).synthesize();
        }
    }
    preSynthesize() {
        // Calling before super() to ensure proper pre-synth of NxProject component and its nested components
        this._ensureNxProjectGraph();
        this._emitPackageJson();
        this.patchPythonProjects([this.project]);
    }
    /**
     * @inheritDoc
     */
    synth() {
        this.resetDefaultTask();
    }
    /**
     * Ensures subprojects don't have a default task
     */
    resetDefaultTask() {
        this.project.subprojects.forEach((subProject) => {
            // Disable default task on subprojects as this isn't supported in a monorepo
            subProject.defaultTask?.reset();
        });
    }
}
exports.NxConfigurator = NxConfigurator;
_a = JSII_RTTI_SYMBOL_1;
NxConfigurator[_a] = { fqn: "@aws/pdk.monorepo.NxConfigurator", version: "0.22.14" };
//# sourceMappingURL=data:application/json;base64,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