"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.inferBuildTarget = void 0;
const cdk_1 = require("projen/lib/cdk");
const java_1 = require("projen/lib/java");
const javascript_1 = require("projen/lib/javascript");
const python_1 = require("projen/lib/python");
const utils_1 = require("../../utils");
/**
 * Defines a fileset for target inputs and outputs.
 * @experimental
 * @internal
 */
class TargetFileset {
    static File(fileset, isWorkspaceRoot = false) {
        return new TargetFileset(fileset, false, isWorkspaceRoot);
    }
    static Directory(fileset, isWorkspaceRoot = false) {
        return new TargetFileset(fileset, true, isWorkspaceRoot);
    }
    static resolveInputs(values) {
        return values.map((v) => (typeof v === "string" ? v : v.filesetStarIfDir));
    }
    static resolveOutputs(values) {
        return values.map((v) => (typeof v === "string" ? v : v.fileset));
    }
    static outputsToInputs(values) {
        return values.map((v) => typeof v === "string" ? v : v.inverse().filesetStarIfDir);
    }
    constructor(fileset, isDirectory, isWorkspaceRoot) {
        this._fileset = fileset;
        this.isDirectory = isDirectory;
        this.isExclude = this._fileset.startsWith("!");
        this.isWorkspaceRoot = isWorkspaceRoot;
    }
    _ensureWildcards(fileset) {
        if (fileset.endsWith("*")) {
            return fileset;
        }
        if (fileset.endsWith("/")) {
            return fileset + "**/*";
        }
        return fileset + "/**/*";
    }
    get fileset() {
        let fileset = this.isExclude ? this._fileset.substring(1) : this._fileset;
        if (fileset.startsWith("/")) {
            fileset = fileset.substring(1);
        }
        if (this.isWorkspaceRoot) {
            return `${this.isExclude ? "!" : ""}{workspaceRoot}/${fileset}`;
        }
        return `${this.isExclude ? "!" : ""}{projectRoot}/${fileset}`;
    }
    inverse() {
        if (this.isExclude) {
            return new TargetFileset(this._fileset.substring(1), this.isDirectory, this.isWorkspaceRoot);
        }
        return new TargetFileset("!" + this._fileset, this.isDirectory, this.isWorkspaceRoot);
    }
    get filesetStarIfDir() {
        if (this.isDirectory) {
            return this._ensureWildcards(this.fileset);
        }
        return this.fileset;
    }
}
/**
 * Infer nx target values for build task
 * @experimental
 * @internal
 */
function inferBuildTarget(project, options) {
    const { inputs = [], outputs = [] } = _inferBuildTargetIO(project);
    if (options?.excludeOutputs !== false) {
        inputs.push(...TargetFileset.outputsToInputs(outputs));
    }
    let dependsOn = ["^build"];
    if (options?.dependsOn != null && options.dependsOn !== true) {
        if (options.dependsOn === false) {
            dependsOn = undefined;
        }
        else {
            dependsOn = options.dependsOn;
        }
    }
    if (outputs.length === 0) {
        return undefined;
    }
    return {
        inputs: TargetFileset.resolveInputs(inputs),
        outputs: TargetFileset.resolveOutputs(outputs),
        dependsOn,
    };
}
exports.inferBuildTarget = inferBuildTarget;
/** @internal */
function _inferBuildTargetIO(project) {
    const inputs = [];
    const outputs = [];
    let includeDefaultInputs = true;
    if (utils_1.ProjectUtils.isNamedInstanceOf(project, cdk_1.JsiiProject)) {
        outputs.push(TargetFileset.File(".jsii"), TargetFileset.Directory(project.libdir), TargetFileset.Directory(project.artifactsDirectory));
    }
    if (javascript_1.Jest.of(project)) {
        outputs.push(TargetFileset.Directory("coverage"), TargetFileset.Directory("test-reports"));
    }
    if (utils_1.ProjectUtils.isNamedInstanceOf(project, python_1.PythonProject)) {
        inputs.push(TargetFileset.Directory("!.env"), TargetFileset.Directory("!.pytest_cache"));
    }
    if (utils_1.ProjectUtils.isNamedInstanceOf(project, java_1.JavaProject)) {
        inputs.push(TargetFileset.File("!.classpath"), TargetFileset.File("!.project"), TargetFileset.File("!.settings"));
        outputs.push(TargetFileset.Directory("target"), TargetFileset.Directory("dist/java"));
    }
    if (includeDefaultInputs) {
        inputs.unshift("default", "^default");
    }
    return {
        inputs,
        outputs,
    };
}
//# sourceMappingURL=data:application/json;base64,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