"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.MonorepoTsProject = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/*! Copyright [Amazon.com](http://amazon.com/), Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0 */
const path = require("path");
const projen_1 = require("projen");
const javascript_1 = require("projen/lib/javascript");
const typescript_1 = require("projen/lib/typescript");
const syncpack_options_1 = require("./syncpack-options");
const nx_configurator_1 = require("../../components/nx-configurator");
const nx_project_1 = require("../../components/nx-project");
const utils_1 = require("../../utils");
/**
 * This project type will bootstrap a monorepo with support for polygot
 * builds, build caching, dependency graph visualization and much more.
 *
 * @pjid monorepo-ts
 */
class MonorepoTsProject extends typescript_1.TypeScriptProject {
    constructor(options) {
        const defaultReleaseBranch = options.defaultReleaseBranch ?? "main";
        super({
            ...options,
            github: options.github ?? false,
            package: options.package ?? false,
            projenCommand: options.packageManager
                ? utils_1.NodePackageUtils.command.projen(options.packageManager)
                : undefined,
            prettier: options.prettier ?? true,
            projenrcTs: true,
            release: options.release ?? false,
            jest: options.jest ?? false,
            sampleCode: false,
            gitignore: [".tmp", ...(options.gitignore ?? [])],
            defaultReleaseBranch,
            eslintOptions: options.eslintOptions ?? {
                dirs: ["."],
                ignorePatterns: ["packages/**/*.*"],
            },
            tsconfig: options.tsconfig ?? {
                compilerOptions: {
                    rootDir: ".",
                },
                include: ["**/*.ts", ".projenrc.ts"],
            },
        });
        this.subNodeProjectResolves = [];
        this.nxConfigurator = new nx_configurator_1.NxConfigurator(this, {
            defaultReleaseBranch,
        });
        this._options = options;
        // engines
        this.package.addEngine("node", ">=16");
        switch (this.package.packageManager) {
            case javascript_1.NodePackageManager.PNPM: {
                // https://pnpm.io/package_json
                // https://github.com/pnpm/pnpm/releases/tag/v8.0.0
                this.package.addEngine("pnpm", ">=8");
                break;
            }
            case javascript_1.NodePackageManager.YARN: {
                this.package.addEngine("yarn", ">=1 <2");
                break;
            }
            case javascript_1.NodePackageManager.YARN2: {
                this.package.addEngine("yarn", ">=2");
                this.gitignore.addPatterns(".yarn/*", ".pnp.cjs", "!.yarn/releases", "!.yarn/plugins");
                break;
            }
        }
        this.workspaceConfig = options.workspaceConfig;
        this.workspacePackages = options.workspaceConfig?.additionalPackages ?? [];
        // Never publish a monorepo root package.
        this.package.addField("private", true);
        // Add alias task for "projen" to synthesize workspace
        this.package.setScript("synth-workspace", utils_1.NodePackageUtils.command.projen(this.package.packageManager));
        // Map tasks to nx run-many
        if (options.scripts == null || options.scripts.build == null) {
            this.nxConfigurator._overrideNxBuildTask(this.buildTask, { target: "build" }, { force: true });
        }
        if (options.scripts == null || options.scripts["pre-compile"] == null) {
            this.nxConfigurator._overrideNxBuildTask(this.preCompileTask, {
                target: "pre-compile",
            });
        }
        if (options.scripts == null || options.scripts.compile == null) {
            this.nxConfigurator._overrideNxBuildTask(this.compileTask, {
                target: "compile",
            });
        }
        if (options.scripts == null || options.scripts["post-compile"] == null) {
            this.nxConfigurator._overrideNxBuildTask(this.postCompileTask, {
                target: "post-compile",
            });
        }
        if (options.scripts == null || options.scripts.test == null) {
            this.nxConfigurator._overrideNxBuildTask(this.testTask, {
                target: "test",
            });
        }
        if (options.scripts == null || options.scripts.eslint == null) {
            // The Projenrc component of TypeScriptProject resets the eslint task as part of preSynthesize which would undo
            // our changes, so we disable further resets.
            this.nxConfigurator._overrideNxBuildTask(this.eslint?.eslintTask, { target: "eslint" }, { disableReset: true });
        }
        if (options.scripts == null || options.scripts.package == null) {
            this.nxConfigurator._overrideNxBuildTask(this.packageTask, {
                target: "package",
            });
        }
        if (options.scripts == null || options.scripts.prepare == null) {
            this.nxConfigurator._overrideNxBuildTask("prepare", {
                target: "prepare",
            });
        }
        if (options.scripts == null || options.scripts.watch == null) {
            this.nxConfigurator._overrideNxBuildTask(this.watchTask, {
                target: "watch",
                noBail: false,
                ignoreCycles: true,
                skipCache: true,
                outputStyle: "stream",
            });
        }
        // Add dependency on nx 16
        this.addPeerDeps("nx@^16");
        this.addDevDeps("nx@^16", "@aws/pdk@^0");
        this.addDeps("aws-cdk-lib", "constructs", "cdk-nag"); // Needed as this can be bundled in @aws/pdk
        this.package.addPackageResolutions("@types/babel__traverse@7.18.2", "wrap-ansi@^7.0.0", "@zkochan/js-yaml@npm:js-yaml@4.1.0");
        if (options.monorepoUpgradeDeps !== false) {
            this.addDevDeps("npm-check-updates", "syncpack");
            const upgradeDepsTask = this.addTask(options.monorepoUpgradeDepsOptions?.taskName || "upgrade-deps");
            upgradeDepsTask.exec(utils_1.NodePackageUtils.command.exec(this.package.packageManager, "npm-check-updates", "--deep", "--rejectVersion", "0.0.0", "-u", "--dep", "prod,dev,peer,optional,bundle", "--target=minor"));
            upgradeDepsTask.exec(utils_1.NodePackageUtils.command.exec(this.package.packageManager, "syncpack", "fix-mismatches"));
            upgradeDepsTask.exec(utils_1.NodePackageUtils.command.install(this.package.packageManager));
            upgradeDepsTask.exec(utils_1.NodePackageUtils.command.exec(this.package.packageManager, "projen"));
            new projen_1.JsonFile(this, ".syncpackrc.json", {
                obj: options.monorepoUpgradeDepsOptions?.syncpackConfig || syncpack_options_1.DEFAULT_CONFIG,
                readonly: true,
            });
        }
    }
    /**
     * @inheritdoc
     */
    get nx() {
        return this.nxConfigurator.nx;
    }
    /**
     * @inheritdoc
     */
    execNxRunManyCommand(options) {
        return this.nxConfigurator.execNxRunManyCommand(options);
    }
    /**
     * @inheritdoc
     */
    composeNxRunManyCommand(options) {
        return this.nxConfigurator.composeNxRunManyCommand(options);
    }
    /**
     * @inheritdoc
     */
    addNxRunManyTask(name, options) {
        return this.nxConfigurator.addNxRunManyTask(name, options);
    }
    /**
     * @inheritdoc
     */
    addImplicitDependency(dependent, dependee) {
        this.nxConfigurator.addImplicitDependency(dependent, dependee);
    }
    /**
     * @inheritdoc
     */
    addJavaDependency(dependent, dependee) {
        this.nxConfigurator.addJavaDependency(dependent, dependee);
    }
    /**
     * @inheritdoc
     */
    addPythonPoetryDependency(dependent, dependee) {
        this.nxConfigurator.addPythonPoetryDependency(dependent, dependee);
    }
    /**
     * Add one or more additional package globs to the workspace.
     * @param packageGlobs paths to the package to include in the workspace (for example packages/my-package)
     */
    addWorkspacePackages(...packageGlobs) {
        // Any subprojects that were added since the last call to this method need to be added first, in order to ensure
        // we add the workspace packages in a sane order.
        const relativeSubProjectWorkspacePackages = this.sortedSubProjects.map((project) => path.relative(this.outdir, project.outdir));
        const existingWorkspacePackages = new Set(this.workspacePackages);
        this.workspacePackages.push(...relativeSubProjectWorkspacePackages.filter((pkg) => !existingWorkspacePackages.has(pkg)));
        // Add the additional packages next
        this.workspacePackages.push(...packageGlobs);
    }
    /** Get consistently sorted list of subprojects */
    get sortedSubProjects() {
        return this.subprojects
            .slice()
            .sort((a, b) => a.name.localeCompare(b.name));
    }
    /**
     * Create symbolic links to all local workspace bins. This enables the usage of bins the same
     * way as consumers of the packages have when installing from the registry.
     */
    linkLocalWorkspaceBins() {
        const bins = [];
        this.subprojects.forEach((subProject) => {
            if (utils_1.ProjectUtils.isNamedInstanceOf(subProject, javascript_1.NodeProject) &&
                subProject.name !== "@aws/pdk") {
                const pkgBins = subProject.package.manifest.bin() || {};
                bins.push(...Object.entries(pkgBins).map(([cmd, bin]) => {
                    const resolvedBin = path.join("$PWD", path.relative(this.outdir, subProject.outdir), bin);
                    return [cmd, resolvedBin];
                }));
            }
        });
        const linkTask = this.addTask("workspace:bin:link", {
            steps: bins.map(([cmd, bin]) => ({
                exec: `ln -s ${bin} ${utils_1.NodePackageUtils.command.bin(this.package.packageManager, cmd)} &>/dev/null; exit 0;`,
            })),
        });
        (this.tasks.tryFind("prepare") || this.addTask("prepare")).spawn(linkTask);
    }
    preSynthesize() {
        utils_1.NodePackageUtils.removeProjenScript(this);
        this.nxConfigurator.preSynthesize();
        super.preSynthesize();
        if (this._options.workspaceConfig?.linkLocalWorkspaceBins === true) {
            this.linkLocalWorkspaceBins();
        }
        if (this.package.packageManager === javascript_1.NodePackageManager.PNPM) {
            // PNPM hoisting hides transitive bundled dependencies which results in
            // transitive dependencies being packed incorrectly.
            this.subprojects.forEach((subProject) => {
                if (utils_1.NodePackageUtils.isNodeProject(subProject) &&
                    getBundledDeps(subProject).length) {
                    const pkgFolder = path.relative(this.root.outdir, subProject.outdir);
                    // Create a symlink in the sub-project node_modules for all transitive deps
                    // before running "package" task
                    subProject.packageTask.prependExec(`monorepo.pnpm-link-bundled-transitive-deps ${pkgFolder}`);
                }
            });
        }
        // Remove any subproject .npmrc files since only the root one matters
        this.subprojects.forEach((subProject) => {
            if (utils_1.NodePackageUtils.isNodeProject(subProject)) {
                subProject.tryRemoveFile(".npmrc");
                utils_1.NodePackageUtils.removeProjenScript(subProject);
            }
        });
    }
    /**
     * @inheritDoc
     */
    synth() {
        this.validateSubProjects();
        this.updateWorkspace();
        this.installNonNodeDependencies();
        // Disable node warnings if configured
        if (this._options.disableNodeWarnings) {
            this.disableNodeWarnings();
        }
        // Prevent sub NodeProject packages from `postSynthesis` which will cause individual/extraneous installs.
        // The workspace package install will handle all the sub NodeProject packages automatically.
        this.subprojects.forEach((subProject) => {
            if (utils_1.NodePackageUtils.isNodeProject(subProject)) {
                const subNodeProject = subProject;
                const subNodeProjectResolver = 
                // @ts-ignore - `resolveDepsAndWritePackageJson` is private
                subNodeProject.package.resolveDepsAndWritePackageJson;
                // @ts-ignore - `installDependencies` is private
                subNodeProject.package.installDependencies = () => {
                    this.subNodeProjectResolves.push(() => subNodeProjectResolver.apply(subNodeProject.package));
                };
                // @ts-ignore - `resolveDepsAndWritePackageJson` is private
                subNodeProject.package.resolveDepsAndWritePackageJson = () => { };
            }
        });
        this.nxConfigurator.synth();
        super.synth();
    }
    /**
     * @inheritDoc
     */
    postSynthesize() {
        super.postSynthesize();
        this.resolveSubNodeProjects();
    }
    /**
     * Resolve sub `NodePackage` dependencies.
     */
    resolveSubNodeProjects() {
        if (this.subNodeProjectResolves.length) {
            if (!this.package.file.changed) {
                // Force workspace install deps since it would not have been invoked during `postSynthesis`.
                // @ts-ignore - `installDependencies` is private
                this.package.installDependencies();
            }
            const completedResolves = this.subNodeProjectResolves.map((resolve) => resolve());
            if (completedResolves.some(Boolean)) {
                // Indicates that a subproject dependency has been resolved from '*', so update the lockfile.
                // @ts-ignore - `installDependencies` is private
                this.package.installDependencies();
            }
        }
        this.subNodeProjectResolves = [];
    }
    /**
     * Ensures subprojects don't have a default task and that all packages use the same package manager.
     */
    validateSubProjects() {
        this.subprojects.forEach((subProject) => {
            // Disable default task on subprojects as this isn't supported in a monorepo
            subProject.defaultTask?.reset();
            if (utils_1.NodePackageUtils.isNodeProject(subProject) &&
                subProject.package.packageManager !== this.package.packageManager) {
                throw new Error(`${subProject.name} packageManager does not match the monorepo packageManager: ${this.package.packageManager}.`);
            }
        });
    }
    /**
     * Add a submodule entry to the appropriate workspace file.
     */
    updateWorkspace() {
        // A final call to addWorkspacePackages will update the list of workspace packages with any subprojects that have
        // not yet been added, in the correct order
        this.addWorkspacePackages();
        let noHoist = this.workspaceConfig?.yarn?.noHoist;
        // Automatically add all sub-project "bundledDependencies" to workspace "hohoist", otherwise they are not bundled in npm package
        if (this.workspaceConfig?.yarn?.disableNoHoistBundled !== true &&
            (this.package.packageManager === javascript_1.NodePackageManager.YARN ||
                this.package.packageManager === javascript_1.NodePackageManager.YARN2)) {
            const noHoistBundled = this.subprojects.flatMap((sub) => {
                if (utils_1.ProjectUtils.isNamedInstanceOf(sub, javascript_1.NodeProject)) {
                    return getBundledDeps(sub).flatMap((dep) => [
                        `${sub.name}/${dep.name}`,
                        `${sub.name}/${dep.name}/*`,
                    ]);
                }
                return [];
            });
            if (noHoistBundled.length) {
                noHoist = [...(noHoist || []), ...noHoistBundled];
            }
        }
        // Add workspaces for each subproject
        if (this.package.packageManager === javascript_1.NodePackageManager.PNPM) {
            new projen_1.YamlFile(this, "pnpm-workspace.yaml", {
                readonly: true,
                obj: {
                    packages: this.workspacePackages,
                },
            });
        }
        this.package.addField("workspaces", {
            packages: this.workspacePackages,
            nohoist: noHoist,
        });
    }
    /**
     * Ensures the install task for non-node projects is executed postinstall.
     *
     * @private
     */
    installNonNodeDependencies() {
        const installProjects = this.subprojects.filter((project) => !utils_1.ProjectUtils.isNamedInstanceOf(project, javascript_1.NodeProject) &&
            project.tasks.tryFind("install"));
        if (installProjects.length > 0) {
            // TODO: Install error on clean repo for postinstall (https://cloud.nx.app/runs/MptQr0BxgF) (https://github.com/nrwl/nx/issues/11210)
            const postinstallTask = this.tasks.tryFind("postinstall") ?? this.addTask("postinstall");
            const nxRunManyInstall = this.composeNxRunManyCommand({
                target: "install",
                projects: installProjects.map((project) => project.name),
                parallel: 1,
            });
            postinstallTask.exec(utils_1.NodePackageUtils.command.exec(this.package.packageManager, ...nxRunManyInstall));
            // Ensure that install-py follows dependency order
            installProjects.forEach((p) => {
                nx_project_1.NxProject.ensure(p).setTarget("install", {
                    dependsOn: ["^install"],
                });
            });
        }
    }
    /**
     * Suppress Node warnings from being presented in the console when running builds.
     */
    disableNodeWarnings() {
        this.tasks.addEnvironment("NODE_NO_WARNINGS", "1");
        this.subprojects.forEach((subProject) => subProject.tasks.addEnvironment("NODE_NO_WARNINGS", "1"));
    }
}
exports.MonorepoTsProject = MonorepoTsProject;
_a = JSII_RTTI_SYMBOL_1;
MonorepoTsProject[_a] = { fqn: "@aws/pdk.monorepo.MonorepoTsProject", version: "0.22.14" };
/**
 * Gets bundled dependencies for a given project
 */
function getBundledDeps(project) {
    return project.deps.all.filter((dep) => dep.type === projen_1.DependencyType.BUNDLED);
}
//# sourceMappingURL=data:application/json;base64,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