/*! Copyright [Amazon.com](http://amazon.com/), Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0 */
import { Project } from "projen";
import { NodePackageManager, NodeProject, NodePackage } from "projen/lib/javascript";
/**
 * Utility functions for working with different Node package managers.
 * @experimental
 */
export declare namespace NodePackageUtils {
    /** Indicates if project is a node based project */
    function isNodeProject(project: Project): project is NodeProject;
    /**
     * Finds the NodePackageManager for the given proejct or returns a default type.
     *
     * @param project a project to retrieve the package manager for.
     * @returns NodePackageManager
     */
    function getNodePackageManager(project: Project, defaultPackageManager?: NodePackageManager): NodePackageManager;
    /**
     * Remove the "projen" script from package.json scripts, which causes recursive projen execution
     * for other scripts in format of "yarn projen [command]".
     * @param project NodeProject to remove "projen" script
     * @see https://github.com/projen/projen/blob/37983be94b37ee839ef3337a1b24b014a6c29f4f/src/javascript/node-project.ts#L512
     */
    function removeProjenScript(project: NodeProject): void;
    /**
     * Find the nearest {@link NodePackage} within scope. This will traverse parent
     * tree until finds projen with {@link NodePackage} component, or will throw
     * error if none found. Use {@link #tryFindNodePackage} if you do not want to
     * throw error.
     * @param scope The leaf project scope
     * @param {boolean} [recursive=false] Indicates if ancestral tree should be traversed
     * @returns {NodeProject} The NodeProject component for scope
     * @throws Error if {@link NodePackage} not found in tree of scope
     */
    function findNodePackage(scope: Project, recursive?: boolean): NodePackage;
    /**
     * Try to find the nearest {@link NodePackage} within scope. This will traverse parent
     * tree until finds projen with {@link NodePackage} component.
     * @param scope The leaf project scope
     * @param {boolean} [recursive=false] Indicates if ancestral tree should be traversed
     * @returns {NodeProject} The NodeProject component for scope, or undefined if no projects are node based.
     */
    function tryFindNodePackage(scope: Project, recursive?: boolean): NodePackage | undefined;
    /**
     * Command based utilities
     */
    namespace command {
        /**
         * Get command to run a script defined in the package.json
         */
        function runScript(packageManager: NodePackageManager, ...args: string[]): string;
        /**
         * Get command to execute projen or a projen task
         */
        function projen(packageManager: NodePackageManager, ...args: string[]): string;
        /**
         * Get command to execute a shell command
         */
        function exec(packageManager: NodePackageManager, ...args: string[]): string;
        /**
         * Get command to run a package in a temporary environment
         */
        function downloadExec(packageManager: NodePackageManager, ...args: string[]): string;
        /**
         * Get command to install a package
         */
        function install(packageManager: NodePackageManager, ...args: string[]): string;
        /**
         * Get command to run
         */
        function cmd(packageManager: NodePackageManager, ...args: string[]): string;
        function execInWorkspace(packageManager: NodePackageManager, packageName: string, ...args: string[]): string;
        /**
         * Get bash command for executing an executable in the package manager /bin dir.
         * Example: `$(yarn bin)/${cmd}`
         */
        function bin(packageManager: NodePackageManager, _cmd: string): string;
    }
}
