"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.NodePackageUtils = void 0;
const javascript_1 = require("projen/lib/javascript");
const project_1 = require("./project");
/**
 * Utility functions for working with different Node package managers.
 * @experimental
 */
var NodePackageUtils;
(function (NodePackageUtils) {
    /**
     * Append arguments to command string.
     * @internal
     */
    function withArgs(cmd, args) {
        if (args.length) {
            return `${cmd} ${args.join(" ")}`;
        }
        return cmd;
    }
    /** Indicates if project is a node based project */
    function isNodeProject(project) {
        return project_1.ProjectUtils.isNamedInstanceOf(project, javascript_1.NodeProject);
    }
    NodePackageUtils.isNodeProject = isNodeProject;
    /**
     * Finds the NodePackageManager for the given proejct or returns a default type.
     *
     * @param project a project to retrieve the package manager for.
     * @returns NodePackageManager
     */
    function getNodePackageManager(project, defaultPackageManager = javascript_1.NodePackageManager.YARN) {
        return isNodeProject(project)
            ? project.package.packageManager
            : defaultPackageManager;
    }
    NodePackageUtils.getNodePackageManager = getNodePackageManager;
    /**
     * Remove the "projen" script from package.json scripts, which causes recursive projen execution
     * for other scripts in format of "yarn projen [command]".
     * @param project NodeProject to remove "projen" script
     * @see https://github.com/projen/projen/blob/37983be94b37ee839ef3337a1b24b014a6c29f4f/src/javascript/node-project.ts#L512
     */
    function removeProjenScript(project) {
        project.package.removeScript("projen");
    }
    NodePackageUtils.removeProjenScript = removeProjenScript;
    /**
     * Find the nearest {@link NodePackage} within scope. This will traverse parent
     * tree until finds projen with {@link NodePackage} component, or will throw
     * error if none found. Use {@link #tryFindNodePackage} if you do not want to
     * throw error.
     * @param scope The leaf project scope
     * @param {boolean} [recursive=false] Indicates if ancestral tree should be traversed
     * @returns {NodeProject} The NodeProject component for scope
     * @throws Error if {@link NodePackage} not found in tree of scope
     */
    function findNodePackage(scope, recursive = false) {
        const nodePackage = tryFindNodePackage(scope, recursive);
        if (nodePackage) {
            return nodePackage;
        }
        throw new Error(`Project ${scope.name} does not have NodePackage component.`);
    }
    NodePackageUtils.findNodePackage = findNodePackage;
    /**
     * Try to find the nearest {@link NodePackage} within scope. This will traverse parent
     * tree until finds projen with {@link NodePackage} component.
     * @param scope The leaf project scope
     * @param {boolean} [recursive=false] Indicates if ancestral tree should be traversed
     * @returns {NodeProject} The NodeProject component for scope, or undefined if no projects are node based.
     */
    function tryFindNodePackage(scope, recursive = false) {
        let _project = scope;
        while (_project) {
            const nodePackage = _project.components.find((c) => project_1.ProjectUtils.isNamedInstanceOf(c, javascript_1.NodePackage));
            if (nodePackage) {
                return nodePackage;
            }
            if (!recursive) {
                return undefined;
            }
            _project = _project.parent;
        }
        return undefined;
    }
    NodePackageUtils.tryFindNodePackage = tryFindNodePackage;
    /**
     * Command based utilities
     */
    let command;
    (function (command) {
        /**
         * Get command to run a script defined in the package.json
         */
        function runScript(packageManager, ...args) {
            switch (packageManager) {
                case javascript_1.NodePackageManager.YARN:
                case javascript_1.NodePackageManager.YARN2:
                    return withArgs("yarn run", args);
                case javascript_1.NodePackageManager.PNPM:
                    return withArgs("pnpm run", args);
                default:
                    return withArgs("npm run", args);
            }
        }
        command.runScript = runScript;
        /**
         * Get command to execute projen or a projen task
         */
        function projen(packageManager, ...args) {
            return exec(packageManager, "projen", ...args);
        }
        command.projen = projen;
        /**
         * Get command to execute a shell command
         */
        function exec(packageManager, ...args) {
            switch (packageManager) {
                case javascript_1.NodePackageManager.YARN:
                    // "yarn exec" is not propagating transient args (`yarn exec nx run-many --target=build` does not receive `--target=build`)
                    return withArgs("yarn", args);
                case javascript_1.NodePackageManager.YARN2:
                    return withArgs("yarn exec", args);
                case javascript_1.NodePackageManager.PNPM:
                    return withArgs("pnpm exec", args);
                default:
                    return withArgs("npx", args);
            }
        }
        command.exec = exec;
        /**
         * Get command to run a package in a temporary environment
         */
        function downloadExec(packageManager, ...args) {
            switch (packageManager) {
                case javascript_1.NodePackageManager.YARN2:
                    return withArgs("yarn dlx", args);
                case javascript_1.NodePackageManager.PNPM:
                    return withArgs("pnpm dlx", args);
                default:
                    return withArgs("npx", args);
            }
        }
        command.downloadExec = downloadExec;
        /**
         * Get command to install a package
         */
        function install(packageManager, ...args) {
            switch (packageManager) {
                case javascript_1.NodePackageManager.YARN:
                case javascript_1.NodePackageManager.YARN2:
                    return withArgs("yarn install", args);
                case javascript_1.NodePackageManager.PNPM:
                    return withArgs("pnpm i", args);
                default:
                    return withArgs("npm install", args);
            }
        }
        command.install = install;
        /**
         * Get command to run
         */
        function cmd(packageManager, ...args) {
            switch (packageManager) {
                case javascript_1.NodePackageManager.YARN:
                case javascript_1.NodePackageManager.YARN2:
                    return withArgs("yarn", args);
                case javascript_1.NodePackageManager.PNPM:
                    return withArgs("pnpm", args);
                default:
                    return withArgs("npm", args);
            }
        }
        command.cmd = cmd;
        function execInWorkspace(packageManager, packageName, ...args) {
            switch (packageManager) {
                case javascript_1.NodePackageManager.YARN:
                case javascript_1.NodePackageManager.YARN2:
                    return withArgs("yarn workspace", [packageName, ...args]);
                case javascript_1.NodePackageManager.PNPM:
                    return withArgs("pnpm", [
                        `--filter "${packageName}"`,
                        "exec",
                        ...args,
                    ]);
                default:
                    return withArgs("npx", ["-p", packageName, ...args]);
            }
        }
        command.execInWorkspace = execInWorkspace;
        /**
         * Get bash command for executing an executable in the package manager /bin dir.
         * Example: `$(yarn bin)/${cmd}`
         */
        function bin(packageManager, _cmd) {
            switch (packageManager) {
                case javascript_1.NodePackageManager.YARN:
                case javascript_1.NodePackageManager.YARN2:
                    return `$(yarn bin)/${_cmd}`;
                case javascript_1.NodePackageManager.PNPM:
                    return `$(pnpm bin)/${_cmd}`;
                default:
                    return `$(npm bin)/${_cmd}`;
            }
        }
        command.bin = bin;
    })(command = NodePackageUtils.command || (NodePackageUtils.command = {}));
})(NodePackageUtils || (exports.NodePackageUtils = NodePackageUtils = {}));
//# sourceMappingURL=data:application/json;base64,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