"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.buildDependenciesHierarchy = void 0;
const path_1 = __importDefault(require("path"));
const lockfile_file_1 = require("@pnpm/lockfile-file");
const modules_yaml_1 = require("@pnpm/modules-yaml");
const normalize_registries_1 = require("@pnpm/normalize-registries");
const read_modules_dir_1 = require("@pnpm/read-modules-dir");
const read_package_json_1 = require("@pnpm/read-package-json");
const types_1 = require("@pnpm/types");
const normalize_path_1 = __importDefault(require("normalize-path"));
const realpath_missing_1 = __importDefault(require("realpath-missing"));
const resolve_link_target_1 = __importDefault(require("resolve-link-target"));
const getTree_1 = require("./getTree");
const getTreeNodeChildId_1 = require("./getTreeNodeChildId");
const getPkgInfo_1 = require("./getPkgInfo");
async function buildDependenciesHierarchy(projectPaths, maybeOpts) {
    if (!maybeOpts?.lockfileDir) {
        throw new TypeError('opts.lockfileDir is required');
    }
    const modulesDir = await (0, realpath_missing_1.default)(path_1.default.join(maybeOpts.lockfileDir, maybeOpts.modulesDir ?? 'node_modules'));
    const modules = await (0, modules_yaml_1.readModulesManifest)(modulesDir);
    const registries = (0, normalize_registries_1.normalizeRegistries)({
        ...maybeOpts?.registries,
        ...modules?.registries,
    });
    const currentLockfile = (modules?.virtualStoreDir && await (0, lockfile_file_1.readCurrentLockfile)(modules.virtualStoreDir, { ignoreIncompatible: false })) ?? null;
    const wantedLockfile = await (0, lockfile_file_1.readWantedLockfile)(maybeOpts.lockfileDir, { ignoreIncompatible: false });
    const result = {};
    if (!currentLockfile) {
        for (const projectPath of projectPaths) {
            result[projectPath] = {};
        }
        return result;
    }
    const opts = {
        depth: maybeOpts.depth || 0,
        include: maybeOpts.include ?? {
            dependencies: true,
            devDependencies: true,
            optionalDependencies: true,
        },
        lockfileDir: maybeOpts.lockfileDir,
        onlyProjects: maybeOpts.onlyProjects,
        registries,
        search: maybeOpts.search,
        skipped: new Set(modules?.skipped ?? []),
        modulesDir: maybeOpts.modulesDir,
        virtualStoreDir: modules?.virtualStoreDir,
    };
    (await Promise.all(projectPaths.map(async (projectPath) => {
        return [
            projectPath,
            await dependenciesHierarchyForPackage(projectPath, currentLockfile, wantedLockfile, opts),
        ];
    }))).forEach(([projectPath, dependenciesHierarchy]) => {
        result[projectPath] = dependenciesHierarchy;
    });
    return result;
}
exports.buildDependenciesHierarchy = buildDependenciesHierarchy;
async function dependenciesHierarchyForPackage(projectPath, currentLockfile, wantedLockfile, opts) {
    const importerId = (0, lockfile_file_1.getLockfileImporterId)(opts.lockfileDir, projectPath);
    if (!currentLockfile.importers[importerId])
        return {};
    const modulesDir = path_1.default.join(projectPath, opts.modulesDir ?? 'node_modules');
    const savedDeps = getAllDirectDependencies(currentLockfile.importers[importerId]);
    const allDirectDeps = await (0, read_modules_dir_1.readModulesDir)(modulesDir) ?? [];
    const unsavedDeps = allDirectDeps.filter((directDep) => !savedDeps[directDep]);
    const getChildrenTree = getTree_1.getTree.bind(null, {
        currentPackages: currentLockfile.packages ?? {},
        importers: currentLockfile.importers,
        includeOptionalDependencies: opts.include.optionalDependencies,
        lockfileDir: opts.lockfileDir,
        onlyProjects: opts.onlyProjects,
        rewriteLinkVersionDir: projectPath,
        maxDepth: opts.depth,
        modulesDir,
        registries: opts.registries,
        search: opts.search,
        skipped: opts.skipped,
        wantedPackages: wantedLockfile?.packages ?? {},
        virtualStoreDir: opts.virtualStoreDir,
    });
    const parentId = { type: 'importer', importerId };
    const result = {};
    for (const dependenciesField of types_1.DEPENDENCIES_FIELDS.sort().filter(dependenciedField => opts.include[dependenciedField])) {
        const topDeps = currentLockfile.importers[importerId][dependenciesField] ?? {};
        result[dependenciesField] = [];
        Object.entries(topDeps).forEach(([alias, ref]) => {
            const packageInfo = (0, getPkgInfo_1.getPkgInfo)({
                alias,
                currentPackages: currentLockfile.packages ?? {},
                rewriteLinkVersionDir: projectPath,
                linkedPathBaseDir: projectPath,
                ref,
                registries: opts.registries,
                skipped: opts.skipped,
                wantedPackages: wantedLockfile?.packages ?? {},
                virtualStoreDir: opts.virtualStoreDir,
            });
            let newEntry = null;
            const matchedSearched = opts.search?.(packageInfo);
            const nodeId = (0, getTreeNodeChildId_1.getTreeNodeChildId)({
                parentId,
                dep: { alias, ref },
                lockfileDir: opts.lockfileDir,
                importers: currentLockfile.importers,
            });
            if (opts.onlyProjects && nodeId?.type !== 'importer') {
                return;
            }
            else if (nodeId == null) {
                if ((opts.search != null) && !matchedSearched)
                    return;
                newEntry = packageInfo;
            }
            else {
                const dependencies = getChildrenTree(nodeId);
                if (dependencies.length > 0) {
                    newEntry = {
                        ...packageInfo,
                        dependencies,
                    };
                }
                else if ((opts.search == null) || matchedSearched) {
                    newEntry = packageInfo;
                }
            }
            if (newEntry != null) {
                if (matchedSearched) {
                    newEntry.searched = true;
                }
                result[dependenciesField].push(newEntry);
            }
        });
    }
    await Promise.all(unsavedDeps.map(async (unsavedDep) => {
        let pkgPath = path_1.default.join(modulesDir, unsavedDep);
        let version;
        try {
            pkgPath = await (0, resolve_link_target_1.default)(pkgPath);
            version = `link:${(0, normalize_path_1.default)(path_1.default.relative(projectPath, pkgPath))}`;
        }
        catch (err) { // eslint-disable-line
            // if error happened. The package is not a link
            const pkg = await (0, read_package_json_1.safeReadPackageJsonFromDir)(pkgPath);
            version = pkg?.version ?? 'undefined';
        }
        const pkg = {
            alias: unsavedDep,
            isMissing: false,
            isPeer: false,
            isSkipped: false,
            name: unsavedDep,
            path: pkgPath,
            version,
        };
        const matchedSearched = opts.search?.(pkg);
        if ((opts.search != null) && !matchedSearched)
            return;
        const newEntry = pkg;
        if (matchedSearched) {
            newEntry.searched = true;
        }
        result.unsavedDependencies = result.unsavedDependencies ?? [];
        result.unsavedDependencies.push(newEntry);
    }));
    return result;
}
function getAllDirectDependencies(projectSnapshot) {
    return {
        ...projectSnapshot.dependencies,
        ...projectSnapshot.devDependencies,
        ...projectSnapshot.optionalDependencies,
    };
}
//# sourceMappingURL=buildDependenciesHierarchy.js.map