"use strict";
/*********************************************************************************************************************
 Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.

 Licensed under the Apache License, Version 2.0 (the "License").
 You may not use this file except in compliance with the License.
 You may obtain a copy of the License at

 http://www.apache.org/licenses/LICENSE-2.0

 Unless required by applicable law or agreed to in writing, software
 distributed under the License is distributed on an "AS IS" BASIS,
 WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 See the License for the specific language governing permissions and
 limitations under the License.
 ******************************************************************************************************************** */
Object.defineProperty(exports, "__esModule", { value: true });
const aws_sdk_1 = require("aws-sdk"); // eslint-disable-line
const DELIMITER = ":";
const SCOPE = "CLOUDFRONT";
const client = new aws_sdk_1.WAFV2({
    region: "us-east-1",
});
/**
 * Handler for creating a WAF V2 ACL in US-EAST-1.
 */
exports.onEvent = async (event) => {
    const { ID, MANAGED_RULES, CIDR_ALLOW_LIST } = event.ResourceProperties;
    const [WEB_ACL_ID, IP_SET_ID] = event.PhysicalResourceId
        ? event.PhysicalResourceId.split(DELIMITER)
        : [];
    let response = {};
    switch (event.RequestType) {
        case "Create":
            response = await createWaf(ID, MANAGED_RULES, CIDR_ALLOW_LIST);
            break;
        case "Update":
            response = await updateWaf(WEB_ACL_ID, IP_SET_ID, ID, getIpSetName(ID), MANAGED_RULES, CIDR_ALLOW_LIST);
            break;
        case "Delete":
            response = await deleteWaf(WEB_ACL_ID, IP_SET_ID, ID, getIpSetName(ID));
            break;
        default:
            throw new Error(`Invalid RequestType: ${event.RequestType}`);
    }
    return response;
};
/**
 * Generates the name of the IP Set.
 *
 * @param id param passed in.
 * @returns name of IP Set.
 */
const getIpSetName = (id) => `${id}-IPSet`;
/**
 * Returns a set of rules to apply.
 *
 * @param ipSetArn ip set arn
 * @param ipSetName  ip set name
 * @param managedRules  managed rules
 * @param cidrAllowList cidr allow list
 * @returns set of rules to apply.
 */
const getWafRules = (ipSetArn, ipSetName, managedRules, cidrAllowList) => {
    const rules = [];
    if (cidrAllowList) {
        rules.push({
            Name: ipSetName,
            Priority: 1,
            VisibilityConfig: {
                MetricName: ipSetName,
                CloudWatchMetricsEnabled: true,
                SampledRequestsEnabled: true,
            },
            Action: {
                Block: {},
            },
            Statement: {
                NotStatement: {
                    Statement: {
                        IPSetReferenceStatement: {
                            ARN: ipSetArn,
                        },
                    },
                },
            },
        });
    }
    if (managedRules) {
        rules.push(...managedRules
            .map((r) => ({ VendorName: r.vendor, Name: r.name }))
            .map((rule, Priority) => ({
            Name: `${rule.VendorName}-${rule.Name}`,
            Priority,
            Statement: { ManagedRuleGroupStatement: rule },
            OverrideAction: { None: {} },
            VisibilityConfig: {
                MetricName: `${rule.VendorName}-${rule.Name}`,
                CloudWatchMetricsEnabled: true,
                SampledRequestsEnabled: true,
            },
        })));
    }
    return rules;
};
const createWaf = async (id, managedRules, cidrAllowList) => {
    const ipSetName = getIpSetName(id);
    const createIpSetResponse = await client
        .createIPSet({
        Name: ipSetName,
        Scope: SCOPE,
        Addresses: cidrAllowList?.cidrRanges ?? [],
        IPAddressVersion: cidrAllowList?.cidrType ?? "IPV4",
    })
        .promise();
    const createWebAclResponse = await client
        .createWebACL({
        Name: id,
        DefaultAction: { Allow: {} },
        Scope: SCOPE,
        VisibilityConfig: {
            CloudWatchMetricsEnabled: true,
            MetricName: id,
            SampledRequestsEnabled: true,
        },
        Rules: getWafRules(createIpSetResponse.Summary.ARN, ipSetName, managedRules, cidrAllowList),
    })
        .promise();
    return {
        PhysicalResourceId: `${createWebAclResponse.Summary?.Id}${DELIMITER}${createIpSetResponse.Summary?.Id}`,
        Data: {
            WebAclArn: createWebAclResponse.Summary?.ARN,
            WebAclId: createWebAclResponse.Summary?.Id,
            IPSetArn: createIpSetResponse.Summary?.ARN,
            IPSetId: createIpSetResponse.Summary?.Id,
        },
    };
};
const updateWaf = async (webAclId, ipSetId, id, ipSetName, managedRules, cidrAllowList) => {
    const getIpSetResponse = await client
        .getIPSet({
        Id: ipSetId,
        Name: ipSetName,
        Scope: SCOPE,
    })
        .promise();
    await client
        .updateIPSet({
        Id: ipSetId,
        Name: ipSetName,
        Addresses: cidrAllowList?.cidrRanges ?? [],
        Scope: SCOPE,
        LockToken: getIpSetResponse.LockToken,
    })
        .promise();
    const getWebAclResponse = await client
        .getWebACL({
        Id: webAclId,
        Name: id,
        Scope: SCOPE,
    })
        .promise();
    await client
        .updateWebACL({
        Name: id,
        DefaultAction: { Allow: {} },
        Scope: SCOPE,
        VisibilityConfig: {
            CloudWatchMetricsEnabled: true,
            MetricName: id,
            SampledRequestsEnabled: true,
        },
        Rules: getWafRules(getIpSetResponse.IPSet?.ARN, ipSetName, managedRules, cidrAllowList),
        Id: getWebAclResponse.WebACL?.Id,
        LockToken: getWebAclResponse.LockToken,
    })
        .promise();
    return {
        Data: {
            WebAclArn: getWebAclResponse.WebACL?.ARN,
            WebAclId: getWebAclResponse.WebACL?.Id,
            IPSetArn: getIpSetResponse.IPSet?.ARN,
            IPSetId: getIpSetResponse.IPSet?.Id,
        },
    };
};
const deleteWaf = async (webAclId, ipSetId, id, ipSetName) => {
    const getWebAclResponse = await client
        .getWebACL({
        Id: webAclId,
        Name: id,
        Scope: SCOPE,
    })
        .promise();
    await client
        .deleteWebACL({
        Id: webAclId,
        Name: id,
        Scope: SCOPE,
        LockToken: getWebAclResponse.LockToken,
    })
        .promise();
    const getIpSetResponse = await client
        .getIPSet({
        Id: ipSetId,
        Name: ipSetName,
        Scope: SCOPE,
    })
        .promise();
    await client
        .deleteIPSet({
        Id: ipSetId,
        Name: ipSetName,
        Scope: SCOPE,
        LockToken: getIpSetResponse.LockToken,
    })
        .promise();
    return {
        Data: {
            WebAclArn: getWebAclResponse.WebACL?.ARN,
            WebAclId: getWebAclResponse.WebACL?.Id,
            IPSetArn: getIpSetResponse.IPSet?.ARN,
            IPSetId: getIpSetResponse.IPSet?.Id,
        },
    };
};
//# sourceMappingURL=data:application/json;base64,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