"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.LambdaIntegration = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/*! Copyright [Amazon.com](http://amazon.com/), Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0 */
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_lambda_1 = require("aws-cdk-lib/aws-lambda");
const integration_1 = require("./integration");
const snap_start_java_function_1 = require("../functions/snap-start-java-function");
const utils_1 = require("../spec/utils");
/**
 * A lambda integration
 */
class LambdaIntegration extends integration_1.Integration {
    constructor(lambdaFunction) {
        super();
        // Snap Start applies only to versions, so if the function is a SnapStartFunction, we'll reference the current version
        if (lambdaFunction instanceof snap_start_java_function_1.SnapStartFunction) {
            this.lambdaFunction = lambdaFunction.currentVersion;
        }
        else {
            this.lambdaFunction = lambdaFunction;
        }
    }
    /**
     * Render the lambda integration as a snippet of OpenAPI
     */
    render(_props) {
        return {
            type: "AWS_PROXY",
            httpMethod: "POST",
            uri: (0, utils_1.functionInvocationUri)(this.lambdaFunction),
            passthroughBehavior: "WHEN_NO_MATCH",
        };
    }
    getOperationPermissionId(operationId) {
        return `LambdaPermission-${operationId}`;
    }
    /**
     * Grant API Gateway permissions to invoke the lambda
     */
    grant({ scope, api, operationId, method, path, operationLookup, }) {
        // Router permissions are unique to a function
        const routerPermissionId = `LambdaRouterPermission-${this.lambdaFunction.node.addr.slice(-8)}`;
        // Check if we've already granted a router permission for this lambda
        if (scope.node.tryFindChild(routerPermissionId)) {
            return; // The function already has access to all operations
        }
        // Check if a permission has been added for other operations for the same function arn
        const otherOperationPermissions = Object.keys(operationLookup)
            .map((opId) => scope.node.tryFindChild(this.getOperationPermissionId(opId)))
            .filter((permission) => permission &&
            permission instanceof aws_lambda_1.CfnPermission &&
            permission.functionName === this.lambdaFunction.functionArn);
        if (otherOperationPermissions.length > 0) {
            // This lambda function is reused, so we add the "router permission" which allows
            // invocation for any operation, to save exceeding the policy size limit for large
            // numbers of operations.
            otherOperationPermissions.forEach((permission) => scope.node.tryRemoveChild(permission.node.id));
            new aws_lambda_1.CfnPermission(scope, routerPermissionId, {
                action: "lambda:InvokeFunction",
                principal: "apigateway.amazonaws.com",
                functionName: this.lambdaFunction.functionArn,
                sourceArn: aws_cdk_lib_1.Stack.of(scope).formatArn({
                    service: "execute-api",
                    resource: api.restApiId,
                    // Permissions for all
                    resourceName: "*/*/*",
                }),
            });
        }
        else {
            // Add an individual operation permission since this lambda is not reused for multiple operations
            new aws_lambda_1.CfnPermission(scope, this.getOperationPermissionId(operationId), {
                action: "lambda:InvokeFunction",
                principal: "apigateway.amazonaws.com",
                functionName: this.lambdaFunction.functionArn,
                sourceArn: aws_cdk_lib_1.Stack.of(scope).formatArn({
                    service: "execute-api",
                    resource: api.restApiId,
                    // Scope permissions to any stage and a specific method and path of the operation.
                    // Path parameters (eg {param} are replaced with wildcards)
                    resourceName: `*/${method.toUpperCase()}${path.replace(/{[^\}]*\}/g, "*")}`,
                }),
            });
        }
    }
}
exports.LambdaIntegration = LambdaIntegration;
_a = JSII_RTTI_SYMBOL_1;
LambdaIntegration[_a] = { fqn: "@aws/pdk.type_safe_api.LambdaIntegration", version: "0.22.14" };
//# sourceMappingURL=data:application/json;base64,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