"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.prepareApiSpec = exports.generateCorsResponseParameters = exports.concatMethodAndPath = void 0;
const constants_1 = require("./constants");
/**
 * Serialise a method and path into a single string
 */
const concatMethodAndPath = ({ method, path }) => `${method.toLowerCase()}||${path.toLowerCase()}`;
exports.concatMethodAndPath = concatMethodAndPath;
/**
 * Return an array of security scheme references including the api key one if required
 */
const apiKeySecurityReference = (options) => options?.apiKeyRequired ? [{ [constants_1.DefaultAuthorizerIds.API_KEY]: [] }] : [];
/**
 * Generate a "no auth" spec snippet
 */
const noAuthSpecSnippet = (options) => ({
    security: apiKeySecurityReference(options),
    "x-amazon-apigateway-auth": {
        type: "NONE",
    },
});
/**
 * Create the OpenAPI definition with api gateway extensions for the given authorizer
 * @param methodAuthorizer the authorizer used for the method
 * @param options api integration options
 */
const applyMethodAuthorizer = (methodAuthorizer, options) => {
    if (methodAuthorizer || options) {
        if (methodAuthorizer?.authorizerId === constants_1.DefaultAuthorizerIds.NONE) {
            return noAuthSpecSnippet(options);
        }
        else {
            return {
                security: [
                    ...(methodAuthorizer
                        ? [
                            {
                                [methodAuthorizer.authorizerId]: methodAuthorizer.authorizationScopes || [],
                            },
                        ]
                        : []),
                    ...apiKeySecurityReference(options),
                ],
            };
        }
    }
    return {};
};
/**
 * Adds API Gateway integrations and auth to the given operation
 */
const applyMethodIntegration = (path, method, { integrations, corsOptions, apiKeyOptions, defaultAuthorizerReference, }, operation, getOperationName) => {
    const operationName = getOperationName({ method, path });
    if (!(operationName in integrations)) {
        throw new Error(`Missing required integration for operation ${operationName} (${method} ${path})`);
    }
    let { methodAuthorizer, integration, options } = integrations[operationName];
    validateAuthorizerReference(methodAuthorizer, operation.security, operationName);
    let methodApiKeyOptions = options;
    // When no API key options are present on the method, require the API key if it's
    // required by default
    if (!methodApiKeyOptions && apiKeyOptions?.requiredByDefault) {
        methodApiKeyOptions = { apiKeyRequired: true };
    }
    // Can only "require" an API key if it's in a header, since we only define the security
    // scheme we'd reference in this case.
    if (apiKeyOptions?.source !== "HEADER" &&
        methodApiKeyOptions?.apiKeyRequired) {
        throw new Error(`Cannot require an API Key when API Key source is not HEADER: ${operationName} (${method} ${path})`);
    }
    // Apply the default authorizer unless a method authorizer is defined
    methodAuthorizer = methodAuthorizer ?? defaultAuthorizerReference;
    return {
        ...operation,
        responses: Object.fromEntries(Object.entries(operation.responses).map(([statusCode, response]) => [
            statusCode,
            {
                ...response,
                headers: {
                    ...(corsOptions ? getCorsHeaderDefinitions() : {}),
                    // TODO: Consider following response header references
                    ...response.headers,
                },
            },
        ])),
        // https://docs.aws.amazon.com/apigateway/latest/developerguide/api-gateway-swagger-extensions-integration.html
        "x-amazon-apigateway-integration": integration,
        ...applyMethodAuthorizer(methodAuthorizer, methodApiKeyOptions),
    };
};
const getCorsHeaderDefinitions = () => ({
    "Access-Control-Allow-Origin": {
        schema: { type: "string" },
    },
    "Access-Control-Allow-Methods": {
        schema: { type: "string" },
    },
    "Access-Control-Allow-Headers": {
        schema: { type: "string" },
    },
});
const generateCorsResponseHeaders = (corsOptions) => ({
    "Access-Control-Allow-Headers": `'${corsOptions.allowHeaders.join(",")}'`,
    "Access-Control-Allow-Methods": `'${corsOptions.allowMethods.join(",")}'`,
    "Access-Control-Allow-Origin": `'${corsOptions.allowOrigins.join(",")}'`,
});
const generateCorsResponseParameters = (corsOptions, prefix = "method.response.header") => Object.fromEntries(Object.entries(generateCorsResponseHeaders(corsOptions)).map(([header, value]) => [`${prefix}.${header}`, value]));
exports.generateCorsResponseParameters = generateCorsResponseParameters;
/**
 * Generates an "options" method with no auth to respond with the appropriate headers if cors is enabled
 */
const generateCorsOptionsMethod = (pathItem, { corsOptions }) => {
    // Do not generate if already manually defined, or cors not enabled
    if (constants_1.HttpMethods.OPTIONS in pathItem || !corsOptions) {
        return {};
    }
    const statusCode = corsOptions.statusCode;
    return {
        [constants_1.HttpMethods.OPTIONS]: {
            summary: "CORS Support",
            description: "Enable CORS by returning the correct headers",
            responses: {
                [`${statusCode}`]: {
                    description: "Default response for CORS method",
                    headers: getCorsHeaderDefinitions(),
                    content: {},
                },
            },
            // @ts-ignore Ignore apigateway extensions which are not part of default openapi spec type
            "x-amazon-apigateway-integration": {
                type: "mock",
                requestTemplates: {
                    "application/json": `{"statusCode": ${statusCode}}`,
                },
                responses: {
                    default: {
                        statusCode: `${statusCode}`,
                        responseParameters: (0, exports.generateCorsResponseParameters)(corsOptions),
                        responseTemplates: {
                            "application/json": "{}",
                        },
                    },
                },
            },
            // No auth for CORS options requests
            ...noAuthSpecSnippet(),
        },
    };
};
/**
 * Prepares a given api path by adding integrations, configuring auth
 */
const preparePathSpec = (path, pathItem, options, getOperationName) => {
    const supportedHttpMethods = new Set(Object.values(constants_1.HttpMethods));
    const unsupportedMethodsInSpec = Object.keys(pathItem).filter((method) => !supportedHttpMethods.has(method));
    if (unsupportedMethodsInSpec.length > 0) {
        throw new Error(`Path ${path} contains unsupported method${unsupportedMethodsInSpec.length > 1 ? "s" : ""} ${unsupportedMethodsInSpec.join(", ")}. Supported methods are ${Object.values(constants_1.HttpMethods).join(", ")}.`);
    }
    return {
        ...pathItem,
        ...Object.fromEntries(Object.values(constants_1.HttpMethods)
            .filter((method) => pathItem[method])
            .map((method) => [
            method,
            applyMethodIntegration(path, method, options, pathItem[method], getOperationName),
        ])),
        // Generate an 'options' method required for CORS preflight requests if cors is enabled
        ...generateCorsOptionsMethod(pathItem, options),
    };
};
/**
 * Return whether the given OpenAPI object is a reference object
 */
const isRef = (obj) => "$ref" in obj;
/**
 * Validate the construct security schemes against the security schemes in the original spec.
 * Construct-defined authorizers always override those in the spec if they have the same ID, however we validate that
 * we are not overriding an authorizer of a different type to avoid mistakes/mismatches between the spec and the
 * construct.
 * @param constructSecuritySchemes security schemes generated from the construct authorizers
 * @param existingSpecSecuritySchemes security schemes already defined in the spec
 */
const validateSecuritySchemes = (constructSecuritySchemes, existingSpecSecuritySchemes) => {
    if (existingSpecSecuritySchemes) {
        const constructSecuritySchemeIds = new Set(Object.keys(constructSecuritySchemes));
        const existingSecuritySchemeIds = new Set(Object.keys(existingSpecSecuritySchemes));
        const overlappingSecuritySchemeIds = [...constructSecuritySchemeIds].filter((id) => existingSecuritySchemeIds.has(id));
        // Any overlapping security schemes (defined in both the spec (or source smithy model) and the construct) must be of the same type.
        // The one defined in the construct will take precedence since a custom/cognito authorizer can have a resolved arn in the construct,
        // and we allow usage in the model as a forward definition with blank arn.
        overlappingSecuritySchemeIds.forEach((schemeId) => {
            if (!isRef(existingSpecSecuritySchemes[schemeId])) {
                const existingScheme = existingSpecSecuritySchemes[schemeId];
                if (constructSecuritySchemes[schemeId].type !== existingScheme.type) {
                    throw new Error(`Security scheme with id ${schemeId} was of type ${constructSecuritySchemes[schemeId].type} in construct but ${existingScheme.type} in OpenAPI spec or Smithy model.`);
                }
                const constructApiGatewayAuthType = constructSecuritySchemes[schemeId]["x-amazon-apigateway-authtype"];
                const existingApiGatewayAuthType = existingScheme["x-amazon-apigateway-authtype"];
                if (constructApiGatewayAuthType !== existingApiGatewayAuthType) {
                    throw new Error(`Security scheme with id ${schemeId} was of type ${constructApiGatewayAuthType} in construct but ${existingApiGatewayAuthType} in OpenAPI spec or Smithy model.`);
                }
            }
            else {
                throw new Error(`Security scheme with id ${schemeId} is a reference in the OpenAPI spec or Smithy model which is not supported.`);
            }
        });
    }
};
/**
 * Validate the given authorizer reference (either default or at an operation level) defined in the construct against
 * those already in the spec.
 * @param constructAuthorizer the authorizer defined in the construct
 * @param existingSpecAuthorizers the authorizers already defined in the spec
 * @param operation the operation we are validating (for clearer error messages)
 */
const validateAuthorizerReference = (constructAuthorizer, existingSpecAuthorizers, operation = "Default") => {
    // Only need to validate if defined in both - if just one we'll use that.
    if (constructAuthorizer && existingSpecAuthorizers) {
        const mergedSpecAuthorizers = Object.fromEntries(existingSpecAuthorizers.flatMap((securityRequirement) => Object.keys(securityRequirement).map((id) => [
            id,
            securityRequirement[id],
        ])));
        const specAuthorizerIds = Object.keys(mergedSpecAuthorizers);
        if (specAuthorizerIds.length > 1) {
            // Spec defined multiple authorizers but the construct can only specify one
            throw new Error(`${operation} authorizers ${specAuthorizerIds
                .sort()
                .join(", ")} defined in the OpenAPI Spec or Smithy Model would be overridden by single construct authorizer ${constructAuthorizer.authorizerId}`);
        }
        else if (specAuthorizerIds.length === 1) {
            // Single authorizer - check that they have the same id
            if (specAuthorizerIds[0] !== constructAuthorizer.authorizerId) {
                throw new Error(`${operation} authorizer ${specAuthorizerIds[0]} defined in the OpenAPI Spec or Smithy Model would be overridden by construct authorizer ${constructAuthorizer.authorizerId}`);
            }
            // Check that there are no differing scopes between the construct and spec
            const specScopes = new Set(mergedSpecAuthorizers[specAuthorizerIds[0]]);
            const constructScopes = new Set(constructAuthorizer.authorizationScopes);
            const differingScopes = [
                ...[...specScopes].filter((scope) => !constructScopes.has(scope)),
                ...[...constructScopes].filter((scope) => !specScopes.has(scope)),
            ];
            if (differingScopes.length > 0) {
                throw new Error(`${operation} authorizer scopes ${[...specScopes].join(", ")} defined in the OpenAPI Spec or Smithy Model differ from those in the construct (${[
                    ...constructScopes,
                ].join(", ")})`);
            }
        }
        else if (constructAuthorizer.authorizerId !== constants_1.DefaultAuthorizerIds.NONE) {
            // "security" section of spec is [] which means no auth, but the authorizer in the construct is not the "none" authorizer.
            throw new Error(`${operation} explicitly defines no auth in the OpenAPI Spec or Smithy Model which would be overridden by construct authorizer ${constructAuthorizer.authorizerId}`);
        }
    }
};
/**
 * Find all unique header parameters used in operations
 */
const findHeaderParameters = (spec) => {
    const allHeaderParameters = Object.values(spec.paths).flatMap((pathDetails) => Object.values(constants_1.HttpMethods).flatMap((method) => (pathDetails?.[method]?.parameters ?? []).flatMap((parameter) => "in" in parameter && parameter.in === "header" ? [parameter.name] : [])));
    const headerParameterSet = new Set();
    return allHeaderParameters.filter((p) => {
        const seen = headerParameterSet.has(p);
        headerParameterSet.add(p);
        return !seen;
    });
};
/**
 * Prepares the api spec for deployment by adding integrations, configuring auth, etc
 */
const prepareApiSpec = (spec, options) => {
    // Reverse lookup for the operation name given a method and path
    const operationNameByPath = Object.fromEntries(Object.entries(options.operationLookup).map(([operationName, methodAndPath]) => [
        (0, exports.concatMethodAndPath)(methodAndPath),
        operationName,
    ]));
    const getOperationName = (methodAndPath) => operationNameByPath[(0, exports.concatMethodAndPath)(methodAndPath)];
    validateSecuritySchemes(options.securitySchemes, spec.components?.securitySchemes);
    validateAuthorizerReference(options.defaultAuthorizerReference, spec.security);
    // If there are cors options, add any header parameters defined in the spec as allowed headers to
    // save users from having to manually specify these (or face cors issues!)
    const corsOptions = options.corsOptions
        ? {
            ...options.corsOptions,
            allowHeaders: [
                ...options.corsOptions.allowHeaders,
                ...findHeaderParameters(spec),
            ],
        }
        : undefined;
    const updatedOptions = {
        ...options,
        corsOptions,
    };
    return {
        ...spec,
        // https://docs.aws.amazon.com/apigateway/latest/developerguide/api-gateway-swagger-extensions-request-validators.html
        "x-amazon-apigateway-request-validators": {
            all: {
                validateRequestBody: true,
                validateRequestParameters: true,
            },
        },
        "x-amazon-apigateway-request-validator": "all",
        // https://docs.aws.amazon.com/apigateway/latest/developerguide/api-gateway-swagger-extensions-gateway-responses.html
        "x-amazon-apigateway-gateway-responses": {
            BAD_REQUEST_BODY: {
                statusCode: 400,
                responseTemplates: {
                    "application/json": '{"message": "$context.error.validationErrorString"}',
                },
                ...(corsOptions
                    ? {
                        responseParameters: (0, exports.generateCorsResponseParameters)(corsOptions, "gatewayresponse.header"),
                    }
                    : {}),
            },
        },
        paths: {
            ...Object.fromEntries(Object.entries(spec.paths).map(([path, pathDetails]) => [
                path,
                preparePathSpec(path, pathDetails, updatedOptions, getOperationName),
            ])),
        },
        components: {
            ...spec.components,
            securitySchemes: {
                // Apply any security schemes that already exist in the spec
                ...spec.components?.securitySchemes,
                // Construct security schemes override any in the spec with the same id
                ...updatedOptions.securitySchemes,
            },
        },
        ...(updatedOptions.apiKeyOptions
            ? {
                // https://docs.aws.amazon.com/apigateway/latest/developerguide/api-gateway-swagger-extensions-api-key-source.html
                "x-amazon-apigateway-api-key-source": updatedOptions.apiKeyOptions.source,
            }
            : {}),
    };
};
exports.prepareApiSpec = prepareApiSpec;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoicHJlcGFyZS1zcGVjLmpzIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsicHJlcGFyZS1zcGVjLnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7OztBQUdBLDJDQUFnRTtBQVloRTs7R0FFRztBQUNJLE1BQU0sbUJBQW1CLEdBQUcsQ0FBQyxFQUFFLE1BQU0sRUFBRSxJQUFJLEVBQWlCLEVBQUUsRUFBRSxDQUNyRSxHQUFHLE1BQU0sQ0FBQyxXQUFXLEVBQUUsS0FBSyxJQUFJLENBQUMsV0FBVyxFQUFFLEVBQUUsQ0FBQztBQUR0QyxRQUFBLG1CQUFtQix1QkFDbUI7QUEyRW5EOztHQUVHO0FBQ0gsTUFBTSx1QkFBdUIsR0FBRyxDQUFDLE9BQWlDLEVBQUUsRUFBRSxDQUNwRSxPQUFPLEVBQUUsY0FBYyxDQUFDLENBQUMsQ0FBQyxDQUFDLEVBQUUsQ0FBQyxnQ0FBb0IsQ0FBQyxPQUFPLENBQUMsRUFBRSxFQUFFLEVBQUUsQ0FBQyxDQUFDLENBQUMsQ0FBQyxFQUFFLENBQUM7QUFFMUU7O0dBRUc7QUFDSCxNQUFNLGlCQUFpQixHQUFHLENBQUMsT0FBaUMsRUFBRSxFQUFFLENBQUMsQ0FBQztJQUNoRSxRQUFRLEVBQUUsdUJBQXVCLENBQUMsT0FBTyxDQUFDO0lBQzFDLDBCQUEwQixFQUFFO1FBQzFCLElBQUksRUFBRSxNQUFNO0tBQ2I7Q0FDRixDQUFDLENBQUM7QUFFSDs7OztHQUlHO0FBQ0gsTUFBTSxxQkFBcUIsR0FBRyxDQUM1QixnQkFBZ0QsRUFDaEQsT0FBaUMsRUFDakMsRUFBRTtJQUNGLElBQUksZ0JBQWdCLElBQUksT0FBTyxFQUFFO1FBQy9CLElBQUksZ0JBQWdCLEVBQUUsWUFBWSxLQUFLLGdDQUFvQixDQUFDLElBQUksRUFBRTtZQUNoRSxPQUFPLGlCQUFpQixDQUFDLE9BQU8sQ0FBQyxDQUFDO1NBQ25DO2FBQU07WUFDTCxPQUFPO2dCQUNMLFFBQVEsRUFBRTtvQkFDUixHQUFHLENBQUMsZ0JBQWdCO3dCQUNsQixDQUFDLENBQUM7NEJBQ0U7Z0NBQ0UsQ0FBQyxnQkFBZ0IsQ0FBQyxZQUFZLENBQUMsRUFDN0IsZ0JBQWdCLENBQUMsbUJBQW1CLElBQUksRUFBRTs2QkFDN0M7eUJBQ0Y7d0JBQ0gsQ0FBQyxDQUFDLEVBQUUsQ0FBQztvQkFDUCxHQUFHLHVCQUF1QixDQUFDLE9BQU8sQ0FBQztpQkFDcEM7YUFDRixDQUFDO1NBQ0g7S0FDRjtJQUNELE9BQU8sRUFBRSxDQUFDO0FBQ1osQ0FBQyxDQUFDO0FBRUY7O0dBRUc7QUFDSCxNQUFNLHNCQUFzQixHQUFHLENBQzdCLElBQVksRUFDWixNQUFjLEVBQ2QsRUFDRSxZQUFZLEVBQ1osV0FBVyxFQUNYLGFBQWEsRUFDYiwwQkFBMEIsR0FDSixFQUN4QixTQUFvQyxFQUNwQyxnQkFBMEQsRUFDbkIsRUFBRTtJQUN6QyxNQUFNLGFBQWEsR0FBRyxnQkFBZ0IsQ0FBQyxFQUFFLE1BQU0sRUFBRSxJQUFJLEVBQUUsQ0FBQyxDQUFDO0lBQ3pELElBQUksQ0FBQyxDQUFDLGFBQWEsSUFBSSxZQUFZLENBQUMsRUFBRTtRQUNwQyxNQUFNLElBQUksS0FBSyxDQUNiLDhDQUE4QyxhQUFhLEtBQUssTUFBTSxJQUFJLElBQUksR0FBRyxDQUNsRixDQUFDO0tBQ0g7SUFFRCxJQUFJLEVBQUUsZ0JBQWdCLEVBQUUsV0FBVyxFQUFFLE9BQU8sRUFBRSxHQUM1QyxZQUFZLENBQUMsYUFBOEMsQ0FBQyxDQUFDO0lBRS9ELDJCQUEyQixDQUN6QixnQkFBZ0IsRUFDaEIsU0FBUyxDQUFDLFFBQVEsRUFDbEIsYUFBYSxDQUNkLENBQUM7SUFFRixJQUFJLG1CQUFtQixHQUF3QyxPQUFPLENBQUM7SUFFdkUsaUZBQWlGO0lBQ2pGLHNCQUFzQjtJQUN0QixJQUFJLENBQUMsbUJBQW1CLElBQUksYUFBYSxFQUFFLGlCQUFpQixFQUFFO1FBQzVELG1CQUFtQixHQUFHLEVBQUUsY0FBYyxFQUFFLElBQUksRUFBRSxDQUFDO0tBQ2hEO0lBRUQsdUZBQXVGO0lBQ3ZGLHNDQUFzQztJQUN0QyxJQUNFLGFBQWEsRUFBRSxNQUFNLEtBQUssUUFBUTtRQUNsQyxtQkFBbUIsRUFBRSxjQUFjLEVBQ25DO1FBQ0EsTUFBTSxJQUFJLEtBQUssQ0FDYixnRUFBZ0UsYUFBYSxLQUFLLE1BQU0sSUFBSSxJQUFJLEdBQUcsQ0FDcEcsQ0FBQztLQUNIO0lBRUQscUVBQXFFO0lBQ3JFLGdCQUFnQixHQUFHLGdCQUFnQixJQUFJLDBCQUEwQixDQUFDO0lBRWxFLE9BQU87UUFDTCxHQUFHLFNBQVM7UUFDWixTQUFTLEVBQUUsTUFBTSxDQUFDLFdBQVcsQ0FDM0IsTUFBTSxDQUFDLE9BQU8sQ0FBQyxTQUFTLENBQUMsU0FBUyxDQUFDLENBQUMsR0FBRyxDQUFDLENBQUMsQ0FBQyxVQUFVLEVBQUUsUUFBUSxDQUFDLEVBQUUsRUFBRSxDQUFDO1lBQ2xFLFVBQVU7WUFDVjtnQkFDRSxHQUFHLFFBQVE7Z0JBQ1gsT0FBTyxFQUFFO29CQUNQLEdBQUcsQ0FBQyxXQUFXLENBQUMsQ0FBQyxDQUFDLHdCQUF3QixFQUFFLENBQUMsQ0FBQyxDQUFDLEVBQUUsQ0FBQztvQkFDbEQsc0RBQXNEO29CQUN0RCxHQUFJLFFBQXFDLENBQUMsT0FBTztpQkFDbEQ7YUFDRjtTQUNGLENBQUMsQ0FDSDtRQUNELCtHQUErRztRQUMvRyxpQ0FBaUMsRUFBRSxXQUFXO1FBQzlDLEdBQUcscUJBQXFCLENBQUMsZ0JBQWdCLEVBQUUsbUJBQW1CLENBQUM7S0FDekQsQ0FBQztBQUNYLENBQUMsQ0FBQztBQUVGLE1BQU0sd0JBQXdCLEdBQUcsR0FFL0IsRUFBRSxDQUFDLENBQUM7SUFDSiw2QkFBNkIsRUFBRTtRQUM3QixNQUFNLEVBQUUsRUFBRSxJQUFJLEVBQUUsUUFBUSxFQUFFO0tBQzNCO0lBQ0QsOEJBQThCLEVBQUU7UUFDOUIsTUFBTSxFQUFFLEVBQUUsSUFBSSxFQUFFLFFBQVEsRUFBRTtLQUMzQjtJQUNELDhCQUE4QixFQUFFO1FBQzlCLE1BQU0sRUFBRSxFQUFFLElBQUksRUFBRSxRQUFRLEVBQUU7S0FDM0I7Q0FDRixDQUFDLENBQUM7QUFFSCxNQUFNLDJCQUEyQixHQUFHLENBQ2xDLFdBQWtDLEVBQ1AsRUFBRSxDQUFDLENBQUM7SUFDL0IsOEJBQThCLEVBQUUsSUFBSSxXQUFXLENBQUMsWUFBWSxDQUFDLElBQUksQ0FBQyxHQUFHLENBQUMsR0FBRztJQUN6RSw4QkFBOEIsRUFBRSxJQUFJLFdBQVcsQ0FBQyxZQUFZLENBQUMsSUFBSSxDQUFDLEdBQUcsQ0FBQyxHQUFHO0lBQ3pFLDZCQUE2QixFQUFFLElBQUksV0FBVyxDQUFDLFlBQVksQ0FBQyxJQUFJLENBQUMsR0FBRyxDQUFDLEdBQUc7Q0FDekUsQ0FBQyxDQUFDO0FBRUksTUFBTSw4QkFBOEIsR0FBRyxDQUM1QyxXQUFrQyxFQUNsQyxTQUFpQix3QkFBd0IsRUFDZCxFQUFFLENBQzdCLE1BQU0sQ0FBQyxXQUFXLENBQ2hCLE1BQU0sQ0FBQyxPQUFPLENBQUMsMkJBQTJCLENBQUMsV0FBVyxDQUFDLENBQUMsQ0FBQyxHQUFHLENBQzFELENBQUMsQ0FBQyxNQUFNLEVBQUUsS0FBSyxDQUFDLEVBQUUsRUFBRSxDQUFDLENBQUMsR0FBRyxNQUFNLElBQUksTUFBTSxFQUFFLEVBQUUsS0FBSyxDQUFDLENBQ3BELENBQ0YsQ0FBQztBQVJTLFFBQUEsOEJBQThCLGtDQVF2QztBQUVKOztHQUVHO0FBQ0gsTUFBTSx5QkFBeUIsR0FBRyxDQUNoQyxRQUFrQyxFQUNsQyxFQUFFLFdBQVcsRUFBeUIsRUFDWixFQUFFO0lBQzVCLG1FQUFtRTtJQUNuRSxJQUFJLHVCQUFXLENBQUMsT0FBTyxJQUFJLFFBQVEsSUFBSSxDQUFDLFdBQVcsRUFBRTtRQUNuRCxPQUFPLEVBQUUsQ0FBQztLQUNYO0lBRUQsTUFBTSxVQUFVLEdBQUcsV0FBVyxDQUFDLFVBQVUsQ0FBQztJQUUxQyxPQUFPO1FBQ0wsQ0FBQyx1QkFBVyxDQUFDLE9BQU8sQ0FBQyxFQUFFO1lBQ3JCLE9BQU8sRUFBRSxjQUFjO1lBQ3ZCLFdBQVcsRUFBRSw4Q0FBOEM7WUFDM0QsU0FBUyxFQUFFO2dCQUNULENBQUMsR0FBRyxVQUFVLEVBQUUsQ0FBQyxFQUFFO29CQUNqQixXQUFXLEVBQUUsa0NBQWtDO29CQUMvQyxPQUFPLEVBQUUsd0JBQXdCLEVBQUU7b0JBQ25DLE9BQU8sRUFBRSxFQUFFO2lCQUNaO2FBQ0Y7WUFDRCwwRkFBMEY7WUFDMUYsaUNBQWlDLEVBQUU7Z0JBQ2pDLElBQUksRUFBRSxNQUFNO2dCQUNaLGdCQUFnQixFQUFFO29CQUNoQixrQkFBa0IsRUFBRSxrQkFBa0IsVUFBVSxHQUFHO2lCQUNwRDtnQkFDRCxTQUFTLEVBQUU7b0JBQ1QsT0FBTyxFQUFFO3dCQUNQLFVBQVUsRUFBRSxHQUFHLFVBQVUsRUFBRTt3QkFDM0Isa0JBQWtCLEVBQUUsSUFBQSxzQ0FBOEIsRUFBQyxXQUFXLENBQUM7d0JBQy9ELGlCQUFpQixFQUFFOzRCQUNqQixrQkFBa0IsRUFBRSxJQUFJO3lCQUN6QjtxQkFDRjtpQkFDRjthQUNGO1lBQ0Qsb0NBQW9DO1lBQ3BDLEdBQUcsaUJBQWlCLEVBQUU7U0FDdkI7S0FDRixDQUFDO0FBQ0osQ0FBQyxDQUFDO0FBRUY7O0dBRUc7QUFDSCxNQUFNLGVBQWUsR0FBRyxDQUN0QixJQUFZLEVBQ1osUUFBa0MsRUFDbEMsT0FBOEIsRUFDOUIsZ0JBQTBELEVBQ2hDLEVBQUU7SUFDNUIsTUFBTSxvQkFBb0IsR0FBRyxJQUFJLEdBQUcsQ0FBUyxNQUFNLENBQUMsTUFBTSxDQUFDLHVCQUFXLENBQUMsQ0FBQyxDQUFDO0lBQ3pFLE1BQU0sd0JBQXdCLEdBQUcsTUFBTSxDQUFDLElBQUksQ0FBQyxRQUFRLENBQUMsQ0FBQyxNQUFNLENBQzNELENBQUMsTUFBTSxFQUFFLEVBQUUsQ0FBQyxDQUFDLG9CQUFvQixDQUFDLEdBQUcsQ0FBQyxNQUFNLENBQUMsQ0FDOUMsQ0FBQztJQUNGLElBQUksd0JBQXdCLENBQUMsTUFBTSxHQUFHLENBQUMsRUFBRTtRQUN2QyxNQUFNLElBQUksS0FBSyxDQUNiLFFBQVEsSUFBSSwrQkFDVix3QkFBd0IsQ0FBQyxNQUFNLEdBQUcsQ0FBQyxDQUFDLENBQUMsQ0FBQyxHQUFHLENBQUMsQ0FBQyxDQUFDLEVBQzlDLElBQUksd0JBQXdCLENBQUMsSUFBSSxDQUMvQixJQUFJLENBQ0wsMkJBQTJCLE1BQU0sQ0FBQyxNQUFNLENBQUMsdUJBQVcsQ0FBQyxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsR0FBRyxDQUNyRSxDQUFDO0tBQ0g7SUFFRCxPQUFPO1FBQ0wsR0FBRyxRQUFRO1FBQ1gsR0FBRyxNQUFNLENBQUMsV0FBVyxDQUNuQixNQUFNLENBQUMsTUFBTSxDQUFDLHVCQUFXLENBQUM7YUFDdkIsTUFBTSxDQUFDLENBQUMsTUFBTSxFQUFFLEVBQUUsQ0FBQyxRQUFRLENBQUMsTUFBTSxDQUFDLENBQUM7YUFDcEMsR0FBRyxDQUFDLENBQUMsTUFBTSxFQUFFLEVBQUUsQ0FBQztZQUNmLE1BQU07WUFDTixzQkFBc0IsQ0FDcEIsSUFBSSxFQUNKLE1BQU0sRUFDTixPQUFPLEVBQ1AsUUFBUSxDQUFDLE1BQU0sQ0FBRSxFQUNqQixnQkFBZ0IsQ0FDakI7U0FDRixDQUFDLENBQ0w7UUFDRCx1RkFBdUY7UUFDdkYsR0FBRyx5QkFBeUIsQ0FBQyxRQUFRLEVBQUUsT0FBTyxDQUFDO0tBQ2hELENBQUM7QUFDSixDQUFDLENBQUM7QUFFRjs7R0FFRztBQUNILE1BQU0sS0FBSyxHQUFHLENBQUMsR0FBUSxFQUFvQyxFQUFFLENBQUMsTUFBTSxJQUFJLEdBQUcsQ0FBQztBQUU1RTs7Ozs7OztHQU9HO0FBQ0gsTUFBTSx1QkFBdUIsR0FBRyxDQUM5Qix3QkFBMkUsRUFDM0UsMkJBRUMsRUFDRCxFQUFFO0lBQ0YsSUFBSSwyQkFBMkIsRUFBRTtRQUMvQixNQUFNLDBCQUEwQixHQUFHLElBQUksR0FBRyxDQUN4QyxNQUFNLENBQUMsSUFBSSxDQUFDLHdCQUF3QixDQUFDLENBQ3RDLENBQUM7UUFDRixNQUFNLHlCQUF5QixHQUFHLElBQUksR0FBRyxDQUN2QyxNQUFNLENBQUMsSUFBSSxDQUFDLDJCQUEyQixDQUFDLENBQ3pDLENBQUM7UUFFRixNQUFNLDRCQUE0QixHQUFHLENBQUMsR0FBRywwQkFBMEIsQ0FBQyxDQUFDLE1BQU0sQ0FDekUsQ0FBQyxFQUFFLEVBQUUsRUFBRSxDQUFDLHlCQUF5QixDQUFDLEdBQUcsQ0FBQyxFQUFFLENBQUMsQ0FDMUMsQ0FBQztRQUVGLG1JQUFtSTtRQUNuSSxvSUFBb0k7UUFDcEksMEVBQTBFO1FBQzFFLDRCQUE0QixDQUFDLE9BQU8sQ0FBQyxDQUFDLFFBQVEsRUFBRSxFQUFFO1lBQ2hELElBQUksQ0FBQyxLQUFLLENBQUMsMkJBQTJCLENBQUMsUUFBUSxDQUFDLENBQUMsRUFBRTtnQkFDakQsTUFBTSxjQUFjLEdBQUcsMkJBQTJCLENBQ2hELFFBQVEsQ0FDeUIsQ0FBQztnQkFFcEMsSUFBSSx3QkFBd0IsQ0FBQyxRQUFRLENBQUMsQ0FBQyxJQUFJLEtBQUssY0FBYyxDQUFDLElBQUksRUFBRTtvQkFDbkUsTUFBTSxJQUFJLEtBQUssQ0FDYiwyQkFBMkIsUUFBUSxnQkFBZ0Isd0JBQXdCLENBQUMsUUFBUSxDQUFDLENBQUMsSUFBSSxxQkFBcUIsY0FBYyxDQUFDLElBQUksbUNBQW1DLENBQ3RLLENBQUM7aUJBQ0g7Z0JBQ0QsTUFBTSwyQkFBMkIsR0FDL0Isd0JBQXdCLENBQUMsUUFBUSxDQUNsQyxDQUFDLDhCQUE4QixDQUFDLENBQUM7Z0JBQ2xDLE1BQU0sMEJBQTBCLEdBQUksY0FBc0IsQ0FDeEQsOEJBQThCLENBQy9CLENBQUM7Z0JBRUYsSUFBSSwyQkFBMkIsS0FBSywwQkFBMEIsRUFBRTtvQkFDOUQsTUFBTSxJQUFJLEtBQUssQ0FDYiwyQkFBMkIsUUFBUSxnQkFBZ0IsMkJBQTJCLHFCQUFxQiwwQkFBMEIsbUNBQW1DLENBQ2pLLENBQUM7aUJBQ0g7YUFDRjtpQkFBTTtnQkFDTCxNQUFNLElBQUksS0FBSyxDQUNiLDJCQUEyQixRQUFRLDZFQUE2RSxDQUNqSCxDQUFDO2FBQ0g7UUFDSCxDQUFDLENBQUMsQ0FBQztLQUNKO0FBQ0gsQ0FBQyxDQUFDO0FBRUY7Ozs7OztHQU1HO0FBQ0gsTUFBTSwyQkFBMkIsR0FBRyxDQUNsQyxtQkFBbUQsRUFDbkQsdUJBQStELEVBQy9ELFlBQW9CLFNBQVMsRUFDN0IsRUFBRTtJQUNGLHlFQUF5RTtJQUN6RSxJQUFJLG1CQUFtQixJQUFJLHVCQUF1QixFQUFFO1FBQ2xELE1BQU0scUJBQXFCLEdBQUcsTUFBTSxDQUFDLFdBQVcsQ0FDOUMsdUJBQXVCLENBQUMsT0FBTyxDQUFDLENBQUMsbUJBQW1CLEVBQUUsRUFBRSxDQUN0RCxNQUFNLENBQUMsSUFBSSxDQUFDLG1CQUFtQixDQUFDLENBQUMsR0FBRyxDQUFDLENBQUMsRUFBRSxFQUFFLEVBQUUsQ0FBQztZQUMzQyxFQUFFO1lBQ0YsbUJBQW1CLENBQUMsRUFBRSxDQUFDO1NBQ3hCLENBQUMsQ0FDSCxDQUNGLENBQUM7UUFDRixNQUFNLGlCQUFpQixHQUFHLE1BQU0sQ0FBQyxJQUFJLENBQUMscUJBQXFCLENBQUMsQ0FBQztRQUU3RCxJQUFJLGlCQUFpQixDQUFDLE1BQU0sR0FBRyxDQUFDLEVBQUU7WUFDaEMsMkVBQTJFO1lBQzNFLE1BQU0sSUFBSSxLQUFLLENBQ2IsR0FBRyxTQUFTLGdCQUFnQixpQkFBaUI7aUJBQzFDLElBQUksRUFBRTtpQkFDTixJQUFJLENBQ0gsSUFBSSxDQUNMLG1HQUNELG1CQUFtQixDQUFDLFlBQ3RCLEVBQUUsQ0FDSCxDQUFDO1NBQ0g7YUFBTSxJQUFJLGlCQUFpQixDQUFDLE1BQU0sS0FBSyxDQUFDLEVBQUU7WUFDekMsdURBQXVEO1lBQ3ZELElBQUksaUJBQWlCLENBQUMsQ0FBQyxDQUFDLEtBQUssbUJBQW1CLENBQUMsWUFBWSxFQUFFO2dCQUM3RCxNQUFNLElBQUksS0FBSyxDQUNiLEdBQUcsU0FBUyxlQUFlLGlCQUFpQixDQUFDLENBQUMsQ0FBQyw0RkFBNEYsbUJBQW1CLENBQUMsWUFBWSxFQUFFLENBQzlLLENBQUM7YUFDSDtZQUVELDBFQUEwRTtZQUMxRSxNQUFNLFVBQVUsR0FBRyxJQUFJLEdBQUcsQ0FBQyxxQkFBcUIsQ0FBQyxpQkFBaUIsQ0FBQyxDQUFDLENBQUMsQ0FBQyxDQUFDLENBQUM7WUFDeEUsTUFBTSxlQUFlLEdBQUcsSUFBSSxHQUFHLENBQUMsbUJBQW1CLENBQUMsbUJBQW1CLENBQUMsQ0FBQztZQUN6RSxNQUFNLGVBQWUsR0FBRztnQkFDdEIsR0FBRyxDQUFDLEdBQUcsVUFBVSxDQUFDLENBQUMsTUFBTSxDQUFDLENBQUMsS0FBSyxFQUFFLEVBQUUsQ0FBQyxDQUFDLGVBQWUsQ0FBQyxHQUFHLENBQUMsS0FBSyxDQUFDLENBQUM7Z0JBQ2pFLEdBQUcsQ0FBQyxHQUFHLGVBQWUsQ0FBQyxDQUFDLE1BQU0sQ0FBQyxDQUFDLEtBQUssRUFBRSxFQUFFLENBQUMsQ0FBQyxVQUFVLENBQUMsR0FBRyxDQUFDLEtBQUssQ0FBQyxDQUFDO2FBQ2xFLENBQUM7WUFDRixJQUFJLGVBQWUsQ0FBQyxNQUFNLEdBQUcsQ0FBQyxFQUFFO2dCQUM5QixNQUFNLElBQUksS0FBSyxDQUNiLEdBQUcsU0FBUyxzQkFBc0IsQ0FBQyxHQUFHLFVBQVUsQ0FBQyxDQUFDLElBQUksQ0FDcEQsSUFBSSxDQUNMLG9GQUFvRjtvQkFDbkYsR0FBRyxlQUFlO2lCQUNuQixDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsR0FBRyxDQUNoQixDQUFDO2FBQ0g7U0FDRjthQUFNLElBQUksbUJBQW1CLENBQUMsWUFBWSxLQUFLLGdDQUFvQixDQUFDLElBQUksRUFBRTtZQUN6RSwwSEFBMEg7WUFDMUgsTUFBTSxJQUFJLEtBQUssQ0FDYixHQUFHLFNBQVMscUhBQXFILG1CQUFtQixDQUFDLFlBQVksRUFBRSxDQUNwSyxDQUFDO1NBQ0g7S0FDRjtBQUNILENBQUMsQ0FBQztBQUVGOztHQUVHO0FBQ0gsTUFBTSxvQkFBb0IsR0FBRyxDQUFDLElBQXdCLEVBQVksRUFBRTtJQUNsRSxNQUFNLG1CQUFtQixHQUFHLE1BQU0sQ0FBQyxNQUFNLENBQUMsSUFBSSxDQUFDLEtBQUssQ0FBQyxDQUFDLE9BQU8sQ0FBQyxDQUFDLFdBQVcsRUFBRSxFQUFFLENBQzVFLE1BQU0sQ0FBQyxNQUFNLENBQUMsdUJBQVcsQ0FBQyxDQUFDLE9BQU8sQ0FBQyxDQUFDLE1BQU0sRUFBRSxFQUFFLENBQzVDLENBQUMsV0FBVyxFQUFFLENBQUMsTUFBTSxDQUFDLEVBQUUsVUFBVSxJQUFJLEVBQUUsQ0FBQyxDQUFDLE9BQU8sQ0FBQyxDQUFDLFNBQVMsRUFBRSxFQUFFLENBQzlELElBQUksSUFBSSxTQUFTLElBQUksU0FBUyxDQUFDLEVBQUUsS0FBSyxRQUFRLENBQUMsQ0FBQyxDQUFDLENBQUMsU0FBUyxDQUFDLElBQUksQ0FBQyxDQUFDLENBQUMsQ0FBQyxFQUFFLENBQ3ZFLENBQ0YsQ0FDRixDQUFDO0lBQ0YsTUFBTSxrQkFBa0IsR0FBRyxJQUFJLEdBQUcsRUFBVSxDQUFDO0lBQzdDLE9BQU8sbUJBQW1CLENBQUMsTUFBTSxDQUFDLENBQUMsQ0FBQyxFQUFFLEVBQUU7UUFDdEMsTUFBTSxJQUFJLEdBQUcsa0JBQWtCLENBQUMsR0FBRyxDQUFDLENBQUMsQ0FBQyxDQUFDO1FBQ3ZDLGtCQUFrQixDQUFDLEdBQUcsQ0FBQyxDQUFDLENBQUMsQ0FBQztRQUMxQixPQUFPLENBQUMsSUFBSSxDQUFDO0lBQ2YsQ0FBQyxDQUFDLENBQUM7QUFDTCxDQUFDLENBQUM7QUFFRjs7R0FFRztBQUNJLE1BQU0sY0FBYyxHQUFHLENBQzVCLElBQXdCLEVBQ3hCLE9BQThCLEVBQ1YsRUFBRTtJQUN0QixnRUFBZ0U7SUFDaEUsTUFBTSxtQkFBbUIsR0FBRyxNQUFNLENBQUMsV0FBVyxDQUM1QyxNQUFNLENBQUMsT0FBTyxDQUFnQixPQUFPLENBQUMsZUFBZSxDQUFDLENBQUMsR0FBRyxDQUN4RCxDQUFDLENBQUMsYUFBYSxFQUFFLGFBQWEsQ0FBQyxFQUFFLEVBQUUsQ0FBQztRQUNsQyxJQUFBLDJCQUFtQixFQUFDLGFBQWEsQ0FBQztRQUNsQyxhQUFhO0tBQ2QsQ0FDRixDQUNGLENBQUM7SUFDRixNQUFNLGdCQUFnQixHQUFHLENBQUMsYUFBNEIsRUFBRSxFQUFFLENBQ3hELG1CQUFtQixDQUFDLElBQUEsMkJBQW1CLEVBQUMsYUFBYSxDQUFDLENBQUMsQ0FBQztJQUUxRCx1QkFBdUIsQ0FDckIsT0FBTyxDQUFDLGVBQWUsRUFDdkIsSUFBSSxDQUFDLFVBQVUsRUFBRSxlQUFlLENBQ2pDLENBQUM7SUFDRiwyQkFBMkIsQ0FDekIsT0FBTyxDQUFDLDBCQUEwQixFQUNsQyxJQUFJLENBQUMsUUFBUSxDQUNkLENBQUM7SUFFRixpR0FBaUc7SUFDakcsMEVBQTBFO0lBQzFFLE1BQU0sV0FBVyxHQUFzQyxPQUFPLENBQUMsV0FBVztRQUN4RSxDQUFDLENBQUM7WUFDRSxHQUFHLE9BQU8sQ0FBQyxXQUFXO1lBQ3RCLFlBQVksRUFBRTtnQkFDWixHQUFHLE9BQU8sQ0FBQyxXQUFXLENBQUMsWUFBWTtnQkFDbkMsR0FBRyxvQkFBb0IsQ0FBQyxJQUFJLENBQUM7YUFDOUI7U0FDRjtRQUNILENBQUMsQ0FBQyxTQUFTLENBQUM7SUFFZCxNQUFNLGNBQWMsR0FBMEI7UUFDNUMsR0FBRyxPQUFPO1FBQ1YsV0FBVztLQUNaLENBQUM7SUFFRixPQUFPO1FBQ0wsR0FBRyxJQUFJO1FBQ1Asc0hBQXNIO1FBQ3RILHdDQUF3QyxFQUFFO1lBQ3hDLEdBQUcsRUFBRTtnQkFDSCxtQkFBbUIsRUFBRSxJQUFJO2dCQUN6Qix5QkFBeUIsRUFBRSxJQUFJO2FBQ2hDO1NBQ0Y7UUFDRCx1Q0FBdUMsRUFBRSxLQUFLO1FBQzlDLHFIQUFxSDtRQUNySCx1Q0FBdUMsRUFBRTtZQUN2QyxnQkFBZ0IsRUFBRTtnQkFDaEIsVUFBVSxFQUFFLEdBQUc7Z0JBQ2YsaUJBQWlCLEVBQUU7b0JBQ2pCLGtCQUFrQixFQUNoQixxREFBcUQ7aUJBQ3hEO2dCQUNELEdBQUcsQ0FBQyxXQUFXO29CQUNiLENBQUMsQ0FBQzt3QkFDRSxrQkFBa0IsRUFBRSxJQUFBLHNDQUE4QixFQUNoRCxXQUFXLEVBQ1gsd0JBQXdCLENBQ3pCO3FCQUNGO29CQUNILENBQUMsQ0FBQyxFQUFFLENBQUM7YUFDUjtTQUNGO1FBQ0QsS0FBSyxFQUFFO1lBQ0wsR0FBRyxNQUFNLENBQUMsV0FBVyxDQUNuQixNQUFNLENBQUMsT0FBTyxDQUFDLElBQUksQ0FBQyxLQUFLLENBQUMsQ0FBQyxHQUFHLENBQUMsQ0FBQyxDQUFDLElBQUksRUFBRSxXQUFXLENBQUMsRUFBRSxFQUFFLENBQUM7Z0JBQ3RELElBQUk7Z0JBQ0osZUFBZSxDQUFDLElBQUksRUFBRSxXQUFZLEVBQUUsY0FBYyxFQUFFLGdCQUFnQixDQUFDO2FBQ3RFLENBQUMsQ0FDSDtTQUNGO1FBQ0QsVUFBVSxFQUFFO1lBQ1YsR0FBRyxJQUFJLENBQUMsVUFBVTtZQUNsQixlQUFlLEVBQUU7Z0JBQ2YsNERBQTREO2dCQUM1RCxHQUFHLElBQUksQ0FBQyxVQUFVLEVBQUUsZUFBZTtnQkFDbkMsdUVBQXVFO2dCQUN2RSxHQUFHLGNBQWMsQ0FBQyxlQUFlO2FBQ2xDO1NBQ0Y7UUFDRCxHQUFHLENBQUMsY0FBYyxDQUFDLGFBQWE7WUFDOUIsQ0FBQyxDQUFDO2dCQUNFLGtIQUFrSDtnQkFDbEgsb0NBQW9DLEVBQ2xDLGNBQWMsQ0FBQyxhQUFhLENBQUMsTUFBTTthQUN0QztZQUNILENBQUMsQ0FBQyxFQUFFLENBQUM7S0FDRCxDQUFDO0FBQ1gsQ0FBQyxDQUFDO0FBL0ZXLFFBQUEsY0FBYyxrQkErRnpCIiwic291cmNlc0NvbnRlbnQiOlsiLyohIENvcHlyaWdodCBbQW1hem9uLmNvbV0oaHR0cDovL2FtYXpvbi5jb20vKSwgSW5jLiBvciBpdHMgYWZmaWxpYXRlcy4gQWxsIFJpZ2h0cyBSZXNlcnZlZC5cblNQRFgtTGljZW5zZS1JZGVudGlmaWVyOiBBcGFjaGUtMi4wICovXG5pbXBvcnQgdHlwZSB7IE9wZW5BUElWMyB9IGZyb20gXCJvcGVuYXBpLXR5cGVzXCI7XG5pbXBvcnQgeyBEZWZhdWx0QXV0aG9yaXplcklkcywgSHR0cE1ldGhvZHMgfSBmcm9tIFwiLi9jb25zdGFudHNcIjtcbmltcG9ydCB7IEFwaUdhdGV3YXlJbnRlZ3JhdGlvbiB9IGZyb20gXCIuLi9pbnRlZ3JhdGlvbnNcIjtcbmltcG9ydCB0eXBlIHtcbiAgTWV0aG9kLFxuICBNZXRob2RBbmRQYXRoLFxuICBPcGVyYXRpb25Mb29rdXAsXG4gIFNlcmlhbGl6ZWRDb3JzT3B0aW9ucyxcbiAgVHlwZVNhZmVBcGlJbnRlZ3JhdGlvbk9wdGlvbnMsXG4gIFR5cGVTYWZlQXBpSW50ZWdyYXRpb25zLFxufSBmcm9tIFwiLi4vc3BlY1wiO1xuaW1wb3J0IHsgU2VyaWFsaXNlZEF1dGhvcml6ZXJSZWZlcmVuY2UgfSBmcm9tIFwiLi4vc3BlYy9hcGktZ2F0ZXdheS1hdXRoXCI7XG5cbi8qKlxuICogU2VyaWFsaXNlIGEgbWV0aG9kIGFuZCBwYXRoIGludG8gYSBzaW5nbGUgc3RyaW5nXG4gKi9cbmV4cG9ydCBjb25zdCBjb25jYXRNZXRob2RBbmRQYXRoID0gKHsgbWV0aG9kLCBwYXRoIH06IE1ldGhvZEFuZFBhdGgpID0+XG4gIGAke21ldGhvZC50b0xvd2VyQ2FzZSgpfXx8JHtwYXRoLnRvTG93ZXJDYXNlKCl9YDtcblxuLyoqXG4gKiBTZXJpYWxpemVkIGludGVncmF0aW9uIGZvciBhIG1ldGhvZFxuICovXG5leHBvcnQgaW50ZXJmYWNlIFNlcmlhbGl6ZWRNZXRob2RJbnRlZ3JhdGlvbiB7XG4gIC8qKlxuICAgKiBUaGUgbGFtYmRhIGZ1bmN0aW9uIGludm9jYXRpb24gdXJpIGZvciB0aGUgYXBpIG1ldGhvZFxuICAgKi9cbiAgcmVhZG9ubHkgaW50ZWdyYXRpb246IEFwaUdhdGV3YXlJbnRlZ3JhdGlvbjtcbiAgLyoqXG4gICAqIFRoZSBhdXRob3JpemVyIChpZiBhbnkpIHRvIGFwcGx5IHRvIHRoZSBtZXRob2RcbiAgICovXG4gIHJlYWRvbmx5IG1ldGhvZEF1dGhvcml6ZXI/OiBTZXJpYWxpc2VkQXV0aG9yaXplclJlZmVyZW5jZTtcbiAgLyoqXG4gICAqIE9wdGlvbnMgZm9yIHRoZSBpbnRlZ3JhdGlvblxuICAgKi9cbiAgcmVhZG9ubHkgb3B0aW9ucz86IFR5cGVTYWZlQXBpSW50ZWdyYXRpb25PcHRpb25zO1xufVxuXG4vKipcbiAqIE9wdGlvbnMgZm9yIEFQSSBrZXlzXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgU2VyaWFsaXplZEFwaUtleU9wdGlvbnMge1xuICAvKipcbiAgICogU291cmNlIHR5cGUgZm9yIGFuIEFQSSBrZXlcbiAgICovXG4gIHJlYWRvbmx5IHNvdXJjZTogc3RyaW5nO1xuICAvKipcbiAgICogU2V0IHRvIHRydWUgdG8gcmVxdWlyZSBhbiBBUEkga2V5IG9uIGFsbCBvcGVyYXRpb25zIGJ5IGRlZmF1bHQuXG4gICAqIE9ubHkgYXBwbGljYWJsZSB3aGVuIHRoZSBzb3VyY2UgaXMgSEVBREVSLlxuICAgKi9cbiAgcmVhZG9ubHkgcmVxdWlyZWRCeURlZmF1bHQ/OiBib29sZWFuO1xufVxuXG4vKipcbiAqIE9wdGlvbnMgZm9yIHByZXBhcmluZyBhbiBhcGkgc3BlYyBmb3IgZGVwbG95bWVudCBieSBhcGkgZ2F0ZXdheVxuICovXG5leHBvcnQgaW50ZXJmYWNlIFByZXBhcmVBcGlTcGVjT3B0aW9ucyB7XG4gIC8qKlxuICAgKiBJbnRlZ3JhdGlvbnMgZm9yIGFwaSBvcGVyYXRpb25zXG4gICAqL1xuICByZWFkb25seSBpbnRlZ3JhdGlvbnM6IHsgW29wZXJhdGlvbklkOiBzdHJpbmddOiBTZXJpYWxpemVkTWV0aG9kSW50ZWdyYXRpb24gfTtcbiAgLyoqXG4gICAqIE9wdGlvbnMgZm9yIGNyb3NzLW9yaWdpbiByZXNvdXJjZSBzaGFyaW5nXG4gICAqL1xuICByZWFkb25seSBjb3JzT3B0aW9ucz86IFNlcmlhbGl6ZWRDb3JzT3B0aW9ucztcbiAgLyoqXG4gICAqIE9wZXJhdGlvbiBpZCB0byBtZXRob2QgYW5kIHBhdGggbWFwcGluZ1xuICAgKi9cbiAgcmVhZG9ubHkgb3BlcmF0aW9uTG9va3VwOiBPcGVyYXRpb25Mb29rdXA7XG4gIC8qKlxuICAgKiBTZWN1cml0eSBzY2hlbWVzIHRvIGFkZCB0byB0aGUgc3BlY1xuICAgKi9cbiAgcmVhZG9ubHkgc2VjdXJpdHlTY2hlbWVzOiB7IFtrZXk6IHN0cmluZ106IE9wZW5BUElWMy5TZWN1cml0eVNjaGVtZU9iamVjdCB9O1xuICAvKipcbiAgICogVGhlIGRlZmF1bHQgYXV0aG9yaXplciB0byByZWZlcmVuY2VcbiAgICovXG4gIHJlYWRvbmx5IGRlZmF1bHRBdXRob3JpemVyUmVmZXJlbmNlPzogU2VyaWFsaXNlZEF1dGhvcml6ZXJSZWZlcmVuY2U7XG4gIC8qKlxuICAgKiBEZWZhdWx0IG9wdGlvbnMgZm9yIEFQSSBrZXlzXG4gICAqL1xuICByZWFkb25seSBhcGlLZXlPcHRpb25zPzogU2VyaWFsaXplZEFwaUtleU9wdGlvbnM7XG59XG5cbi8qKlxuICogQVBJIGtleSBvcHRpb25zIHdoZW4gcmVuZGVyaW5nIGFuIGF1dGhvcml6ZXJcbiAqL1xuaW50ZXJmYWNlIEF1dGhvcml6ZXJBcGlLZXlPcHRpb25zIHtcbiAgLyoqXG4gICAqIFdoZXRoZXIgYW4gYXBpIGtleSBpcyByZXF1aXJlZCBmb3IgdGhlIG1ldGhvZFxuICAgKi9cbiAgcmVhZG9ubHkgYXBpS2V5UmVxdWlyZWQ/OiBib29sZWFuO1xufVxuXG4vKipcbiAqIFJldHVybiBhbiBhcnJheSBvZiBzZWN1cml0eSBzY2hlbWUgcmVmZXJlbmNlcyBpbmNsdWRpbmcgdGhlIGFwaSBrZXkgb25lIGlmIHJlcXVpcmVkXG4gKi9cbmNvbnN0IGFwaUtleVNlY3VyaXR5UmVmZXJlbmNlID0gKG9wdGlvbnM/OiBBdXRob3JpemVyQXBpS2V5T3B0aW9ucykgPT5cbiAgb3B0aW9ucz8uYXBpS2V5UmVxdWlyZWQgPyBbeyBbRGVmYXVsdEF1dGhvcml6ZXJJZHMuQVBJX0tFWV06IFtdIH1dIDogW107XG5cbi8qKlxuICogR2VuZXJhdGUgYSBcIm5vIGF1dGhcIiBzcGVjIHNuaXBwZXRcbiAqL1xuY29uc3Qgbm9BdXRoU3BlY1NuaXBwZXQgPSAob3B0aW9ucz86IEF1dGhvcml6ZXJBcGlLZXlPcHRpb25zKSA9PiAoe1xuICBzZWN1cml0eTogYXBpS2V5U2VjdXJpdHlSZWZlcmVuY2Uob3B0aW9ucyksXG4gIFwieC1hbWF6b24tYXBpZ2F0ZXdheS1hdXRoXCI6IHtcbiAgICB0eXBlOiBcIk5PTkVcIixcbiAgfSxcbn0pO1xuXG4vKipcbiAqIENyZWF0ZSB0aGUgT3BlbkFQSSBkZWZpbml0aW9uIHdpdGggYXBpIGdhdGV3YXkgZXh0ZW5zaW9ucyBmb3IgdGhlIGdpdmVuIGF1dGhvcml6ZXJcbiAqIEBwYXJhbSBtZXRob2RBdXRob3JpemVyIHRoZSBhdXRob3JpemVyIHVzZWQgZm9yIHRoZSBtZXRob2RcbiAqIEBwYXJhbSBvcHRpb25zIGFwaSBpbnRlZ3JhdGlvbiBvcHRpb25zXG4gKi9cbmNvbnN0IGFwcGx5TWV0aG9kQXV0aG9yaXplciA9IChcbiAgbWV0aG9kQXV0aG9yaXplcj86IFNlcmlhbGlzZWRBdXRob3JpemVyUmVmZXJlbmNlLFxuICBvcHRpb25zPzogQXV0aG9yaXplckFwaUtleU9wdGlvbnNcbikgPT4ge1xuICBpZiAobWV0aG9kQXV0aG9yaXplciB8fCBvcHRpb25zKSB7XG4gICAgaWYgKG1ldGhvZEF1dGhvcml6ZXI/LmF1dGhvcml6ZXJJZCA9PT0gRGVmYXVsdEF1dGhvcml6ZXJJZHMuTk9ORSkge1xuICAgICAgcmV0dXJuIG5vQXV0aFNwZWNTbmlwcGV0KG9wdGlvbnMpO1xuICAgIH0gZWxzZSB7XG4gICAgICByZXR1cm4ge1xuICAgICAgICBzZWN1cml0eTogW1xuICAgICAgICAgIC4uLihtZXRob2RBdXRob3JpemVyXG4gICAgICAgICAgICA/IFtcbiAgICAgICAgICAgICAgICB7XG4gICAgICAgICAgICAgICAgICBbbWV0aG9kQXV0aG9yaXplci5hdXRob3JpemVySWRdOlxuICAgICAgICAgICAgICAgICAgICBtZXRob2RBdXRob3JpemVyLmF1dGhvcml6YXRpb25TY29wZXMgfHwgW10sXG4gICAgICAgICAgICAgICAgfSxcbiAgICAgICAgICAgICAgXVxuICAgICAgICAgICAgOiBbXSksXG4gICAgICAgICAgLi4uYXBpS2V5U2VjdXJpdHlSZWZlcmVuY2Uob3B0aW9ucyksXG4gICAgICAgIF0sXG4gICAgICB9O1xuICAgIH1cbiAgfVxuICByZXR1cm4ge307XG59O1xuXG4vKipcbiAqIEFkZHMgQVBJIEdhdGV3YXkgaW50ZWdyYXRpb25zIGFuZCBhdXRoIHRvIHRoZSBnaXZlbiBvcGVyYXRpb25cbiAqL1xuY29uc3QgYXBwbHlNZXRob2RJbnRlZ3JhdGlvbiA9IChcbiAgcGF0aDogc3RyaW5nLFxuICBtZXRob2Q6IE1ldGhvZCxcbiAge1xuICAgIGludGVncmF0aW9ucyxcbiAgICBjb3JzT3B0aW9ucyxcbiAgICBhcGlLZXlPcHRpb25zLFxuICAgIGRlZmF1bHRBdXRob3JpemVyUmVmZXJlbmNlLFxuICB9OiBQcmVwYXJlQXBpU3BlY09wdGlvbnMsXG4gIG9wZXJhdGlvbjogT3BlbkFQSVYzLk9wZXJhdGlvbk9iamVjdCxcbiAgZ2V0T3BlcmF0aW9uTmFtZTogKG1ldGhvZEFuZFBhdGg6IE1ldGhvZEFuZFBhdGgpID0+IHN0cmluZ1xuKTogT3BlbkFQSVYzLk9wZXJhdGlvbk9iamVjdCB8IHVuZGVmaW5lZCA9PiB7XG4gIGNvbnN0IG9wZXJhdGlvbk5hbWUgPSBnZXRPcGVyYXRpb25OYW1lKHsgbWV0aG9kLCBwYXRoIH0pO1xuICBpZiAoIShvcGVyYXRpb25OYW1lIGluIGludGVncmF0aW9ucykpIHtcbiAgICB0aHJvdyBuZXcgRXJyb3IoXG4gICAgICBgTWlzc2luZyByZXF1aXJlZCBpbnRlZ3JhdGlvbiBmb3Igb3BlcmF0aW9uICR7b3BlcmF0aW9uTmFtZX0gKCR7bWV0aG9kfSAke3BhdGh9KWBcbiAgICApO1xuICB9XG5cbiAgbGV0IHsgbWV0aG9kQXV0aG9yaXplciwgaW50ZWdyYXRpb24sIG9wdGlvbnMgfSA9XG4gICAgaW50ZWdyYXRpb25zW29wZXJhdGlvbk5hbWUgYXMga2V5b2YgVHlwZVNhZmVBcGlJbnRlZ3JhdGlvbnNdO1xuXG4gIHZhbGlkYXRlQXV0aG9yaXplclJlZmVyZW5jZShcbiAgICBtZXRob2RBdXRob3JpemVyLFxuICAgIG9wZXJhdGlvbi5zZWN1cml0eSxcbiAgICBvcGVyYXRpb25OYW1lXG4gICk7XG5cbiAgbGV0IG1ldGhvZEFwaUtleU9wdGlvbnM6IEF1dGhvcml6ZXJBcGlLZXlPcHRpb25zIHwgdW5kZWZpbmVkID0gb3B0aW9ucztcblxuICAvLyBXaGVuIG5vIEFQSSBrZXkgb3B0aW9ucyBhcmUgcHJlc2VudCBvbiB0aGUgbWV0aG9kLCByZXF1aXJlIHRoZSBBUEkga2V5IGlmIGl0J3NcbiAgLy8gcmVxdWlyZWQgYnkgZGVmYXVsdFxuICBpZiAoIW1ldGhvZEFwaUtleU9wdGlvbnMgJiYgYXBpS2V5T3B0aW9ucz8ucmVxdWlyZWRCeURlZmF1bHQpIHtcbiAgICBtZXRob2RBcGlLZXlPcHRpb25zID0geyBhcGlLZXlSZXF1aXJlZDogdHJ1ZSB9O1xuICB9XG5cbiAgLy8gQ2FuIG9ubHkgXCJyZXF1aXJlXCIgYW4gQVBJIGtleSBpZiBpdCdzIGluIGEgaGVhZGVyLCBzaW5jZSB3ZSBvbmx5IGRlZmluZSB0aGUgc2VjdXJpdHlcbiAgLy8gc2NoZW1lIHdlJ2QgcmVmZXJlbmNlIGluIHRoaXMgY2FzZS5cbiAgaWYgKFxuICAgIGFwaUtleU9wdGlvbnM/LnNvdXJjZSAhPT0gXCJIRUFERVJcIiAmJlxuICAgIG1ldGhvZEFwaUtleU9wdGlvbnM/LmFwaUtleVJlcXVpcmVkXG4gICkge1xuICAgIHRocm93IG5ldyBFcnJvcihcbiAgICAgIGBDYW5ub3QgcmVxdWlyZSBhbiBBUEkgS2V5IHdoZW4gQVBJIEtleSBzb3VyY2UgaXMgbm90IEhFQURFUjogJHtvcGVyYXRpb25OYW1lfSAoJHttZXRob2R9ICR7cGF0aH0pYFxuICAgICk7XG4gIH1cblxuICAvLyBBcHBseSB0aGUgZGVmYXVsdCBhdXRob3JpemVyIHVubGVzcyBhIG1ldGhvZCBhdXRob3JpemVyIGlzIGRlZmluZWRcbiAgbWV0aG9kQXV0aG9yaXplciA9IG1ldGhvZEF1dGhvcml6ZXIgPz8gZGVmYXVsdEF1dGhvcml6ZXJSZWZlcmVuY2U7XG5cbiAgcmV0dXJuIHtcbiAgICAuLi5vcGVyYXRpb24sXG4gICAgcmVzcG9uc2VzOiBPYmplY3QuZnJvbUVudHJpZXMoXG4gICAgICBPYmplY3QuZW50cmllcyhvcGVyYXRpb24ucmVzcG9uc2VzKS5tYXAoKFtzdGF0dXNDb2RlLCByZXNwb25zZV0pID0+IFtcbiAgICAgICAgc3RhdHVzQ29kZSxcbiAgICAgICAge1xuICAgICAgICAgIC4uLnJlc3BvbnNlLFxuICAgICAgICAgIGhlYWRlcnM6IHtcbiAgICAgICAgICAgIC4uLihjb3JzT3B0aW9ucyA/IGdldENvcnNIZWFkZXJEZWZpbml0aW9ucygpIDoge30pLFxuICAgICAgICAgICAgLy8gVE9ETzogQ29uc2lkZXIgZm9sbG93aW5nIHJlc3BvbnNlIGhlYWRlciByZWZlcmVuY2VzXG4gICAgICAgICAgICAuLi4ocmVzcG9uc2UgYXMgT3BlbkFQSVYzLlJlc3BvbnNlT2JqZWN0KS5oZWFkZXJzLFxuICAgICAgICAgIH0sXG4gICAgICAgIH0sXG4gICAgICBdKVxuICAgICksXG4gICAgLy8gaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2FwaWdhdGV3YXkvbGF0ZXN0L2RldmVsb3Blcmd1aWRlL2FwaS1nYXRld2F5LXN3YWdnZXItZXh0ZW5zaW9ucy1pbnRlZ3JhdGlvbi5odG1sXG4gICAgXCJ4LWFtYXpvbi1hcGlnYXRld2F5LWludGVncmF0aW9uXCI6IGludGVncmF0aW9uLFxuICAgIC4uLmFwcGx5TWV0aG9kQXV0aG9yaXplcihtZXRob2RBdXRob3JpemVyLCBtZXRob2RBcGlLZXlPcHRpb25zKSxcbiAgfSBhcyBhbnk7XG59O1xuXG5jb25zdCBnZXRDb3JzSGVhZGVyRGVmaW5pdGlvbnMgPSAoKToge1xuICBbbmFtZTogc3RyaW5nXTogT3BlbkFQSVYzLkhlYWRlck9iamVjdDtcbn0gPT4gKHtcbiAgXCJBY2Nlc3MtQ29udHJvbC1BbGxvdy1PcmlnaW5cIjoge1xuICAgIHNjaGVtYTogeyB0eXBlOiBcInN0cmluZ1wiIH0sXG4gIH0sXG4gIFwiQWNjZXNzLUNvbnRyb2wtQWxsb3ctTWV0aG9kc1wiOiB7XG4gICAgc2NoZW1hOiB7IHR5cGU6IFwic3RyaW5nXCIgfSxcbiAgfSxcbiAgXCJBY2Nlc3MtQ29udHJvbC1BbGxvdy1IZWFkZXJzXCI6IHtcbiAgICBzY2hlbWE6IHsgdHlwZTogXCJzdHJpbmdcIiB9LFxuICB9LFxufSk7XG5cbmNvbnN0IGdlbmVyYXRlQ29yc1Jlc3BvbnNlSGVhZGVycyA9IChcbiAgY29yc09wdGlvbnM6IFNlcmlhbGl6ZWRDb3JzT3B0aW9uc1xuKTogeyBba2V5OiBzdHJpbmddOiBzdHJpbmcgfSA9PiAoe1xuICBcIkFjY2Vzcy1Db250cm9sLUFsbG93LUhlYWRlcnNcIjogYCcke2NvcnNPcHRpb25zLmFsbG93SGVhZGVycy5qb2luKFwiLFwiKX0nYCxcbiAgXCJBY2Nlc3MtQ29udHJvbC1BbGxvdy1NZXRob2RzXCI6IGAnJHtjb3JzT3B0aW9ucy5hbGxvd01ldGhvZHMuam9pbihcIixcIil9J2AsXG4gIFwiQWNjZXNzLUNvbnRyb2wtQWxsb3ctT3JpZ2luXCI6IGAnJHtjb3JzT3B0aW9ucy5hbGxvd09yaWdpbnMuam9pbihcIixcIil9J2AsXG59KTtcblxuZXhwb3J0IGNvbnN0IGdlbmVyYXRlQ29yc1Jlc3BvbnNlUGFyYW1ldGVycyA9IChcbiAgY29yc09wdGlvbnM6IFNlcmlhbGl6ZWRDb3JzT3B0aW9ucyxcbiAgcHJlZml4OiBzdHJpbmcgPSBcIm1ldGhvZC5yZXNwb25zZS5oZWFkZXJcIlxuKTogeyBba2V5OiBzdHJpbmddOiBzdHJpbmcgfSA9PlxuICBPYmplY3QuZnJvbUVudHJpZXMoXG4gICAgT2JqZWN0LmVudHJpZXMoZ2VuZXJhdGVDb3JzUmVzcG9uc2VIZWFkZXJzKGNvcnNPcHRpb25zKSkubWFwKFxuICAgICAgKFtoZWFkZXIsIHZhbHVlXSkgPT4gW2Ake3ByZWZpeH0uJHtoZWFkZXJ9YCwgdmFsdWVdXG4gICAgKVxuICApO1xuXG4vKipcbiAqIEdlbmVyYXRlcyBhbiBcIm9wdGlvbnNcIiBtZXRob2Qgd2l0aCBubyBhdXRoIHRvIHJlc3BvbmQgd2l0aCB0aGUgYXBwcm9wcmlhdGUgaGVhZGVycyBpZiBjb3JzIGlzIGVuYWJsZWRcbiAqL1xuY29uc3QgZ2VuZXJhdGVDb3JzT3B0aW9uc01ldGhvZCA9IChcbiAgcGF0aEl0ZW06IE9wZW5BUElWMy5QYXRoSXRlbU9iamVjdCxcbiAgeyBjb3JzT3B0aW9ucyB9OiBQcmVwYXJlQXBpU3BlY09wdGlvbnNcbik6IE9wZW5BUElWMy5QYXRoSXRlbU9iamVjdCA9PiB7XG4gIC8vIERvIG5vdCBnZW5lcmF0ZSBpZiBhbHJlYWR5IG1hbnVhbGx5IGRlZmluZWQsIG9yIGNvcnMgbm90IGVuYWJsZWRcbiAgaWYgKEh0dHBNZXRob2RzLk9QVElPTlMgaW4gcGF0aEl0ZW0gfHwgIWNvcnNPcHRpb25zKSB7XG4gICAgcmV0dXJuIHt9O1xuICB9XG5cbiAgY29uc3Qgc3RhdHVzQ29kZSA9IGNvcnNPcHRpb25zLnN0YXR1c0NvZGU7XG5cbiAgcmV0dXJuIHtcbiAgICBbSHR0cE1ldGhvZHMuT1BUSU9OU106IHtcbiAgICAgIHN1bW1hcnk6IFwiQ09SUyBTdXBwb3J0XCIsXG4gICAgICBkZXNjcmlwdGlvbjogXCJFbmFibGUgQ09SUyBieSByZXR1cm5pbmcgdGhlIGNvcnJlY3QgaGVhZGVyc1wiLFxuICAgICAgcmVzcG9uc2VzOiB7XG4gICAgICAgIFtgJHtzdGF0dXNDb2RlfWBdOiB7XG4gICAgICAgICAgZGVzY3JpcHRpb246IFwiRGVmYXVsdCByZXNwb25zZSBmb3IgQ09SUyBtZXRob2RcIixcbiAgICAgICAgICBoZWFkZXJzOiBnZXRDb3JzSGVhZGVyRGVmaW5pdGlvbnMoKSxcbiAgICAgICAgICBjb250ZW50OiB7fSxcbiAgICAgICAgfSxcbiAgICAgIH0sXG4gICAgICAvLyBAdHMtaWdub3JlIElnbm9yZSBhcGlnYXRld2F5IGV4dGVuc2lvbnMgd2hpY2ggYXJlIG5vdCBwYXJ0IG9mIGRlZmF1bHQgb3BlbmFwaSBzcGVjIHR5cGVcbiAgICAgIFwieC1hbWF6b24tYXBpZ2F0ZXdheS1pbnRlZ3JhdGlvblwiOiB7XG4gICAgICAgIHR5cGU6IFwibW9ja1wiLFxuICAgICAgICByZXF1ZXN0VGVtcGxhdGVzOiB7XG4gICAgICAgICAgXCJhcHBsaWNhdGlvbi9qc29uXCI6IGB7XCJzdGF0dXNDb2RlXCI6ICR7c3RhdHVzQ29kZX19YCxcbiAgICAgICAgfSxcbiAgICAgICAgcmVzcG9uc2VzOiB7XG4gICAgICAgICAgZGVmYXVsdDoge1xuICAgICAgICAgICAgc3RhdHVzQ29kZTogYCR7c3RhdHVzQ29kZX1gLFxuICAgICAgICAgICAgcmVzcG9uc2VQYXJhbWV0ZXJzOiBnZW5lcmF0ZUNvcnNSZXNwb25zZVBhcmFtZXRlcnMoY29yc09wdGlvbnMpLFxuICAgICAgICAgICAgcmVzcG9uc2VUZW1wbGF0ZXM6IHtcbiAgICAgICAgICAgICAgXCJhcHBsaWNhdGlvbi9qc29uXCI6IFwie31cIixcbiAgICAgICAgICAgIH0sXG4gICAgICAgICAgfSxcbiAgICAgICAgfSxcbiAgICAgIH0sXG4gICAgICAvLyBObyBhdXRoIGZvciBDT1JTIG9wdGlvbnMgcmVxdWVzdHNcbiAgICAgIC4uLm5vQXV0aFNwZWNTbmlwcGV0KCksXG4gICAgfSxcbiAgfTtcbn07XG5cbi8qKlxuICogUHJlcGFyZXMgYSBnaXZlbiBhcGkgcGF0aCBieSBhZGRpbmcgaW50ZWdyYXRpb25zLCBjb25maWd1cmluZyBhdXRoXG4gKi9cbmNvbnN0IHByZXBhcmVQYXRoU3BlYyA9IChcbiAgcGF0aDogc3RyaW5nLFxuICBwYXRoSXRlbTogT3BlbkFQSVYzLlBhdGhJdGVtT2JqZWN0LFxuICBvcHRpb25zOiBQcmVwYXJlQXBpU3BlY09wdGlvbnMsXG4gIGdldE9wZXJhdGlvbk5hbWU6IChtZXRob2RBbmRQYXRoOiBNZXRob2RBbmRQYXRoKSA9PiBzdHJpbmdcbik6IE9wZW5BUElWMy5QYXRoSXRlbU9iamVjdCA9PiB7XG4gIGNvbnN0IHN1cHBvcnRlZEh0dHBNZXRob2RzID0gbmV3IFNldDxzdHJpbmc+KE9iamVjdC52YWx1ZXMoSHR0cE1ldGhvZHMpKTtcbiAgY29uc3QgdW5zdXBwb3J0ZWRNZXRob2RzSW5TcGVjID0gT2JqZWN0LmtleXMocGF0aEl0ZW0pLmZpbHRlcihcbiAgICAobWV0aG9kKSA9PiAhc3VwcG9ydGVkSHR0cE1ldGhvZHMuaGFzKG1ldGhvZClcbiAgKTtcbiAgaWYgKHVuc3VwcG9ydGVkTWV0aG9kc0luU3BlYy5sZW5ndGggPiAwKSB7XG4gICAgdGhyb3cgbmV3IEVycm9yKFxuICAgICAgYFBhdGggJHtwYXRofSBjb250YWlucyB1bnN1cHBvcnRlZCBtZXRob2Qke1xuICAgICAgICB1bnN1cHBvcnRlZE1ldGhvZHNJblNwZWMubGVuZ3RoID4gMSA/IFwic1wiIDogXCJcIlxuICAgICAgfSAke3Vuc3VwcG9ydGVkTWV0aG9kc0luU3BlYy5qb2luKFxuICAgICAgICBcIiwgXCJcbiAgICAgICl9LiBTdXBwb3J0ZWQgbWV0aG9kcyBhcmUgJHtPYmplY3QudmFsdWVzKEh0dHBNZXRob2RzKS5qb2luKFwiLCBcIil9LmBcbiAgICApO1xuICB9XG5cbiAgcmV0dXJuIHtcbiAgICAuLi5wYXRoSXRlbSxcbiAgICAuLi5PYmplY3QuZnJvbUVudHJpZXMoXG4gICAgICBPYmplY3QudmFsdWVzKEh0dHBNZXRob2RzKVxuICAgICAgICAuZmlsdGVyKChtZXRob2QpID0+IHBhdGhJdGVtW21ldGhvZF0pXG4gICAgICAgIC5tYXAoKG1ldGhvZCkgPT4gW1xuICAgICAgICAgIG1ldGhvZCxcbiAgICAgICAgICBhcHBseU1ldGhvZEludGVncmF0aW9uKFxuICAgICAgICAgICAgcGF0aCxcbiAgICAgICAgICAgIG1ldGhvZCxcbiAgICAgICAgICAgIG9wdGlvbnMsXG4gICAgICAgICAgICBwYXRoSXRlbVttZXRob2RdISxcbiAgICAgICAgICAgIGdldE9wZXJhdGlvbk5hbWVcbiAgICAgICAgICApLFxuICAgICAgICBdKVxuICAgICksXG4gICAgLy8gR2VuZXJhdGUgYW4gJ29wdGlvbnMnIG1ldGhvZCByZXF1aXJlZCBmb3IgQ09SUyBwcmVmbGlnaHQgcmVxdWVzdHMgaWYgY29ycyBpcyBlbmFibGVkXG4gICAgLi4uZ2VuZXJhdGVDb3JzT3B0aW9uc01ldGhvZChwYXRoSXRlbSwgb3B0aW9ucyksXG4gIH07XG59O1xuXG4vKipcbiAqIFJldHVybiB3aGV0aGVyIHRoZSBnaXZlbiBPcGVuQVBJIG9iamVjdCBpcyBhIHJlZmVyZW5jZSBvYmplY3RcbiAqL1xuY29uc3QgaXNSZWYgPSAob2JqOiBhbnkpOiBvYmogaXMgT3BlbkFQSVYzLlJlZmVyZW5jZU9iamVjdCA9PiBcIiRyZWZcIiBpbiBvYmo7XG5cbi8qKlxuICogVmFsaWRhdGUgdGhlIGNvbnN0cnVjdCBzZWN1cml0eSBzY2hlbWVzIGFnYWluc3QgdGhlIHNlY3VyaXR5IHNjaGVtZXMgaW4gdGhlIG9yaWdpbmFsIHNwZWMuXG4gKiBDb25zdHJ1Y3QtZGVmaW5lZCBhdXRob3JpemVycyBhbHdheXMgb3ZlcnJpZGUgdGhvc2UgaW4gdGhlIHNwZWMgaWYgdGhleSBoYXZlIHRoZSBzYW1lIElELCBob3dldmVyIHdlIHZhbGlkYXRlIHRoYXRcbiAqIHdlIGFyZSBub3Qgb3ZlcnJpZGluZyBhbiBhdXRob3JpemVyIG9mIGEgZGlmZmVyZW50IHR5cGUgdG8gYXZvaWQgbWlzdGFrZXMvbWlzbWF0Y2hlcyBiZXR3ZWVuIHRoZSBzcGVjIGFuZCB0aGVcbiAqIGNvbnN0cnVjdC5cbiAqIEBwYXJhbSBjb25zdHJ1Y3RTZWN1cml0eVNjaGVtZXMgc2VjdXJpdHkgc2NoZW1lcyBnZW5lcmF0ZWQgZnJvbSB0aGUgY29uc3RydWN0IGF1dGhvcml6ZXJzXG4gKiBAcGFyYW0gZXhpc3RpbmdTcGVjU2VjdXJpdHlTY2hlbWVzIHNlY3VyaXR5IHNjaGVtZXMgYWxyZWFkeSBkZWZpbmVkIGluIHRoZSBzcGVjXG4gKi9cbmNvbnN0IHZhbGlkYXRlU2VjdXJpdHlTY2hlbWVzID0gKFxuICBjb25zdHJ1Y3RTZWN1cml0eVNjaGVtZXM6IHsgW2tleTogc3RyaW5nXTogT3BlbkFQSVYzLlNlY3VyaXR5U2NoZW1lT2JqZWN0IH0sXG4gIGV4aXN0aW5nU3BlY1NlY3VyaXR5U2NoZW1lcz86IHtcbiAgICBba2V5OiBzdHJpbmddOiBPcGVuQVBJVjMuU2VjdXJpdHlTY2hlbWVPYmplY3QgfCBPcGVuQVBJVjMuUmVmZXJlbmNlT2JqZWN0O1xuICB9XG4pID0+IHtcbiAgaWYgKGV4aXN0aW5nU3BlY1NlY3VyaXR5U2NoZW1lcykge1xuICAgIGNvbnN0IGNvbnN0cnVjdFNlY3VyaXR5U2NoZW1lSWRzID0gbmV3IFNldChcbiAgICAgIE9iamVjdC5rZXlzKGNvbnN0cnVjdFNlY3VyaXR5U2NoZW1lcylcbiAgICApO1xuICAgIGNvbnN0IGV4aXN0aW5nU2VjdXJpdHlTY2hlbWVJZHMgPSBuZXcgU2V0KFxuICAgICAgT2JqZWN0LmtleXMoZXhpc3RpbmdTcGVjU2VjdXJpdHlTY2hlbWVzKVxuICAgICk7XG5cbiAgICBjb25zdCBvdmVybGFwcGluZ1NlY3VyaXR5U2NoZW1lSWRzID0gWy4uLmNvbnN0cnVjdFNlY3VyaXR5U2NoZW1lSWRzXS5maWx0ZXIoXG4gICAgICAoaWQpID0+IGV4aXN0aW5nU2VjdXJpdHlTY2hlbWVJZHMuaGFzKGlkKVxuICAgICk7XG5cbiAgICAvLyBBbnkgb3ZlcmxhcHBpbmcgc2VjdXJpdHkgc2NoZW1lcyAoZGVmaW5lZCBpbiBib3RoIHRoZSBzcGVjIChvciBzb3VyY2Ugc21pdGh5IG1vZGVsKSBhbmQgdGhlIGNvbnN0cnVjdCkgbXVzdCBiZSBvZiB0aGUgc2FtZSB0eXBlLlxuICAgIC8vIFRoZSBvbmUgZGVmaW5lZCBpbiB0aGUgY29uc3RydWN0IHdpbGwgdGFrZSBwcmVjZWRlbmNlIHNpbmNlIGEgY3VzdG9tL2NvZ25pdG8gYXV0aG9yaXplciBjYW4gaGF2ZSBhIHJlc29sdmVkIGFybiBpbiB0aGUgY29uc3RydWN0LFxuICAgIC8vIGFuZCB3ZSBhbGxvdyB1c2FnZSBpbiB0aGUgbW9kZWwgYXMgYSBmb3J3YXJkIGRlZmluaXRpb24gd2l0aCBibGFuayBhcm4uXG4gICAgb3ZlcmxhcHBpbmdTZWN1cml0eVNjaGVtZUlkcy5mb3JFYWNoKChzY2hlbWVJZCkgPT4ge1xuICAgICAgaWYgKCFpc1JlZihleGlzdGluZ1NwZWNTZWN1cml0eVNjaGVtZXNbc2NoZW1lSWRdKSkge1xuICAgICAgICBjb25zdCBleGlzdGluZ1NjaGVtZSA9IGV4aXN0aW5nU3BlY1NlY3VyaXR5U2NoZW1lc1tcbiAgICAgICAgICBzY2hlbWVJZFxuICAgICAgICBdIGFzIE9wZW5BUElWMy5TZWN1cml0eVNjaGVtZU9iamVjdDtcblxuICAgICAgICBpZiAoY29uc3RydWN0U2VjdXJpdHlTY2hlbWVzW3NjaGVtZUlkXS50eXBlICE9PSBleGlzdGluZ1NjaGVtZS50eXBlKSB7XG4gICAgICAgICAgdGhyb3cgbmV3IEVycm9yKFxuICAgICAgICAgICAgYFNlY3VyaXR5IHNjaGVtZSB3aXRoIGlkICR7c2NoZW1lSWR9IHdhcyBvZiB0eXBlICR7Y29uc3RydWN0U2VjdXJpdHlTY2hlbWVzW3NjaGVtZUlkXS50eXBlfSBpbiBjb25zdHJ1Y3QgYnV0ICR7ZXhpc3RpbmdTY2hlbWUudHlwZX0gaW4gT3BlbkFQSSBzcGVjIG9yIFNtaXRoeSBtb2RlbC5gXG4gICAgICAgICAgKTtcbiAgICAgICAgfVxuICAgICAgICBjb25zdCBjb25zdHJ1Y3RBcGlHYXRld2F5QXV0aFR5cGUgPSAoXG4gICAgICAgICAgY29uc3RydWN0U2VjdXJpdHlTY2hlbWVzW3NjaGVtZUlkXSBhcyBhbnlcbiAgICAgICAgKVtcIngtYW1hem9uLWFwaWdhdGV3YXktYXV0aHR5cGVcIl07XG4gICAgICAgIGNvbnN0IGV4aXN0aW5nQXBpR2F0ZXdheUF1dGhUeXBlID0gKGV4aXN0aW5nU2NoZW1lIGFzIGFueSlbXG4gICAgICAgICAgXCJ4LWFtYXpvbi1hcGlnYXRld2F5LWF1dGh0eXBlXCJcbiAgICAgICAgXTtcblxuICAgICAgICBpZiAoY29uc3RydWN0QXBpR2F0ZXdheUF1dGhUeXBlICE9PSBleGlzdGluZ0FwaUdhdGV3YXlBdXRoVHlwZSkge1xuICAgICAgICAgIHRocm93IG5ldyBFcnJvcihcbiAgICAgICAgICAgIGBTZWN1cml0eSBzY2hlbWUgd2l0aCBpZCAke3NjaGVtZUlkfSB3YXMgb2YgdHlwZSAke2NvbnN0cnVjdEFwaUdhdGV3YXlBdXRoVHlwZX0gaW4gY29uc3RydWN0IGJ1dCAke2V4aXN0aW5nQXBpR2F0ZXdheUF1dGhUeXBlfSBpbiBPcGVuQVBJIHNwZWMgb3IgU21pdGh5IG1vZGVsLmBcbiAgICAgICAgICApO1xuICAgICAgICB9XG4gICAgICB9IGVsc2Uge1xuICAgICAgICB0aHJvdyBuZXcgRXJyb3IoXG4gICAgICAgICAgYFNlY3VyaXR5IHNjaGVtZSB3aXRoIGlkICR7c2NoZW1lSWR9IGlzIGEgcmVmZXJlbmNlIGluIHRoZSBPcGVuQVBJIHNwZWMgb3IgU21pdGh5IG1vZGVsIHdoaWNoIGlzIG5vdCBzdXBwb3J0ZWQuYFxuICAgICAgICApO1xuICAgICAgfVxuICAgIH0pO1xuICB9XG59O1xuXG4vKipcbiAqIFZhbGlkYXRlIHRoZSBnaXZlbiBhdXRob3JpemVyIHJlZmVyZW5jZSAoZWl0aGVyIGRlZmF1bHQgb3IgYXQgYW4gb3BlcmF0aW9uIGxldmVsKSBkZWZpbmVkIGluIHRoZSBjb25zdHJ1Y3QgYWdhaW5zdFxuICogdGhvc2UgYWxyZWFkeSBpbiB0aGUgc3BlYy5cbiAqIEBwYXJhbSBjb25zdHJ1Y3RBdXRob3JpemVyIHRoZSBhdXRob3JpemVyIGRlZmluZWQgaW4gdGhlIGNvbnN0cnVjdFxuICogQHBhcmFtIGV4aXN0aW5nU3BlY0F1dGhvcml6ZXJzIHRoZSBhdXRob3JpemVycyBhbHJlYWR5IGRlZmluZWQgaW4gdGhlIHNwZWNcbiAqIEBwYXJhbSBvcGVyYXRpb24gdGhlIG9wZXJhdGlvbiB3ZSBhcmUgdmFsaWRhdGluZyAoZm9yIGNsZWFyZXIgZXJyb3IgbWVzc2FnZXMpXG4gKi9cbmNvbnN0IHZhbGlkYXRlQXV0aG9yaXplclJlZmVyZW5jZSA9IChcbiAgY29uc3RydWN0QXV0aG9yaXplcj86IFNlcmlhbGlzZWRBdXRob3JpemVyUmVmZXJlbmNlLFxuICBleGlzdGluZ1NwZWNBdXRob3JpemVycz86IE9wZW5BUElWMy5TZWN1cml0eVJlcXVpcmVtZW50T2JqZWN0W10sXG4gIG9wZXJhdGlvbjogc3RyaW5nID0gXCJEZWZhdWx0XCJcbikgPT4ge1xuICAvLyBPbmx5IG5lZWQgdG8gdmFsaWRhdGUgaWYgZGVmaW5lZCBpbiBib3RoIC0gaWYganVzdCBvbmUgd2UnbGwgdXNlIHRoYXQuXG4gIGlmIChjb25zdHJ1Y3RBdXRob3JpemVyICYmIGV4aXN0aW5nU3BlY0F1dGhvcml6ZXJzKSB7XG4gICAgY29uc3QgbWVyZ2VkU3BlY0F1dGhvcml6ZXJzID0gT2JqZWN0LmZyb21FbnRyaWVzKFxuICAgICAgZXhpc3RpbmdTcGVjQXV0aG9yaXplcnMuZmxhdE1hcCgoc2VjdXJpdHlSZXF1aXJlbWVudCkgPT5cbiAgICAgICAgT2JqZWN0LmtleXMoc2VjdXJpdHlSZXF1aXJlbWVudCkubWFwKChpZCkgPT4gW1xuICAgICAgICAgIGlkLFxuICAgICAgICAgIHNlY3VyaXR5UmVxdWlyZW1lbnRbaWRdLFxuICAgICAgICBdKVxuICAgICAgKVxuICAgICk7XG4gICAgY29uc3Qgc3BlY0F1dGhvcml6ZXJJZHMgPSBPYmplY3Qua2V5cyhtZXJnZWRTcGVjQXV0aG9yaXplcnMpO1xuXG4gICAgaWYgKHNwZWNBdXRob3JpemVySWRzLmxlbmd0aCA+IDEpIHtcbiAgICAgIC8vIFNwZWMgZGVmaW5lZCBtdWx0aXBsZSBhdXRob3JpemVycyBidXQgdGhlIGNvbnN0cnVjdCBjYW4gb25seSBzcGVjaWZ5IG9uZVxuICAgICAgdGhyb3cgbmV3IEVycm9yKFxuICAgICAgICBgJHtvcGVyYXRpb259IGF1dGhvcml6ZXJzICR7c3BlY0F1dGhvcml6ZXJJZHNcbiAgICAgICAgICAuc29ydCgpXG4gICAgICAgICAgLmpvaW4oXG4gICAgICAgICAgICBcIiwgXCJcbiAgICAgICAgICApfSBkZWZpbmVkIGluIHRoZSBPcGVuQVBJIFNwZWMgb3IgU21pdGh5IE1vZGVsIHdvdWxkIGJlIG92ZXJyaWRkZW4gYnkgc2luZ2xlIGNvbnN0cnVjdCBhdXRob3JpemVyICR7XG4gICAgICAgICAgY29uc3RydWN0QXV0aG9yaXplci5hdXRob3JpemVySWRcbiAgICAgICAgfWBcbiAgICAgICk7XG4gICAgfSBlbHNlIGlmIChzcGVjQXV0aG9yaXplcklkcy5sZW5ndGggPT09IDEpIHtcbiAgICAgIC8vIFNpbmdsZSBhdXRob3JpemVyIC0gY2hlY2sgdGhhdCB0aGV5IGhhdmUgdGhlIHNhbWUgaWRcbiAgICAgIGlmIChzcGVjQXV0aG9yaXplcklkc1swXSAhPT0gY29uc3RydWN0QXV0aG9yaXplci5hdXRob3JpemVySWQpIHtcbiAgICAgICAgdGhyb3cgbmV3IEVycm9yKFxuICAgICAgICAgIGAke29wZXJhdGlvbn0gYXV0aG9yaXplciAke3NwZWNBdXRob3JpemVySWRzWzBdfSBkZWZpbmVkIGluIHRoZSBPcGVuQVBJIFNwZWMgb3IgU21pdGh5IE1vZGVsIHdvdWxkIGJlIG92ZXJyaWRkZW4gYnkgY29uc3RydWN0IGF1dGhvcml6ZXIgJHtjb25zdHJ1Y3RBdXRob3JpemVyLmF1dGhvcml6ZXJJZH1gXG4gICAgICAgICk7XG4gICAgICB9XG5cbiAgICAgIC8vIENoZWNrIHRoYXQgdGhlcmUgYXJlIG5vIGRpZmZlcmluZyBzY29wZXMgYmV0d2VlbiB0aGUgY29uc3RydWN0IGFuZCBzcGVjXG4gICAgICBjb25zdCBzcGVjU2NvcGVzID0gbmV3IFNldChtZXJnZWRTcGVjQXV0aG9yaXplcnNbc3BlY0F1dGhvcml6ZXJJZHNbMF1dKTtcbiAgICAgIGNvbnN0IGNvbnN0cnVjdFNjb3BlcyA9IG5ldyBTZXQoY29uc3RydWN0QXV0aG9yaXplci5hdXRob3JpemF0aW9uU2NvcGVzKTtcbiAgICAgIGNvbnN0IGRpZmZlcmluZ1Njb3BlcyA9IFtcbiAgICAgICAgLi4uWy4uLnNwZWNTY29wZXNdLmZpbHRlcigoc2NvcGUpID0+ICFjb25zdHJ1Y3RTY29wZXMuaGFzKHNjb3BlKSksXG4gICAgICAgIC4uLlsuLi5jb25zdHJ1Y3RTY29wZXNdLmZpbHRlcigoc2NvcGUpID0+ICFzcGVjU2NvcGVzLmhhcyhzY29wZSkpLFxuICAgICAgXTtcbiAgICAgIGlmIChkaWZmZXJpbmdTY29wZXMubGVuZ3RoID4gMCkge1xuICAgICAgICB0aHJvdyBuZXcgRXJyb3IoXG4gICAgICAgICAgYCR7b3BlcmF0aW9ufSBhdXRob3JpemVyIHNjb3BlcyAke1suLi5zcGVjU2NvcGVzXS5qb2luKFxuICAgICAgICAgICAgXCIsIFwiXG4gICAgICAgICAgKX0gZGVmaW5lZCBpbiB0aGUgT3BlbkFQSSBTcGVjIG9yIFNtaXRoeSBNb2RlbCBkaWZmZXIgZnJvbSB0aG9zZSBpbiB0aGUgY29uc3RydWN0ICgke1tcbiAgICAgICAgICAgIC4uLmNvbnN0cnVjdFNjb3BlcyxcbiAgICAgICAgICBdLmpvaW4oXCIsIFwiKX0pYFxuICAgICAgICApO1xuICAgICAgfVxuICAgIH0gZWxzZSBpZiAoY29uc3RydWN0QXV0aG9yaXplci5hdXRob3JpemVySWQgIT09IERlZmF1bHRBdXRob3JpemVySWRzLk5PTkUpIHtcbiAgICAgIC8vIFwic2VjdXJpdHlcIiBzZWN0aW9uIG9mIHNwZWMgaXMgW10gd2hpY2ggbWVhbnMgbm8gYXV0aCwgYnV0IHRoZSBhdXRob3JpemVyIGluIHRoZSBjb25zdHJ1Y3QgaXMgbm90IHRoZSBcIm5vbmVcIiBhdXRob3JpemVyLlxuICAgICAgdGhyb3cgbmV3IEVycm9yKFxuICAgICAgICBgJHtvcGVyYXRpb259IGV4cGxpY2l0bHkgZGVmaW5lcyBubyBhdXRoIGluIHRoZSBPcGVuQVBJIFNwZWMgb3IgU21pdGh5IE1vZGVsIHdoaWNoIHdvdWxkIGJlIG92ZXJyaWRkZW4gYnkgY29uc3RydWN0IGF1dGhvcml6ZXIgJHtjb25zdHJ1Y3RBdXRob3JpemVyLmF1dGhvcml6ZXJJZH1gXG4gICAgICApO1xuICAgIH1cbiAgfVxufTtcblxuLyoqXG4gKiBGaW5kIGFsbCB1bmlxdWUgaGVhZGVyIHBhcmFtZXRlcnMgdXNlZCBpbiBvcGVyYXRpb25zXG4gKi9cbmNvbnN0IGZpbmRIZWFkZXJQYXJhbWV0ZXJzID0gKHNwZWM6IE9wZW5BUElWMy5Eb2N1bWVudCk6IHN0cmluZ1tdID0+IHtcbiAgY29uc3QgYWxsSGVhZGVyUGFyYW1ldGVycyA9IE9iamVjdC52YWx1ZXMoc3BlYy5wYXRocykuZmxhdE1hcCgocGF0aERldGFpbHMpID0+XG4gICAgT2JqZWN0LnZhbHVlcyhIdHRwTWV0aG9kcykuZmxhdE1hcCgobWV0aG9kKSA9PlxuICAgICAgKHBhdGhEZXRhaWxzPy5bbWV0aG9kXT8ucGFyYW1ldGVycyA/PyBbXSkuZmxhdE1hcCgocGFyYW1ldGVyKSA9PlxuICAgICAgICBcImluXCIgaW4gcGFyYW1ldGVyICYmIHBhcmFtZXRlci5pbiA9PT0gXCJoZWFkZXJcIiA/IFtwYXJhbWV0ZXIubmFtZV0gOiBbXVxuICAgICAgKVxuICAgIClcbiAgKTtcbiAgY29uc3QgaGVhZGVyUGFyYW1ldGVyU2V0ID0gbmV3IFNldDxzdHJpbmc+KCk7XG4gIHJldHVybiBhbGxIZWFkZXJQYXJhbWV0ZXJzLmZpbHRlcigocCkgPT4ge1xuICAgIGNvbnN0IHNlZW4gPSBoZWFkZXJQYXJhbWV0ZXJTZXQuaGFzKHApO1xuICAgIGhlYWRlclBhcmFtZXRlclNldC5hZGQocCk7XG4gICAgcmV0dXJuICFzZWVuO1xuICB9KTtcbn07XG5cbi8qKlxuICogUHJlcGFyZXMgdGhlIGFwaSBzcGVjIGZvciBkZXBsb3ltZW50IGJ5IGFkZGluZyBpbnRlZ3JhdGlvbnMsIGNvbmZpZ3VyaW5nIGF1dGgsIGV0Y1xuICovXG5leHBvcnQgY29uc3QgcHJlcGFyZUFwaVNwZWMgPSAoXG4gIHNwZWM6IE9wZW5BUElWMy5Eb2N1bWVudCxcbiAgb3B0aW9uczogUHJlcGFyZUFwaVNwZWNPcHRpb25zXG4pOiBPcGVuQVBJVjMuRG9jdW1lbnQgPT4ge1xuICAvLyBSZXZlcnNlIGxvb2t1cCBmb3IgdGhlIG9wZXJhdGlvbiBuYW1lIGdpdmVuIGEgbWV0aG9kIGFuZCBwYXRoXG4gIGNvbnN0IG9wZXJhdGlvbk5hbWVCeVBhdGggPSBPYmplY3QuZnJvbUVudHJpZXMoXG4gICAgT2JqZWN0LmVudHJpZXM8TWV0aG9kQW5kUGF0aD4ob3B0aW9ucy5vcGVyYXRpb25Mb29rdXApLm1hcChcbiAgICAgIChbb3BlcmF0aW9uTmFtZSwgbWV0aG9kQW5kUGF0aF0pID0+IFtcbiAgICAgICAgY29uY2F0TWV0aG9kQW5kUGF0aChtZXRob2RBbmRQYXRoKSxcbiAgICAgICAgb3BlcmF0aW9uTmFtZSxcbiAgICAgIF1cbiAgICApXG4gICk7XG4gIGNvbnN0IGdldE9wZXJhdGlvbk5hbWUgPSAobWV0aG9kQW5kUGF0aDogTWV0aG9kQW5kUGF0aCkgPT5cbiAgICBvcGVyYXRpb25OYW1lQnlQYXRoW2NvbmNhdE1ldGhvZEFuZFBhdGgobWV0aG9kQW5kUGF0aCldO1xuXG4gIHZhbGlkYXRlU2VjdXJpdHlTY2hlbWVzKFxuICAgIG9wdGlvbnMuc2VjdXJpdHlTY2hlbWVzLFxuICAgIHNwZWMuY29tcG9uZW50cz8uc2VjdXJpdHlTY2hlbWVzXG4gICk7XG4gIHZhbGlkYXRlQXV0aG9yaXplclJlZmVyZW5jZShcbiAgICBvcHRpb25zLmRlZmF1bHRBdXRob3JpemVyUmVmZXJlbmNlLFxuICAgIHNwZWMuc2VjdXJpdHlcbiAgKTtcblxuICAvLyBJZiB0aGVyZSBhcmUgY29ycyBvcHRpb25zLCBhZGQgYW55IGhlYWRlciBwYXJhbWV0ZXJzIGRlZmluZWQgaW4gdGhlIHNwZWMgYXMgYWxsb3dlZCBoZWFkZXJzIHRvXG4gIC8vIHNhdmUgdXNlcnMgZnJvbSBoYXZpbmcgdG8gbWFudWFsbHkgc3BlY2lmeSB0aGVzZSAob3IgZmFjZSBjb3JzIGlzc3VlcyEpXG4gIGNvbnN0IGNvcnNPcHRpb25zOiBTZXJpYWxpemVkQ29yc09wdGlvbnMgfCB1bmRlZmluZWQgPSBvcHRpb25zLmNvcnNPcHRpb25zXG4gICAgPyB7XG4gICAgICAgIC4uLm9wdGlvbnMuY29yc09wdGlvbnMsXG4gICAgICAgIGFsbG93SGVhZGVyczogW1xuICAgICAgICAgIC4uLm9wdGlvbnMuY29yc09wdGlvbnMuYWxsb3dIZWFkZXJzLFxuICAgICAgICAgIC4uLmZpbmRIZWFkZXJQYXJhbWV0ZXJzKHNwZWMpLFxuICAgICAgICBdLFxuICAgICAgfVxuICAgIDogdW5kZWZpbmVkO1xuXG4gIGNvbnN0IHVwZGF0ZWRPcHRpb25zOiBQcmVwYXJlQXBpU3BlY09wdGlvbnMgPSB7XG4gICAgLi4ub3B0aW9ucyxcbiAgICBjb3JzT3B0aW9ucyxcbiAgfTtcblxuICByZXR1cm4ge1xuICAgIC4uLnNwZWMsXG4gICAgLy8gaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2FwaWdhdGV3YXkvbGF0ZXN0L2RldmVsb3Blcmd1aWRlL2FwaS1nYXRld2F5LXN3YWdnZXItZXh0ZW5zaW9ucy1yZXF1ZXN0LXZhbGlkYXRvcnMuaHRtbFxuICAgIFwieC1hbWF6b24tYXBpZ2F0ZXdheS1yZXF1ZXN0LXZhbGlkYXRvcnNcIjoge1xuICAgICAgYWxsOiB7XG4gICAgICAgIHZhbGlkYXRlUmVxdWVzdEJvZHk6IHRydWUsXG4gICAgICAgIHZhbGlkYXRlUmVxdWVzdFBhcmFtZXRlcnM6IHRydWUsXG4gICAgICB9LFxuICAgIH0sXG4gICAgXCJ4LWFtYXpvbi1hcGlnYXRld2F5LXJlcXVlc3QtdmFsaWRhdG9yXCI6IFwiYWxsXCIsXG4gICAgLy8gaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2FwaWdhdGV3YXkvbGF0ZXN0L2RldmVsb3Blcmd1aWRlL2FwaS1nYXRld2F5LXN3YWdnZXItZXh0ZW5zaW9ucy1nYXRld2F5LXJlc3BvbnNlcy5odG1sXG4gICAgXCJ4LWFtYXpvbi1hcGlnYXRld2F5LWdhdGV3YXktcmVzcG9uc2VzXCI6IHtcbiAgICAgIEJBRF9SRVFVRVNUX0JPRFk6IHtcbiAgICAgICAgc3RhdHVzQ29kZTogNDAwLFxuICAgICAgICByZXNwb25zZVRlbXBsYXRlczoge1xuICAgICAgICAgIFwiYXBwbGljYXRpb24vanNvblwiOlxuICAgICAgICAgICAgJ3tcIm1lc3NhZ2VcIjogXCIkY29udGV4dC5lcnJvci52YWxpZGF0aW9uRXJyb3JTdHJpbmdcIn0nLFxuICAgICAgICB9LFxuICAgICAgICAuLi4oY29yc09wdGlvbnNcbiAgICAgICAgICA/IHtcbiAgICAgICAgICAgICAgcmVzcG9uc2VQYXJhbWV0ZXJzOiBnZW5lcmF0ZUNvcnNSZXNwb25zZVBhcmFtZXRlcnMoXG4gICAgICAgICAgICAgICAgY29yc09wdGlvbnMsXG4gICAgICAgICAgICAgICAgXCJnYXRld2F5cmVzcG9uc2UuaGVhZGVyXCJcbiAgICAgICAgICAgICAgKSxcbiAgICAgICAgICAgIH1cbiAgICAgICAgICA6IHt9KSxcbiAgICAgIH0sXG4gICAgfSxcbiAgICBwYXRoczoge1xuICAgICAgLi4uT2JqZWN0LmZyb21FbnRyaWVzKFxuICAgICAgICBPYmplY3QuZW50cmllcyhzcGVjLnBhdGhzKS5tYXAoKFtwYXRoLCBwYXRoRGV0YWlsc10pID0+IFtcbiAgICAgICAgICBwYXRoLFxuICAgICAgICAgIHByZXBhcmVQYXRoU3BlYyhwYXRoLCBwYXRoRGV0YWlscyEsIHVwZGF0ZWRPcHRpb25zLCBnZXRPcGVyYXRpb25OYW1lKSxcbiAgICAgICAgXSlcbiAgICAgICksXG4gICAgfSxcbiAgICBjb21wb25lbnRzOiB7XG4gICAgICAuLi5zcGVjLmNvbXBvbmVudHMsXG4gICAgICBzZWN1cml0eVNjaGVtZXM6IHtcbiAgICAgICAgLy8gQXBwbHkgYW55IHNlY3VyaXR5IHNjaGVtZXMgdGhhdCBhbHJlYWR5IGV4aXN0IGluIHRoZSBzcGVjXG4gICAgICAgIC4uLnNwZWMuY29tcG9uZW50cz8uc2VjdXJpdHlTY2hlbWVzLFxuICAgICAgICAvLyBDb25zdHJ1Y3Qgc2VjdXJpdHkgc2NoZW1lcyBvdmVycmlkZSBhbnkgaW4gdGhlIHNwZWMgd2l0aCB0aGUgc2FtZSBpZFxuICAgICAgICAuLi51cGRhdGVkT3B0aW9ucy5zZWN1cml0eVNjaGVtZXMsXG4gICAgICB9LFxuICAgIH0sXG4gICAgLi4uKHVwZGF0ZWRPcHRpb25zLmFwaUtleU9wdGlvbnNcbiAgICAgID8ge1xuICAgICAgICAgIC8vIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9hcGlnYXRld2F5L2xhdGVzdC9kZXZlbG9wZXJndWlkZS9hcGktZ2F0ZXdheS1zd2FnZ2VyLWV4dGVuc2lvbnMtYXBpLWtleS1zb3VyY2UuaHRtbFxuICAgICAgICAgIFwieC1hbWF6b24tYXBpZ2F0ZXdheS1hcGkta2V5LXNvdXJjZVwiOlxuICAgICAgICAgICAgdXBkYXRlZE9wdGlvbnMuYXBpS2V5T3B0aW9ucy5zb3VyY2UsXG4gICAgICAgIH1cbiAgICAgIDoge30pLFxuICB9IGFzIGFueTtcbn07XG4iXX0=