"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.TypeSafeRestApi = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/*! Copyright [Amazon.com](http://amazon.com/), Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0 */
const fs = require("fs");
const path = require("path");
const pdk_nag_1 = require("../../pdk-nag");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_apigateway_1 = require("aws-cdk-lib/aws-apigateway");
const aws_iam_1 = require("aws-cdk-lib/aws-iam");
const aws_lambda_1 = require("aws-cdk-lib/aws-lambda");
const aws_logs_1 = require("aws-cdk-lib/aws-logs");
const aws_s3_assets_1 = require("aws-cdk-lib/aws-s3-assets");
const custom_resources_1 = require("aws-cdk-lib/custom-resources");
const cdk_nag_1 = require("cdk-nag");
const constructs_1 = require("constructs");
const prepare_spec_1 = require("./prepare-spec-event-handler/prepare-spec");
const api_gateway_auth_1 = require("./spec/api-gateway-auth");
const api_gateway_integrations_1 = require("./spec/api-gateway-integrations");
const open_api_gateway_web_acl_1 = require("./waf/open-api-gateway-web-acl");
/**
 * A construct for creating an api gateway rest api based on the definition in the OpenAPI spec.
 */
class TypeSafeRestApi extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        const { integrations, specPath, operationLookup, defaultAuthorizer, corsOptions, ...options } = props;
        // Upload the spec to s3 as an asset
        const inputSpecAsset = new aws_s3_assets_1.Asset(this, "InputSpec", {
            path: specPath,
        });
        // We'll output the prepared spec in the same asset bucket
        const preparedSpecOutputKeyPrefix = `${inputSpecAsset.s3ObjectKey}-prepared`;
        const stack = aws_cdk_lib_1.Stack.of(this);
        const prepareSpecLambdaName = `${pdk_nag_1.PDKNag.getStackPrefix(stack)
            .split("/")
            .join("-")}PrepareSpec${this.node.addr.slice(-8).toUpperCase()}`;
        const prepareSpecRole = new aws_iam_1.Role(this, "PrepareSpecRole", {
            assumedBy: new aws_iam_1.ServicePrincipal("lambda.amazonaws.com"),
            inlinePolicies: {
                logs: new aws_iam_1.PolicyDocument({
                    statements: [
                        new aws_iam_1.PolicyStatement({
                            effect: aws_iam_1.Effect.ALLOW,
                            actions: [
                                "logs:CreateLogGroup",
                                "logs:CreateLogStream",
                                "logs:PutLogEvents",
                            ],
                            resources: [
                                `arn:aws:logs:${stack.region}:${stack.account}:log-group:/aws/lambda/${prepareSpecLambdaName}`,
                                `arn:aws:logs:${stack.region}:${stack.account}:log-group:/aws/lambda/${prepareSpecLambdaName}:*`,
                            ],
                        }),
                    ],
                }),
                s3: new aws_iam_1.PolicyDocument({
                    statements: [
                        new aws_iam_1.PolicyStatement({
                            effect: aws_iam_1.Effect.ALLOW,
                            actions: ["s3:getObject"],
                            resources: [
                                inputSpecAsset.bucket.arnForObjects(inputSpecAsset.s3ObjectKey),
                            ],
                        }),
                        new aws_iam_1.PolicyStatement({
                            effect: aws_iam_1.Effect.ALLOW,
                            actions: ["s3:putObject"],
                            resources: [
                                // The output file will include a hash of the prepared spec, which is not known until deploy time since
                                // tokens must be resolved
                                inputSpecAsset.bucket.arnForObjects(`${preparedSpecOutputKeyPrefix}/*`),
                            ],
                        }),
                    ],
                }),
            },
        });
        ["AwsSolutions-IAM5", "AwsPrototyping-IAMNoWildcardPermissions"].forEach((RuleId) => {
            cdk_nag_1.NagSuppressions.addResourceSuppressions(prepareSpecRole, [
                {
                    id: RuleId,
                    reason: "Cloudwatch resources have been scoped down to the LogGroup level, however * is still needed as stream names are created just in time.",
                    appliesTo: [
                        {
                            regex: `/^Resource::arn:aws:logs:${pdk_nag_1.PDKNag.getStackRegionRegex(stack)}:${pdk_nag_1.PDKNag.getStackAccountRegex(stack)}:log-group:/aws/lambda/${prepareSpecLambdaName}:\*/g`,
                        },
                    ],
                },
                {
                    id: RuleId,
                    reason: "S3 resources have been scoped down to the appropriate prefix in the CDK asset bucket, however * is still needed as since the prepared spec hash is not known until deploy time.",
                    appliesTo: [
                        {
                            regex: `/^Resource::arn:${pdk_nag_1.PDKNag.getStackPartitionRegex(stack)}:s3:.*/${preparedSpecOutputKeyPrefix}/\*/g`,
                        },
                    ],
                },
            ], true);
        });
        // Create a custom resource for preparing the spec for deployment (adding integrations, authorizers, etc)
        const prepareSpec = new aws_lambda_1.Function(this, "PrepareSpec", {
            handler: "index.handler",
            runtime: aws_lambda_1.Runtime.NODEJS_18_X,
            code: aws_lambda_1.Code.fromAsset(path.join(__dirname, "./prepare-spec-event-handler")),
            timeout: aws_cdk_lib_1.Duration.seconds(30),
            role: prepareSpecRole,
            functionName: prepareSpecLambdaName,
        });
        const providerFunctionName = `${prepareSpecLambdaName}-Provider`;
        const providerRole = new aws_iam_1.Role(this, "PrepareSpecProviderRole", {
            assumedBy: new aws_iam_1.ServicePrincipal("lambda.amazonaws.com"),
            inlinePolicies: {
                logs: new aws_iam_1.PolicyDocument({
                    statements: [
                        new aws_iam_1.PolicyStatement({
                            effect: aws_iam_1.Effect.ALLOW,
                            actions: [
                                "logs:CreateLogGroup",
                                "logs:CreateLogStream",
                                "logs:PutLogEvents",
                            ],
                            resources: [
                                `arn:aws:logs:${stack.region}:${stack.account}:log-group:/aws/lambda/${providerFunctionName}`,
                                `arn:aws:logs:${stack.region}:${stack.account}:log-group:/aws/lambda/${providerFunctionName}:*`,
                            ],
                        }),
                    ],
                }),
            },
        });
        const provider = new custom_resources_1.Provider(this, "PrepareSpecResourceProvider", {
            onEventHandler: prepareSpec,
            role: providerRole,
            providerFunctionName,
        });
        ["AwsSolutions-IAM5", "AwsPrototyping-IAMNoWildcardPermissions"].forEach((RuleId) => {
            cdk_nag_1.NagSuppressions.addResourceSuppressions(providerRole, [
                {
                    id: RuleId,
                    reason: "Cloudwatch resources have been scoped down to the LogGroup level, however * is still needed as stream names are created just in time.",
                },
            ], true);
        });
        ["AwsSolutions-L1", "AwsPrototyping-LambdaLatestVersion"].forEach((RuleId) => {
            cdk_nag_1.NagSuppressions.addResourceSuppressions(provider, [
                {
                    id: RuleId,
                    reason: "Latest runtime cannot be configured. CDK will need to upgrade the Provider construct accordingly.",
                },
            ], true);
        });
        const serializedCorsOptions = corsOptions && {
            allowHeaders: corsOptions.allowHeaders || [
                ...aws_apigateway_1.Cors.DEFAULT_HEADERS,
                "x-amz-content-sha256",
            ],
            allowMethods: corsOptions.allowMethods || aws_apigateway_1.Cors.ALL_METHODS,
            allowOrigins: corsOptions.allowOrigins,
            statusCode: corsOptions.statusCode || 204,
        };
        const prepareSpecOptions = {
            defaultAuthorizerReference: (0, api_gateway_auth_1.serializeAsAuthorizerReference)(defaultAuthorizer),
            integrations: Object.fromEntries(Object.entries(integrations).map(([operationId, integration]) => [
                operationId,
                {
                    integration: integration.integration.render({
                        operationId,
                        scope: this,
                        ...operationLookup[operationId],
                        corsOptions: serializedCorsOptions,
                        operationLookup,
                    }),
                    methodAuthorizer: (0, api_gateway_auth_1.serializeAsAuthorizerReference)(integration.authorizer),
                    options: integration.options,
                },
            ])),
            securitySchemes: (0, api_gateway_auth_1.prepareSecuritySchemes)(this, integrations, defaultAuthorizer, options.apiKeyOptions),
            corsOptions: serializedCorsOptions,
            operationLookup,
            apiKeyOptions: options.apiKeyOptions,
        };
        // Spec preparation will happen in a custom resource lambda so that references to lambda integrations etc can be
        // resolved. However, we also prepare inline to perform some additional validation at synth time.
        const spec = JSON.parse(fs.readFileSync(specPath, "utf-8"));
        this.extendedApiSpecification = (0, prepare_spec_1.prepareApiSpec)(spec, prepareSpecOptions);
        const prepareApiSpecCustomResourceProperties = {
            inputSpecLocation: {
                bucket: inputSpecAsset.bucket.bucketName,
                key: inputSpecAsset.s3ObjectKey,
            },
            outputSpecLocation: {
                bucket: inputSpecAsset.bucket.bucketName,
                key: preparedSpecOutputKeyPrefix,
            },
            ...prepareSpecOptions,
        };
        const prepareSpecCustomResource = new aws_cdk_lib_1.CustomResource(this, "PrepareSpecResource", {
            serviceToken: provider.serviceToken,
            properties: prepareApiSpecCustomResourceProperties,
        });
        // Create the api gateway resources from the spec, augmenting the spec with the properties specific to api gateway
        // such as integrations or auth types
        this.api = new aws_apigateway_1.SpecRestApi(this, id, {
            apiDefinition: this.node.tryGetContext("type-safe-api-local")
                ? aws_apigateway_1.ApiDefinition.fromInline(this.extendedApiSpecification)
                : aws_apigateway_1.ApiDefinition.fromBucket(inputSpecAsset.bucket, prepareSpecCustomResource.getAttString("outputSpecKey")),
            deployOptions: {
                accessLogDestination: new aws_apigateway_1.LogGroupLogDestination(new aws_logs_1.LogGroup(this, `AccessLogs`)),
                accessLogFormat: aws_apigateway_1.AccessLogFormat.clf(),
                loggingLevel: aws_apigateway_1.MethodLoggingLevel.INFO,
            },
            ...options,
        });
        this.api.node.addDependency(prepareSpecCustomResource);
        // While the api will be updated when the output path from the custom resource changes, CDK still needs to know when
        // to redeploy the api. This is achieved by including a hash of the spec in the logical id (internalised in the
        // addToLogicalId method since this is how changes of individual resources/methods etc trigger redeployments in CDK)
        this.api.latestDeployment?.addToLogicalId(this.extendedApiSpecification);
        // Grant API Gateway permission to invoke the integrations
        Object.keys(integrations).forEach((operationId) => {
            integrations[operationId].integration.grant({
                operationId,
                scope: this,
                api: this.api,
                ...operationLookup[operationId],
                operationLookup,
            });
        });
        // Grant API Gateway permission to invoke each custom authorizer lambda (if any)
        (0, api_gateway_integrations_1.getAuthorizerFunctions)(props).forEach(({ label, function: lambda }) => {
            new aws_lambda_1.CfnPermission(this, `LambdaPermission-${label}`, {
                action: "lambda:InvokeFunction",
                principal: "apigateway.amazonaws.com",
                functionName: lambda.functionArn,
                sourceArn: stack.formatArn({
                    service: "execute-api",
                    resource: this.api.restApiId,
                    resourceName: "*/*",
                }),
            });
        });
        // Create and associate the web acl if not disabled
        if (!props.webAclOptions?.disable) {
            const acl = new open_api_gateway_web_acl_1.OpenApiGatewayWebAcl(this, `${id}-Acl`, {
                ...props.webAclOptions,
                apiDeploymentStageArn: this.api.deploymentStage.stageArn,
            });
            this.webAcl = acl.webAcl;
            this.ipSet = acl.ipSet;
            this.webAclAssociation = acl.webAclAssociation;
        }
        ["AwsSolutions-IAM4", "AwsPrototyping-IAMNoManagedPolicies"].forEach((RuleId) => {
            cdk_nag_1.NagSuppressions.addResourceSuppressions(this, [
                {
                    id: RuleId,
                    reason: "Cloudwatch Role requires access to create/read groups at the root level.",
                    appliesTo: [
                        {
                            regex: `/^Policy::arn:${pdk_nag_1.PDKNag.getStackPartitionRegex(stack)}:iam::aws:policy/service-role/AmazonAPIGatewayPushToCloudWatchLogs$/g`,
                        },
                    ],
                },
            ], true);
        });
        ["AwsSolutions-APIG2", "AwsPrototyping-APIGWRequestValidation"].forEach((RuleId) => {
            cdk_nag_1.NagSuppressions.addResourceSuppressions(this, [
                {
                    id: RuleId,
                    reason: "This construct implements fine grained validation via OpenApi.",
                },
            ], true);
        });
    }
}
exports.TypeSafeRestApi = TypeSafeRestApi;
_a = JSII_RTTI_SYMBOL_1;
TypeSafeRestApi[_a] = { fqn: "@aws/pdk.type_safe_api.TypeSafeRestApi", version: "0.22.14" };
//# sourceMappingURL=data:application/json;base64,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