"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.OpenApiGatewayWebAcl = void 0;
/*! Copyright [Amazon.com](http://amazon.com/), Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0 */
const pdk_nag_1 = require("../../../pdk-nag");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_wafv2_1 = require("aws-cdk-lib/aws-wafv2");
const constructs_1 = require("constructs");
/**
 * Associate an AWS WAF v2 Web ACL with the given api
 */
class OpenApiGatewayWebAcl extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        const aclName = `${pdk_nag_1.PDKNag.getStackPrefix(aws_cdk_lib_1.Stack.of(this))
            .split("/")
            .join("-")}-${id}-WebAcl`;
        const ipSetName = `${aclName}-IPSet`;
        // Create the IP Set if requested
        this.ipSet = props.cidrAllowList
            ? new aws_wafv2_1.CfnIPSet(this, "IPSet", {
                name: ipSetName,
                addresses: props.cidrAllowList.cidrRanges,
                ipAddressVersion: props.cidrAllowList.cidrType,
                scope: "REGIONAL",
            })
            : undefined;
        const managedRules = props.managedRules ?? [
            { vendor: "AWS", name: "AWSManagedRulesCommonRuleSet" },
        ];
        const rules = [
            // Add a rule for the IP Set if specified
            ...(this.ipSet
                ? [
                    {
                        name: ipSetName,
                        priority: 1,
                        visibilityConfig: {
                            metricName: ipSetName,
                            cloudWatchMetricsEnabled: true,
                            sampledRequestsEnabled: true,
                        },
                        action: {
                            block: {},
                        },
                        statement: {
                            notStatement: {
                                statement: {
                                    ipSetReferenceStatement: {
                                        arn: this.ipSet.attrArn,
                                    },
                                },
                            },
                        },
                    },
                ]
                : []),
            // Add the managed rules
            ...managedRules.map(({ vendor, name }, i) => ({
                name: `${vendor}-${name}`,
                priority: i + 2,
                statement: { managedRuleGroupStatement: { vendorName: vendor, name } },
                overrideAction: { none: {} },
                visibilityConfig: {
                    metricName: `${aclName}-${vendor}-${name}`,
                    cloudWatchMetricsEnabled: true,
                    sampledRequestsEnabled: true,
                },
            })),
        ];
        this.webAcl = new aws_wafv2_1.CfnWebACL(this, "WebACL", {
            name: aclName,
            defaultAction: {
                // Allow by default, and use rules to deny unwanted requests
                allow: {},
            },
            scope: "REGIONAL",
            visibilityConfig: {
                cloudWatchMetricsEnabled: true,
                sampledRequestsEnabled: true,
                metricName: aclName,
            },
            rules,
        });
        this.webAclAssociation = new aws_wafv2_1.CfnWebACLAssociation(this, "WebACLAssociation", {
            resourceArn: props.apiDeploymentStageArn,
            webAclArn: this.webAcl.attrArn,
        });
    }
}
exports.OpenApiGatewayWebAcl = OpenApiGatewayWebAcl;
//# sourceMappingURL=data:application/json;base64,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