"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.OpenApiToolsJsonFile = void 0;
/*! Copyright [Amazon.com](http://amazon.com/), Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0 */
const monorepo_1 = require("../../../../monorepo");
const projen_1 = require("projen");
/**
 * Represents an openapitools.json file
 * @see https://github.com/OpenAPITools/openapi-generator-cli#configuration
 */
class OpenApiToolsJsonFile extends projen_1.JsonFile {
    /**
     * Retrieves an instance of OpenApiToolsJsonFile if one is associated to the given project.
     *
     * @param project project instance.
     */
    static of(project) {
        return project.components.find((c) => monorepo_1.ProjectUtils.isNamedInstanceOf(c, OpenApiToolsJsonFile));
    }
    /**
     * Retrieves an instance of OpenApiToolsJsonFile if one is associated to the given project,
     * otherwise creates a OpenApiToolsJsonFile instance for the project.
     *
     * @param project project instance.
     */
    static ensure(project) {
        return (OpenApiToolsJsonFile.of(project) || new OpenApiToolsJsonFile(project));
    }
    constructor(project) {
        if (OpenApiToolsJsonFile.of(project)) {
            throw new Error(`Project ${project.name} already has associated OpenApiToolsJsonFile component.`);
        }
        super(project, "openapitools.json", {
            obj: {
                // Schema is located in node_modules when generator cli is installed in tmp dir
                $schema: "node_modules/@openapitools/openapi-generator-cli/config.schema.json",
                spaces: 2,
                "generator-cli": () => this.config,
            },
        });
        /**
         * Configuration for OpenAPI Generator CLI
         * @private
         */
        this.config = {
            version: "6.3.0",
            storageDir: "~/.open-api-generator-cli",
        };
    }
    /**
     * Update the OpenAPI Generator config. Shallow-merges with any existing default config
     */
    addOpenApiGeneratorCliConfig(config) {
        this.config = {
            ...this.config,
            ...config,
        };
    }
}
exports.OpenApiToolsJsonFile = OpenApiToolsJsonFile;
//# sourceMappingURL=data:application/json;base64,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