import { Project } from "projen";
import { Language, Library } from "../../languages";
import { MockResponseDataGenerationOptions } from "../../types";
import { GeneratedHandlersProjects } from "../generate";
/**
 * Enum for generator directories for non-runtime generators
 */
export declare enum OtherGenerators {
    DOCS = "docs",
    TYPESCRIPT_CDK_INFRASTRUCTURE = "typescript-cdk-infrastructure",
    PYTHON_CDK_INFRASTRUCTURE = "python-cdk-infrastructure",
    JAVA_CDK_INFRASTRUCTURE = "java-cdk-infrastructure",
    TYPESCRIPT_LAMBDA_HANDLERS = "typescript-lambda-handlers",
    PYTHON_LAMBDA_HANDLERS = "python-lambda-handlers",
    JAVA_LAMBDA_HANDLERS = "java-lambda-handlers"
}
export declare enum TypeSafeApiScript {
    PARSE_OPENAPI_SPEC = "type-safe-api.parse-openapi-spec",
    GENERATE = "type-safe-api.generate",
    GENERATE_MOCK_DATA = "type-safe-api.generate-mock-data",
    GENERATE_HTML_REDOC_DOCS = "type-safe-api.generate-html-redoc-docs",
    CLEAN_OPENAPI_GENERATED_CODE = "type-safe-api.clean-openapi-generated-code",
    COPY_GRADLE_WRAPPER = "type-safe-api.copy-gradle-wrapper"
}
/**
 * Generator directory for openapi generation containing templates, config etc.
 */
export type GeneratorDirectory = Language | Library | OtherGenerators;
/**
 * Types of normalizers supported by openapi-generator
 * @see https://openapi-generator.tech/docs/customization/#openapi-normalizer
 */
export type OpenApiNormalizer = "KEEP_ONLY_FIRST_TAG_IN_OPERATION";
/**
 * Options for generating code or docs using OpenAPI Generator CLI
 */
export interface GenerationOptions {
    /**
     * The OpenAPI generator to use to generate the code/docs
     */
    readonly generator: string;
    /**
     * The directory to use for OpenAPI generation
     */
    readonly generatorDirectory: GeneratorDirectory;
    /**
     * The path of the OpenAPI spec to generate code from
     */
    readonly specPath: string;
    /**
     * Additional properties to pass to the generate cli
     */
    readonly additionalProperties?: {
        [key: string]: string;
    };
    /**
     * Supply the relative path from the code project root to the source code directory in which custom generated files
     * (eg. operation config) should be placed.
     */
    readonly srcDir?: string;
    /**
     * Normalizers to apply to the spec prior to generation, if any
     * @see https://openapi-generator.tech/docs/customization/#openapi-normalizer
     */
    readonly normalizers?: Partial<Record<OpenApiNormalizer, boolean>>;
    /**
     * Vendor extensions to add for code generation, allowing custom properties to be passed to the generator templates
     * Keys should begin with "x-"
     */
    readonly extraVendorExtensions?: Record<string, string | boolean>;
    /**
     * Generate alias as model.
     * Defines whether "primitive types" defined at the model/schema level will be wrapped in a model (ie maps/lists)
     * @see https://openapi-generator.tech/docs/globals/
     * @default true
     */
    readonly generateAliasAsModel?: boolean;
}
/**
 * Return the environment that should be used for executing type safe api commands
 */
export declare const getTypeSafeApiTaskEnvironment: () => {
    [key: string]: string;
};
/**
 * Build a command for running a script from this project's bin
 */
export declare const buildTypeSafeApiExecCommand: (script: TypeSafeApiScript, args?: string) => string;
/**
 * Generate code or docs by invoking the root generate script
 */
export declare const buildInvokeOpenApiGeneratorCommandArgs: (options: GenerationOptions) => string;
/**
 * Builds a command to clean up files which were previously generated by openapi generator
 */
export declare const buildCleanOpenApiGeneratedCodeCommand: () => string;
/**
 * Options for generating mock data json files
 */
export interface MockDataGenerationOptions extends MockResponseDataGenerationOptions {
    /**
     * The path of the OpenAPI spec to generate data for
     */
    readonly specPath: string;
    /**
     * Output sub directory relative to the outdir in which to generate mock data
     * Mock data will be written to a directory named 'mocks' within the sub directory
     * @default .
     */
    readonly outputSubDir?: string;
}
/**
 * Invoke the mock data generator script
 */
export declare const buildInvokeMockDataGeneratorCommand: (options: MockDataGenerationOptions) => string;
/**
 * Return vendor extensions containing details about the handler projects
 */
export declare const getHandlersProjectVendorExtensions: (targetProject: Project, { java, python, typescript }: GeneratedHandlersProjects) => Record<string, string | boolean>;
