"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.getHandlersProjectVendorExtensions = exports.buildInvokeMockDataGeneratorCommand = exports.buildCleanOpenApiGeneratedCodeCommand = exports.buildInvokeOpenApiGeneratorCommandArgs = exports.buildTypeSafeApiExecCommand = exports.getTypeSafeApiTaskEnvironment = exports.TypeSafeApiScript = exports.OtherGenerators = void 0;
/*! Copyright [Amazon.com](http://amazon.com/), Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0 */
const path = require("path");
const readPkg = require("read-pkg-up");
/**
 * Enum for generator directories for non-runtime generators
 */
var OtherGenerators;
(function (OtherGenerators) {
    OtherGenerators["DOCS"] = "docs";
    // Infrastructure
    OtherGenerators["TYPESCRIPT_CDK_INFRASTRUCTURE"] = "typescript-cdk-infrastructure";
    OtherGenerators["PYTHON_CDK_INFRASTRUCTURE"] = "python-cdk-infrastructure";
    OtherGenerators["JAVA_CDK_INFRASTRUCTURE"] = "java-cdk-infrastructure";
    // Handlers
    OtherGenerators["TYPESCRIPT_LAMBDA_HANDLERS"] = "typescript-lambda-handlers";
    OtherGenerators["PYTHON_LAMBDA_HANDLERS"] = "python-lambda-handlers";
    OtherGenerators["JAVA_LAMBDA_HANDLERS"] = "java-lambda-handlers";
})(OtherGenerators || (exports.OtherGenerators = OtherGenerators = {}));
var TypeSafeApiScript;
(function (TypeSafeApiScript) {
    TypeSafeApiScript["PARSE_OPENAPI_SPEC"] = "type-safe-api.parse-openapi-spec";
    TypeSafeApiScript["GENERATE"] = "type-safe-api.generate";
    TypeSafeApiScript["GENERATE_MOCK_DATA"] = "type-safe-api.generate-mock-data";
    TypeSafeApiScript["GENERATE_HTML_REDOC_DOCS"] = "type-safe-api.generate-html-redoc-docs";
    TypeSafeApiScript["CLEAN_OPENAPI_GENERATED_CODE"] = "type-safe-api.clean-openapi-generated-code";
    TypeSafeApiScript["COPY_GRADLE_WRAPPER"] = "type-safe-api.copy-gradle-wrapper";
})(TypeSafeApiScript || (exports.TypeSafeApiScript = TypeSafeApiScript = {}));
/**
 * Get the current package version
 */
const getPackageVersion = () => {
    const { packageJson } = readPkg.sync({
        cwd: path.resolve(__dirname),
    });
    return packageJson.version;
};
/**
 * Return the environment that should be used for executing type safe api commands
 */
const getTypeSafeApiTaskEnvironment = () => ({
    AWS_PDK_VERSION: getPackageVersion(),
});
exports.getTypeSafeApiTaskEnvironment = getTypeSafeApiTaskEnvironment;
/**
 * Build a command for running a script from this project's bin
 */
const buildTypeSafeApiExecCommand = (script, args) => {
    return `npx --yes -p @aws/pdk@$AWS_PDK_VERSION ${script}${args ? ` ${args}` : ""}`;
};
exports.buildTypeSafeApiExecCommand = buildTypeSafeApiExecCommand;
const serializeProperties = (properties) => Object.entries(properties)
    .map(([key, value]) => `${key}=${value}`)
    .join(",");
/**
 * Generate code or docs by invoking the root generate script
 */
const buildInvokeOpenApiGeneratorCommandArgs = (options) => {
    const srcDir = options.srcDir ?? "src";
    const additionalProperties = options.additionalProperties
        ? ` --additional-properties "${serializeProperties(options.additionalProperties)}"`
        : "";
    const normalizers = options.normalizers
        ? ` --openapi-normalizer "${serializeProperties(Object.fromEntries(Object.entries(options.normalizers).map(([k, v]) => [k, `${v}`])))}"`
        : "";
    const extensions = options.extraVendorExtensions
        ? ` --extra-vendor-extensions '${JSON.stringify(options.extraVendorExtensions)}'`
        : "";
    const generateAliasAsModel = options.generateAliasAsModel ?? true ? " --generate-alias-as-model" : "";
    const specPath = options.specPath;
    const outputPath = ".";
    return `--generator ${options.generator} --spec-path ${specPath} --output-path ${outputPath} --generator-dir ${options.generatorDirectory} --src-dir ${srcDir}${additionalProperties}${normalizers}${extensions}${generateAliasAsModel}`;
};
exports.buildInvokeOpenApiGeneratorCommandArgs = buildInvokeOpenApiGeneratorCommandArgs;
/**
 * Builds a command to clean up files which were previously generated by openapi generator
 */
const buildCleanOpenApiGeneratedCodeCommand = () => {
    return (0, exports.buildTypeSafeApiExecCommand)(TypeSafeApiScript.CLEAN_OPENAPI_GENERATED_CODE, `--code-path .`);
};
exports.buildCleanOpenApiGeneratedCodeCommand = buildCleanOpenApiGeneratedCodeCommand;
/**
 * Invoke the mock data generator script
 */
const buildInvokeMockDataGeneratorCommand = (options) => {
    const outputPath = options.outputSubDir ?? ".";
    const locale = options.locale ? ` --locale ${options.locale}` : "";
    const maxArrayLength = options.maxArrayLength !== undefined
        ? ` --max-array-length ${options.maxArrayLength}`
        : "";
    const seed = options.seed !== undefined ? ` --seed ${options.seed}` : "";
    return (0, exports.buildTypeSafeApiExecCommand)(TypeSafeApiScript.GENERATE_MOCK_DATA, `--spec-path ${options.specPath} --output-path ${outputPath}${locale}${maxArrayLength}${seed}`);
};
exports.buildInvokeMockDataGeneratorCommand = buildInvokeMockDataGeneratorCommand;
/**
 * Return vendor extensions containing details about the handler projects
 */
const getHandlersProjectVendorExtensions = (targetProject, { java, python, typescript }) => ({
    "x-handlers-python-module": python?.moduleName ?? "",
    "x-handlers-java-package": java?.packageName ?? "",
    "x-handlers-typescript-asset-path": typescript
        ? path.join(path.relative(targetProject.outdir, typescript.outdir), "dist", "lambda")
        : "",
    "x-handlers-python-asset-path": python
        ? path.join(path.relative(targetProject.outdir, python.outdir), "dist", "lambda")
        : "",
    "x-handlers-java-asset-path": java
        ? path.join(path.relative(targetProject.outdir, java.outdir), java.distdir, ...java.pom.groupId.split("."), java.pom.artifactId, java.pom.version, `${java.pom.artifactId}-${java.pom.version}.jar`)
        : "",
});
exports.getHandlersProjectVendorExtensions = getHandlersProjectVendorExtensions;
//# sourceMappingURL=data:application/json;base64,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