"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.GeneratedJavaHandlersProject = void 0;
/*! Copyright [Amazon.com](http://amazon.com/), Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0 */
const path = require("path");
const projen_1 = require("projen");
const java_1 = require("projen/lib/java");
const open_api_generator_handlebars_ignore_file_1 = require("../components/open-api-generator-handlebars-ignore-file");
const open_api_generator_ignore_file_1 = require("../components/open-api-generator-ignore-file");
const open_api_tools_json_file_1 = require("../components/open-api-tools-json-file");
const type_safe_api_command_environment_1 = require("../components/type-safe-api-command-environment");
const utils_1 = require("../components/utils");
class GeneratedJavaHandlersProject extends java_1.JavaProject {
    constructor(options) {
        super({
            sample: false,
            junit: false,
            ...options,
        });
        this.buildGenerateCommandArgs = () => {
            return (0, utils_1.buildInvokeOpenApiGeneratorCommandArgs)({
                generator: "java",
                specPath: this.options.specPath,
                generatorDirectory: utils_1.OtherGenerators.JAVA_LAMBDA_HANDLERS,
                srcDir: this.srcDir,
                normalizers: {
                    KEEP_ONLY_FIRST_TAG_IN_OPERATION: true,
                },
                extraVendorExtensions: {
                    "x-handlers-package": this.packageName,
                    "x-runtime-package": this.options.generatedJavaTypes.packageName,
                },
                // Do not generate map/list types. Generator will use built in HashMap, ArrayList instead
                generateAliasAsModel: false,
            });
        };
        type_safe_api_command_environment_1.TypeSafeApiCommandEnvironment.ensure(this);
        this.options = options;
        this.packageName = `${this.pom.groupId}.${this.name}.handlers`;
        this.srcDir = path.join("src", "main", "java", ...this.packageName.split("."));
        [
            `${options.generatedJavaTypes.pom.groupId}/${options.generatedJavaTypes.pom.artifactId}@${options.generatedJavaTypes.pom.version}`,
        ]
            .filter((dep) => !this.deps.tryGetDependency(dep.split("@")[0], projen_1.DependencyType.RUNTIME))
            .forEach((dep) => this.addDependency(dep));
        // Remove the projen test dependency since otherwise it takes precedence, causing projen to be unavailable at synth time
        this.deps.removeDependency("io.github.cdklabs/projen", projen_1.DependencyType.TEST);
        // Add a dependency on the generated java types repository
        this.pom.addRepository({
            url: `file://${path.relative(this.outdir, options.generatedJavaTypes.outdir)}/dist/java`,
            id: `${options.generatedJavaTypes.pom.groupId}-${options.generatedJavaTypes.pom.artifactId}-repo`,
        });
        // Ignore everything for the first mustache pass
        const openapiGeneratorIgnore = new open_api_generator_ignore_file_1.OpenApiGeneratorIgnoreFile(this);
        openapiGeneratorIgnore.addPatterns("/*", "**/*", "*");
        // Ignore everything but the handler files for the handlebars pass
        const openapiGeneratorHandlebarsIgnore = new open_api_generator_handlebars_ignore_file_1.OpenApiGeneratorHandlebarsIgnoreFile(this);
        openapiGeneratorHandlebarsIgnore.addPatterns("/*", "**/*", "*", 
        // This will be split into a file per targeted handler
        `!${this.srcDir}/__all_handlers.java`);
        // Add OpenAPI Generator cli configuration
        open_api_tools_json_file_1.OpenApiToolsJsonFile.ensure(this).addOpenApiGeneratorCliConfig(options.openApiGeneratorCliConfig);
        const generateTask = this.addTask("generate");
        generateTask.exec((0, utils_1.buildCleanOpenApiGeneratedCodeCommand)());
        generateTask.exec((0, utils_1.buildTypeSafeApiExecCommand)(utils_1.TypeSafeApiScript.GENERATE, this.buildGenerateCommandArgs()));
        this.preCompileTask.spawn(generateTask);
        // Ignore the openapi generator metadata files
        this.gitignore.addPatterns(".openapi-generator");
        // Use the maven shade plugin to build a "super jar" which we can deploy to AWS Lambda
        this.pom.addPlugin("org.apache.maven.plugins/maven-shade-plugin@3.3.0", {
            configuration: {
                createDependencyReducedPom: false,
                transformers: [
                    {
                        // Transformer required for merging log4j2 plugins cache file
                        // https://docs.aws.amazon.com/lambda/latest/dg/java-logging.html#java-logging-cdk
                        transformer: {
                            "@implementation": "com.github.edwgiz.maven_shade_plugin.log4j2_cache_transformer.PluginsCacheFileTransformer",
                        },
                    },
                ],
            },
            executions: [
                {
                    id: "shade-task",
                    phase: "package",
                    goals: ["shade"],
                },
            ],
            dependencies: [
                "com.github.edwgiz/maven-shade-plugin.log4j2-cachefile-transformer@2.15.0",
            ],
        });
        // Log4j2 configuration for powertools logger
        new projen_1.SampleDir(this, "src/main/resources", {
            files: {
                "log4j2.xml": `<?xml version="1.0" encoding="UTF-8"?>
<Configuration>
    <Appenders>
        <Console name="JsonAppender" target="SYSTEM_OUT">
            <JsonTemplateLayout eventTemplateUri="classpath:LambdaJsonLayout.json" />
        </Console>
    </Appenders>
    <Loggers>
        <Logger name="JsonLogger" level="INFO" additivity="false">
            <AppenderRef ref="JsonAppender"/>
        </Logger>
        <Root level="info">
            <AppenderRef ref="JsonAppender"/>
        </Root>
    </Loggers>
</Configuration>`,
            },
        });
    }
}
exports.GeneratedJavaHandlersProject = GeneratedJavaHandlersProject;
//# sourceMappingURL=data:application/json;base64,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