"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.GeneratedPythonHandlersProject = void 0;
/*! Copyright [Amazon.com](http://amazon.com/), Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0 */
const path = require("path");
const projen_1 = require("projen");
const python_1 = require("projen/lib/python");
const open_api_generator_handlebars_ignore_file_1 = require("../components/open-api-generator-handlebars-ignore-file");
const open_api_generator_ignore_file_1 = require("../components/open-api-generator-ignore-file");
const open_api_tools_json_file_1 = require("../components/open-api-tools-json-file");
const type_safe_api_command_environment_1 = require("../components/type-safe-api-command-environment");
const utils_1 = require("../components/utils");
class GeneratedPythonHandlersProject extends python_1.PythonProject {
    constructor(options) {
        super({
            pytest: false,
            poetryOptions: {
                packages: [{ include: options.moduleName }],
                // Module must be explicitly added to include since poetry excludes everything in .gitignore by default
                include: [options.moduleName, `${options.moduleName}/**/*.py`],
            },
            ...options,
            sample: false,
            poetry: true,
        });
        this.buildGenerateCommandArgs = () => {
            return (0, utils_1.buildInvokeOpenApiGeneratorCommandArgs)({
                generator: "python-nextgen",
                specPath: this.options.specPath,
                generatorDirectory: utils_1.OtherGenerators.PYTHON_LAMBDA_HANDLERS,
                // Tell the generator where python source files live
                srcDir: this.moduleName,
                normalizers: {
                    KEEP_ONLY_FIRST_TAG_IN_OPERATION: true,
                },
                extraVendorExtensions: {
                    "x-runtime-module-name": this.options.generatedPythonTypes.moduleName,
                },
            });
        };
        type_safe_api_command_environment_1.TypeSafeApiCommandEnvironment.ensure(this);
        this.options = options;
        [
            "python@^3.9",
            `${options.generatedPythonTypes.name}@{path="${path.relative(this.outdir, options.generatedPythonTypes.outdir)}", develop=true}`,
        ]
            .filter((dep) => !this.deps.tryGetDependency(dep, projen_1.DependencyType.RUNTIME))
            .forEach((dep) => this.addDependency(dep));
        // Ignore everything for the first mustache pass
        const openapiGeneratorIgnore = new open_api_generator_ignore_file_1.OpenApiGeneratorIgnoreFile(this);
        openapiGeneratorIgnore.addPatterns("/*", "**/*", "*");
        // Ignore everything but the handler files for the handlebars pass
        const openapiGeneratorHandlebarsIgnore = new open_api_generator_handlebars_ignore_file_1.OpenApiGeneratorHandlebarsIgnoreFile(this);
        openapiGeneratorHandlebarsIgnore.addPatterns("/*", "**/*", "*", 
        // This will be split into a file per targeted handler
        `!${this.moduleName}/__all_handlers.py`);
        // Add OpenAPI Generator cli configuration
        open_api_tools_json_file_1.OpenApiToolsJsonFile.ensure(this).addOpenApiGeneratorCliConfig({
            version: "6.6.0",
            ...options.openApiGeneratorCliConfig,
        });
        const generateTask = this.addTask("generate");
        generateTask.exec((0, utils_1.buildCleanOpenApiGeneratedCodeCommand)());
        generateTask.exec((0, utils_1.buildTypeSafeApiExecCommand)(utils_1.TypeSafeApiScript.GENERATE, this.buildGenerateCommandArgs()));
        this.preCompileTask.spawn(generateTask);
        // Ignore the generated code
        this.gitignore.addPatterns(".openapi-generator");
        // Write __init__.py as sample code
        new projen_1.SampleFile(this, path.join(this.moduleName, "__init__.py"), {
            contents: "#",
        });
        // Package task to build a distributable which can be deployed to lambda
        this.packageTask.exec("mkdir -p dist/lambda && rm -rf dist/lambda/*");
        this.packageTask.exec(`cp -r ${this.moduleName} dist/lambda/${this.moduleName}`);
        this.packageTask.exec("poetry export --without-hashes --format=requirements.txt > dist/lambda/requirements.txt");
        this.packageTask.exec("pip install -r dist/lambda/requirements.txt --target dist/lambda --upgrade");
    }
}
exports.GeneratedPythonHandlersProject = GeneratedPythonHandlersProject;
//# sourceMappingURL=data:application/json;base64,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