"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.GeneratedTypescriptHandlersProject = void 0;
/*! Copyright [Amazon.com](http://amazon.com/), Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0 */
const path = require("path");
const projen_1 = require("projen");
const javascript_1 = require("projen/lib/javascript");
const typescript_1 = require("projen/lib/typescript");
const open_api_generator_handlebars_ignore_file_1 = require("../components/open-api-generator-handlebars-ignore-file");
const open_api_generator_ignore_file_1 = require("../components/open-api-generator-ignore-file");
const open_api_tools_json_file_1 = require("../components/open-api-tools-json-file");
const type_safe_api_command_environment_1 = require("../components/type-safe-api-command-environment");
const utils_1 = require("../components/utils");
class GeneratedTypescriptHandlersProject extends typescript_1.TypeScriptProject {
    constructor(options) {
        super({
            ...options,
            sampleCode: false,
            tsconfig: {
                compilerOptions: {
                    lib: ["dom", "es2019"],
                    // Generated code imports all models, and may not reference them all
                    noUnusedLocals: false,
                    noUnusedParameters: false,
                    skipLibCheck: true,
                    ...options?.tsconfig?.compilerOptions,
                },
            },
            npmignoreEnabled: false,
        });
        this.buildGenerateCommandArgs = () => {
            return (0, utils_1.buildInvokeOpenApiGeneratorCommandArgs)({
                generator: "typescript-fetch",
                specPath: this.options.specPath,
                generatorDirectory: utils_1.OtherGenerators.TYPESCRIPT_LAMBDA_HANDLERS,
                srcDir: this.srcdir,
                normalizers: {
                    KEEP_ONLY_FIRST_TAG_IN_OPERATION: true,
                },
                extraVendorExtensions: {
                    "x-runtime-package-name": this.options.generatedTypescriptTypes.package.packageName,
                },
            });
        };
        this.options = options;
        type_safe_api_command_environment_1.TypeSafeApiCommandEnvironment.ensure(this);
        this.addDeps(...[
            // If within a monorepo, add a regular dependency. Otherwise, use a file dependency to ensure the runtime can be
            // resolved
            options.isWithinMonorepo
                ? options.generatedTypescriptTypes.package.packageName
                : `${options.generatedTypescriptTypes.package.packageName}@file:${path.relative(this.outdir, options.generatedTypescriptTypes.outdir)}`,
        ].filter((dep) => !this.deps.tryGetDependency(dep, projen_1.DependencyType.RUNTIME)));
        this.addDevDeps("esbuild", "@types/aws-lambda");
        // Minimal .npmignore to avoid impacting OpenAPI Generator
        const npmignore = new projen_1.IgnoreFile(this, ".npmignore");
        npmignore.addPatterns("/.projen/", "/src", "/dist");
        // Ignore everything for the regular open api generator pass
        const openapiGeneratorIgnore = new open_api_generator_ignore_file_1.OpenApiGeneratorIgnoreFile(this);
        openapiGeneratorIgnore.addPatterns("/*", "**/*", "*");
        // Ignore everything but the handler files for the handlebars pass
        const openapiGeneratorHandlebarsIgnore = new open_api_generator_handlebars_ignore_file_1.OpenApiGeneratorHandlebarsIgnoreFile(this);
        openapiGeneratorHandlebarsIgnore.addPatterns("/*", "**/*", "*", 
        // This will be split into a file per targeted handler
        `!${this.srcdir}/__all_handlers.ts`);
        // Add OpenAPI Generator cli configuration
        open_api_tools_json_file_1.OpenApiToolsJsonFile.ensure(this).addOpenApiGeneratorCliConfig(options.openApiGeneratorCliConfig);
        const generateTask = this.addTask("generate");
        generateTask.exec((0, utils_1.buildCleanOpenApiGeneratedCodeCommand)());
        generateTask.exec((0, utils_1.buildTypeSafeApiExecCommand)(utils_1.TypeSafeApiScript.GENERATE, this.buildGenerateCommandArgs()));
        this.preCompileTask.spawn(generateTask);
        // Ignore the openapi generator metadata
        this.gitignore.addPatterns(".openapi-generator");
        // Create a separate lambda bundle for each handler as part of the package task.
        // Note that every typescript file directly in src is bundled by default, but users may specify their own
        // entry point globs if they prefer a different directory structure.
        this.packageTask.exec(`mkdir -p dist/lambda && rm -rf dist/lambda/*`);
        this.packageTask.exec(`esbuild --bundle ${(options.handlerEntryPoints ?? [`${this.srcdir}/*.ts`]).join(" ")} --platform=node --outdir=dist/lambda`);
        // Move each bundled file into a separate directory
        this.packageTask.exec("for f in $(ls dist/lambda); do mkdir dist/lambda/$(basename $f .js) && mv dist/lambda/$f dist/lambda/$(basename $f .js)/index.js; done");
        // Create an empty index.ts sample on synth so that tsc is happy if the handlers project is configured
        // but no operations have @handler(language: "typescript")
        new projen_1.SampleDir(this, this.srcdir, {
            files: {
                "index.ts": "",
            },
        });
        // If we're not in a monorepo, we need to link the generated types such that the local dependency can be resolved
        if (!options.isWithinMonorepo) {
            switch (this.package.packageManager) {
                case javascript_1.NodePackageManager.NPM:
                case javascript_1.NodePackageManager.YARN:
                case javascript_1.NodePackageManager.YARN2:
                    this.tasks
                        .tryFind("install")
                        ?.prependExec(`${this.package.packageManager} link ${this.options.generatedTypescriptTypes.package.packageName}`);
                    break;
                case javascript_1.NodePackageManager.PNPM:
                    this.tasks
                        .tryFind("install")
                        ?.prependExec(`${this.package.packageManager} link /${path.relative(this.outdir, this.options.generatedTypescriptTypes.outdir)}`);
                    break;
                default:
                    console.warn(`Unknown package manager ${this.package.packageManager}. Cannot link generated typescript client.`);
            }
        }
    }
}
exports.GeneratedTypescriptHandlersProject = GeneratedTypescriptHandlersProject;
//# sourceMappingURL=data:application/json;base64,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