"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.GeneratedJavaCdkInfrastructureProject = void 0;
/*! Copyright [Amazon.com](http://amazon.com/), Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0 */
const path = require("path");
const projen_1 = require("projen");
const java_1 = require("projen/lib/java");
const open_api_generator_handlebars_ignore_file_1 = require("../../components/open-api-generator-handlebars-ignore-file");
const open_api_generator_ignore_file_1 = require("../../components/open-api-generator-ignore-file");
const open_api_tools_json_file_1 = require("../../components/open-api-tools-json-file");
const type_safe_api_command_environment_1 = require("../../components/type-safe-api-command-environment");
const utils_1 = require("../../components/utils");
class GeneratedJavaCdkInfrastructureProject extends java_1.JavaProject {
    constructor(options) {
        super({
            ...options,
            sample: false,
            junit: false,
        });
        this.buildGenerateCommandArgs = () => {
            return (0, utils_1.buildInvokeOpenApiGeneratorCommandArgs)({
                generator: "java",
                specPath: this.options.specPath,
                generatorDirectory: utils_1.OtherGenerators.JAVA_CDK_INFRASTRUCTURE,
                srcDir: this.srcDir,
                normalizers: {
                    KEEP_ONLY_FIRST_TAG_IN_OPERATION: true,
                },
                extraVendorExtensions: {
                    "x-infrastructure-package": this.packageName,
                    "x-runtime-package": this.options.generatedJavaTypes.packageName,
                    // Enable mock integration generation by default
                    "x-enable-mock-integrations": !this.options.mockDataOptions?.disable,
                    ...(0, utils_1.getHandlersProjectVendorExtensions)(this, this.options.generatedHandlers),
                },
                // Do not generate map/list types. Generator will use built in HashMap, ArrayList instead
                generateAliasAsModel: false,
            });
        };
        this.buildGenerateMockDataCommand = () => {
            return (0, utils_1.buildInvokeMockDataGeneratorCommand)({
                specPath: this.options.specPath,
                // Write the mocks to the resources directory
                outputSubDir: "src/main/resources",
                ...this.options.mockDataOptions,
            });
        };
        type_safe_api_command_environment_1.TypeSafeApiCommandEnvironment.ensure(this);
        this.options = options;
        this.packageName = `${this.pom.groupId}.${this.name}.infra`;
        this.srcDir = path.join("src", "main", "java", ...this.packageName.split("."));
        [
            "software.aws/pdk@^0",
            "software.constructs/constructs@^10",
            "software.amazon.awscdk/aws-cdk-lib@^2",
            "io.github.cdklabs/cdknag@^2",
            "org.projectlombok/lombok@^1",
            "com.fasterxml.jackson.core/jackson-databind@^2",
            "io.github.cdklabs/projen@^0",
            `${options.generatedJavaTypes.pom.groupId}/${options.generatedJavaTypes.pom.artifactId}@${options.generatedJavaTypes.pom.version}`,
        ]
            .filter((dep) => !this.deps.tryGetDependency(dep.split("@")[0], projen_1.DependencyType.RUNTIME))
            .forEach((dep) => this.addDependency(dep));
        // Remove the projen test dependency since otherwise it takes precedence, causing projen to be unavailable at synth time
        this.deps.removeDependency("io.github.cdklabs/projen", projen_1.DependencyType.TEST);
        // Add a dependency on the generated java types repository
        this.pom.addRepository({
            url: `file://${path.relative(this.outdir, options.generatedJavaTypes.outdir)}/dist/java`,
            id: `${options.generatedJavaTypes.pom.groupId}-${options.generatedJavaTypes.pom.artifactId}-repo`,
        });
        // Ignore everything but the target files
        const openapiGeneratorIgnore = new open_api_generator_ignore_file_1.OpenApiGeneratorIgnoreFile(this);
        openapiGeneratorIgnore.addPatterns("/*", "**/*", "*", `!${this.srcDir}/Api.java`, `!${this.srcDir}/ApiProps.java`, `!${this.srcDir}/MockIntegrations.java`);
        const openapiGeneratorHandlebarsIgnore = new open_api_generator_handlebars_ignore_file_1.OpenApiGeneratorHandlebarsIgnoreFile(this);
        openapiGeneratorHandlebarsIgnore.addPatterns("/*", "**/*", "*", `!${this.srcDir}/__functions.java`);
        // Add OpenAPI Generator cli configuration
        open_api_tools_json_file_1.OpenApiToolsJsonFile.ensure(this).addOpenApiGeneratorCliConfig(options.openApiGeneratorCliConfig);
        const generateTask = this.addTask("generate");
        generateTask.exec((0, utils_1.buildCleanOpenApiGeneratedCodeCommand)());
        generateTask.exec((0, utils_1.buildTypeSafeApiExecCommand)(utils_1.TypeSafeApiScript.GENERATE, this.buildGenerateCommandArgs()));
        // Copy the parsed spec into the resources directory so that it's included in the jar
        generateTask.exec("mkdir -p src/main/resources");
        generateTask.exec(`cp -f ${this.options.specPath} src/main/resources/.api.json`);
        // Absolute path of this project is required for determining the path to the handlers jar,
        // since java executes from the jar which could be anywhere in the filesystem (eg the .m2 directory).
        // While ugly, since this is written at build time and not checked in it remains portable.
        generateTask.exec("echo $(pwd) > src/main/resources/project-absolute-path.txt");
        if (!this.options.mockDataOptions?.disable) {
            generateTask.exec(this.buildGenerateMockDataCommand());
        }
        this.preCompileTask.spawn(generateTask);
        // Ignore the generated code
        this.gitignore.addPatterns("src", ".openapi-generator");
    }
}
exports.GeneratedJavaCdkInfrastructureProject = GeneratedJavaCdkInfrastructureProject;
//# sourceMappingURL=data:application/json;base64,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