"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.GeneratedPythonCdkInfrastructureProject = void 0;
/*! Copyright [Amazon.com](http://amazon.com/), Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0 */
const path = require("path");
const projen_1 = require("projen");
const python_1 = require("projen/lib/python");
const open_api_generator_handlebars_ignore_file_1 = require("../../components/open-api-generator-handlebars-ignore-file");
const open_api_generator_ignore_file_1 = require("../../components/open-api-generator-ignore-file");
const open_api_tools_json_file_1 = require("../../components/open-api-tools-json-file");
const type_safe_api_command_environment_1 = require("../../components/type-safe-api-command-environment");
const utils_1 = require("../../components/utils");
class GeneratedPythonCdkInfrastructureProject extends python_1.PythonProject {
    constructor(options) {
        super({
            ...options,
            sample: false,
            pytest: false,
            poetry: true,
            poetryOptions: {
                packages: [{ include: options.moduleName }],
                // Module must be explicitly added to include since poetry excludes everything in .gitignore by default
                include: [options.moduleName, `${options.moduleName}/**/*.py`],
            },
        });
        this.buildGenerateCommandArgs = () => {
            return (0, utils_1.buildInvokeOpenApiGeneratorCommandArgs)({
                generator: "python-nextgen",
                specPath: this.options.specPath,
                generatorDirectory: utils_1.OtherGenerators.PYTHON_CDK_INFRASTRUCTURE,
                // Tell the generator where python source files live
                srcDir: this.moduleName,
                normalizers: {
                    KEEP_ONLY_FIRST_TAG_IN_OPERATION: true,
                },
                extraVendorExtensions: {
                    "x-runtime-module-name": this.options.generatedPythonTypes.moduleName,
                    // Spec path relative to the source directory
                    "x-relative-spec-path": path.join("..", this.options.specPath),
                    // Enable mock integration generation by default
                    "x-enable-mock-integrations": !this.options.mockDataOptions?.disable,
                    ...(0, utils_1.getHandlersProjectVendorExtensions)(this, this.options.generatedHandlers),
                },
            });
        };
        this.buildGenerateMockDataCommand = () => {
            return (0, utils_1.buildInvokeMockDataGeneratorCommand)({
                specPath: this.options.specPath,
                ...this.options.mockDataOptions,
            });
        };
        type_safe_api_command_environment_1.TypeSafeApiCommandEnvironment.ensure(this);
        this.options = options;
        [
            "aws_pdk@^0",
            "constructs@^10",
            "aws-cdk-lib@^2",
            "cdk-nag@^2",
            "python@^3.9",
            `${options.generatedPythonTypes.name}@{path="${path.relative(this.outdir, options.generatedPythonTypes.outdir)}", develop=true}`,
        ]
            .filter((dep) => !this.deps.tryGetDependency(dep, projen_1.DependencyType.RUNTIME))
            .forEach((dep) => this.addDependency(dep));
        // Ignore everything but the target files
        const openapiGeneratorIgnore = new open_api_generator_ignore_file_1.OpenApiGeneratorIgnoreFile(this);
        openapiGeneratorIgnore.addPatterns("/*", "**/*", "*", `!${this.moduleName}/__init__.py`, `!${this.moduleName}/api.py`, `!${this.moduleName}/mock_integrations.py`);
        const openapiGeneratorHandlebarsIgnore = new open_api_generator_handlebars_ignore_file_1.OpenApiGeneratorHandlebarsIgnoreFile(this);
        openapiGeneratorHandlebarsIgnore.addPatterns("/*", "**/*", "*", `!${this.moduleName}/__functions.py`);
        // Add OpenAPI Generator cli configuration
        open_api_tools_json_file_1.OpenApiToolsJsonFile.ensure(this).addOpenApiGeneratorCliConfig({
            version: "6.6.0",
            ...options.openApiGeneratorCliConfig,
        });
        const generateTask = this.addTask("generate");
        generateTask.exec((0, utils_1.buildCleanOpenApiGeneratedCodeCommand)());
        generateTask.exec((0, utils_1.buildTypeSafeApiExecCommand)(utils_1.TypeSafeApiScript.GENERATE, this.buildGenerateCommandArgs()));
        if (!this.options.mockDataOptions?.disable) {
            generateTask.exec(this.buildGenerateMockDataCommand());
        }
        this.preCompileTask.spawn(generateTask);
        // Ignore the generated code
        this.gitignore.addPatterns(this.moduleName, ".openapi-generator", "mocks");
        // The poetry install that runs as part of post synthesis expects there to be some code present, but code isn't
        // generated until build time. This means that the first install will fail when either generating the project for
        // the first time or checking out a fresh copy (since generated code is not checked in to version control). We
        // therefore add a blank __init__.py as our first install step to keep poetry happy until the generator overwrites
        // it.
        this.tasks
            .tryFind("install")
            ?.prependExec(`mkdir -p ${this.moduleName} && touch ${this.moduleName}/__init__.py`);
    }
}
exports.GeneratedPythonCdkInfrastructureProject = GeneratedPythonCdkInfrastructureProject;
//# sourceMappingURL=data:application/json;base64,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