"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.GeneratedTypescriptCdkInfrastructureProject = void 0;
/*! Copyright [Amazon.com](http://amazon.com/), Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0 */
const path = require("path");
const projen_1 = require("projen");
const javascript_1 = require("projen/lib/javascript");
const typescript_1 = require("projen/lib/typescript");
const open_api_generator_handlebars_ignore_file_1 = require("../../components/open-api-generator-handlebars-ignore-file");
const open_api_generator_ignore_file_1 = require("../../components/open-api-generator-ignore-file");
const open_api_tools_json_file_1 = require("../../components/open-api-tools-json-file");
const type_safe_api_command_environment_1 = require("../../components/type-safe-api-command-environment");
const utils_1 = require("../../components/utils");
class GeneratedTypescriptCdkInfrastructureProject extends typescript_1.TypeScriptProject {
    constructor(options) {
        super({
            ...options,
            sampleCode: false,
            jest: false,
            eslint: false,
            prettier: false,
            tsconfig: {
                compilerOptions: {
                    lib: ["dom", "es2019"],
                    // Generated code imports all models, and may not reference them all
                    noUnusedLocals: false,
                    noUnusedParameters: false,
                    skipLibCheck: true,
                    ...options?.tsconfig?.compilerOptions,
                },
            },
            npmignoreEnabled: false,
        });
        /**
         * Path to the packaged copy of the openapi specification
         * @private
         */
        this.packagedSpecPath = "assets/api.json";
        this.buildGenerateCommandArgs = () => {
            return (0, utils_1.buildInvokeOpenApiGeneratorCommandArgs)({
                generator: "typescript-fetch",
                specPath: this.options.specPath,
                generatorDirectory: utils_1.OtherGenerators.TYPESCRIPT_CDK_INFRASTRUCTURE,
                srcDir: this.srcdir,
                normalizers: {
                    KEEP_ONLY_FIRST_TAG_IN_OPERATION: true,
                },
                extraVendorExtensions: {
                    "x-runtime-package-name": this.options.generatedTypescriptTypes.package.packageName,
                    // Spec path relative to the source directory
                    "x-relative-spec-path": path.join("..", this.packagedSpecPath),
                    // Enable mock integration generation by default
                    "x-enable-mock-integrations": !this.options.mockDataOptions?.disable,
                    ...(0, utils_1.getHandlersProjectVendorExtensions)(this, this.options.generatedHandlers),
                },
            });
        };
        this.buildGenerateMockDataCommand = () => {
            return (0, utils_1.buildInvokeMockDataGeneratorCommand)({
                specPath: this.options.specPath,
                ...this.options.mockDataOptions,
            });
        };
        type_safe_api_command_environment_1.TypeSafeApiCommandEnvironment.ensure(this);
        this.options = options;
        this.addDevDeps("@types/aws-lambda");
        this.addDeps(...[
            "@aws/pdk",
            "constructs",
            "aws-cdk-lib",
            "cdk-nag",
            // If within a monorepo, add a regular dependency. Otherwise, use a file dependency to ensure the types can be
            // resolved
            options.isWithinMonorepo
                ? options.generatedTypescriptTypes.package.packageName
                : `${options.generatedTypescriptTypes.package.packageName}@file:${path.relative(this.outdir, options.generatedTypescriptTypes.outdir)}`,
        ].filter((dep) => !this.deps.tryGetDependency(dep, projen_1.DependencyType.RUNTIME)));
        // Minimal .npmignore to avoid impacting OpenAPI Generator
        const npmignore = new projen_1.IgnoreFile(this, ".npmignore");
        npmignore.addPatterns("/.projen/", "/src", "/dist");
        // Ignore everything but the target files
        const openapiGeneratorIgnore = new open_api_generator_ignore_file_1.OpenApiGeneratorIgnoreFile(this);
        openapiGeneratorIgnore.addPatterns("/*", "**/*", "*", `!${this.srcdir}/index.ts`, `!${this.srcdir}/api.ts`, `!${this.srcdir}/mock-integrations.ts`);
        const openapiGeneratorHandlebarsIgnore = new open_api_generator_handlebars_ignore_file_1.OpenApiGeneratorHandlebarsIgnoreFile(this);
        openapiGeneratorHandlebarsIgnore.addPatterns("/*", "**/*", "*", `!${this.srcdir}/__functions.ts`);
        // Add OpenAPI Generator cli configuration
        open_api_tools_json_file_1.OpenApiToolsJsonFile.ensure(this).addOpenApiGeneratorCliConfig(options.openApiGeneratorCliConfig);
        const generateTask = this.addTask("generate");
        generateTask.exec((0, utils_1.buildCleanOpenApiGeneratedCodeCommand)());
        generateTask.exec((0, utils_1.buildTypeSafeApiExecCommand)(utils_1.TypeSafeApiScript.GENERATE, this.buildGenerateCommandArgs()));
        if (!this.options.mockDataOptions?.disable) {
            generateTask.exec(this.buildGenerateMockDataCommand());
        }
        // Copy the api spec to within the package
        generateTask.exec(`mkdir -p ${path.dirname(this.packagedSpecPath)}`);
        generateTask.exec(`cp -f ${this.options.specPath} ${this.packagedSpecPath}`);
        this.gitignore.addPatterns(`/${this.packagedSpecPath}`);
        this.preCompileTask.spawn(generateTask);
        // Ignore the generated code
        this.gitignore.addPatterns(this.srcdir, ".openapi-generator", "mocks");
        // If we're not in a monorepo, we need to link the generated types such that the local dependency can be resolved
        if (!options.isWithinMonorepo) {
            switch (this.package.packageManager) {
                case javascript_1.NodePackageManager.NPM:
                case javascript_1.NodePackageManager.YARN:
                case javascript_1.NodePackageManager.YARN2:
                    this.tasks
                        .tryFind("install")
                        ?.prependExec(`${this.package.packageManager} link ${this.options.generatedTypescriptTypes.package.packageName}`);
                    break;
                case javascript_1.NodePackageManager.PNPM:
                    this.tasks
                        .tryFind("install")
                        ?.prependExec(`${this.package.packageManager} link /${path.relative(this.outdir, this.options.generatedTypescriptTypes.outdir)}`);
                    break;
                default:
                    console.warn(`Unknown package manager ${this.package.packageManager}. Cannot link generated typescript client.`);
            }
        }
    }
}
exports.GeneratedTypescriptCdkInfrastructureProject = GeneratedTypescriptCdkInfrastructureProject;
//# sourceMappingURL=data:application/json;base64,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