"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.TypescriptReactQueryHooksLibrary = void 0;
/*! Copyright [Amazon.com](http://amazon.com/), Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0 */
const monorepo_1 = require("../../../../monorepo");
const javascript_1 = require("projen/lib/javascript");
const typescript_1 = require("projen/lib/typescript");
const languages_1 = require("../../languages");
const open_api_generator_handlebars_ignore_file_1 = require("../components/open-api-generator-handlebars-ignore-file");
const open_api_generator_ignore_file_1 = require("../components/open-api-generator-ignore-file");
const open_api_tools_json_file_1 = require("../components/open-api-tools-json-file");
const type_safe_api_command_environment_1 = require("../components/type-safe-api-command-environment");
const utils_1 = require("../components/utils");
/**
 * Typescript project containing generated react-query hooks
 */
class TypescriptReactQueryHooksLibrary extends typescript_1.TypeScriptProject {
    constructor(options) {
        super({
            ...options,
            sampleCode: false,
            tsconfig: {
                ...options.tsconfig,
                compilerOptions: {
                    jsx: javascript_1.TypeScriptJsxMode.REACT,
                    lib: ["dom", "es2019"],
                    // Generated code isn't very strict!
                    strict: false,
                    alwaysStrict: false,
                    noImplicitAny: false,
                    noImplicitReturns: false,
                    noImplicitThis: false,
                    noUnusedLocals: false,
                    noUnusedParameters: false,
                    strictNullChecks: false,
                    strictPropertyInitialization: false,
                    skipLibCheck: true,
                    ...options?.tsconfig?.compilerOptions,
                },
            },
            eslint: false,
            // Disable tests unless explicitly enabled
            jest: options.jest ?? false,
            npmignoreEnabled: false,
        });
        this.buildGenerateCommandArgs = () => {
            return (0, utils_1.buildInvokeOpenApiGeneratorCommandArgs)({
                generator: "typescript-fetch",
                specPath: this.options.specPath,
                generatorDirectory: languages_1.Library.TYPESCRIPT_REACT_QUERY_HOOKS,
                additionalProperties: {
                    npmName: this.package.packageName,
                    typescriptThreePlus: "true",
                    useSingleParameter: "true",
                    supportsES6: "true",
                },
                srcDir: this.srcdir,
                normalizers: {
                    KEEP_ONLY_FIRST_TAG_IN_OPERATION: true,
                },
            });
        };
        type_safe_api_command_environment_1.TypeSafeApiCommandEnvironment.ensure(this);
        this.options = options;
        // Disable strict peer dependencies for pnpm as the default typescript project dependencies have type mismatches
        // (ts-jest@27 and @types/jest@28)
        if (this.package.packageManager === javascript_1.NodePackageManager.PNPM) {
            this.npmrc.addConfig("strict-peer-dependencies", "false");
        }
        // Add dependencies on react-query and react
        this.addDeps("@tanstack/react-query");
        this.addDevDeps("react", "@types/react");
        this.addPeerDeps("react");
        // Tell OpenAPI Generator CLI not to generate files that we will generate via this project, or don't need.
        const openapiGeneratorIgnore = new open_api_generator_ignore_file_1.OpenApiGeneratorIgnoreFile(this);
        openapiGeneratorIgnore.addPatterns(...TypescriptReactQueryHooksLibrary.openApiIgnorePatterns);
        // Ignore the hooks, since they're generated with the handlebars templating engine
        const hooksPattern = "**/*Hooks.ts";
        openapiGeneratorIgnore.addPatterns(hooksPattern);
        // The hooks are generated using the handlebars templating engine, so we include a handlebars ignore file
        const handlebarsIgnore = new open_api_generator_handlebars_ignore_file_1.OpenApiGeneratorHandlebarsIgnoreFile(this);
        handlebarsIgnore.addPatterns(`!${hooksPattern}`);
        // Add OpenAPI Generator cli configuration
        open_api_tools_json_file_1.OpenApiToolsJsonFile.ensure(this).addOpenApiGeneratorCliConfig(options.openApiGeneratorCliConfig);
        const generateTask = this.addTask("generate");
        generateTask.exec((0, utils_1.buildCleanOpenApiGeneratedCodeCommand)());
        generateTask.exec((0, utils_1.buildTypeSafeApiExecCommand)(utils_1.TypeSafeApiScript.GENERATE, this.buildGenerateCommandArgs()));
        this.preCompileTask.spawn(generateTask);
        // Ignore all the generated code
        this.gitignore.addPatterns("src", ".npmignore", "README.md", ".openapi-generator");
        // If we're not in a monorepo, we need to link the generated client such that any local dependency on it can be
        // resolved
        if (!options.isWithinMonorepo) {
            switch (this.package.packageManager) {
                case javascript_1.NodePackageManager.NPM:
                case javascript_1.NodePackageManager.YARN:
                case javascript_1.NodePackageManager.YARN2:
                    this.tasks
                        .tryFind("install")
                        ?.exec(monorepo_1.NodePackageUtils.command.cmd(this.package.packageManager, "link"));
                    break;
                case javascript_1.NodePackageManager.PNPM:
                    // Nothing to do for pnpm, since the pnpm link command handles both the dependant and dependee
                    break;
                default:
                    console.warn(`Unknown package manager ${this.package.packageManager}. Cannot link generated typescript runtime project.`);
            }
        }
    }
}
exports.TypescriptReactQueryHooksLibrary = TypescriptReactQueryHooksLibrary;
/**
 * Patterns that are excluded from code generation
 */
TypescriptReactQueryHooksLibrary.openApiIgnorePatterns = [
    "package.json",
    "tsconfig.json",
    "tsconfig.esm.json",
    ".npmignore",
];
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoidHlwZXNjcmlwdC1yZWFjdC1xdWVyeS1ob29rcy1saWJyYXJ5LmpzIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsidHlwZXNjcmlwdC1yZWFjdC1xdWVyeS1ob29rcy1saWJyYXJ5LnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7OztBQUFBO3NDQUNzQztBQUN0Qyw0Q0FBaUQ7QUFDakQsc0RBQThFO0FBQzlFLHNEQUEwRDtBQUMxRCwrQ0FBMEM7QUFLMUMsdUhBQStHO0FBQy9HLGlHQUEwRjtBQUMxRixxRkFBOEU7QUFDOUUsdUdBQWdHO0FBQ2hHLCtDQUs2QjtBQWM3Qjs7R0FFRztBQUNILE1BQWEsZ0NBQWlDLFNBQVEsOEJBQWlCO0lBaUJyRSxZQUFZLE9BQXlEO1FBQ25FLEtBQUssQ0FBQztZQUNKLEdBQUksT0FBZTtZQUNuQixVQUFVLEVBQUUsS0FBSztZQUNqQixRQUFRLEVBQUU7Z0JBQ1IsR0FBRyxPQUFPLENBQUMsUUFBUTtnQkFDbkIsZUFBZSxFQUFFO29CQUNmLEdBQUcsRUFBRSw4QkFBaUIsQ0FBQyxLQUFLO29CQUM1QixHQUFHLEVBQUUsQ0FBQyxLQUFLLEVBQUUsUUFBUSxDQUFDO29CQUN0QixvQ0FBb0M7b0JBQ3BDLE1BQU0sRUFBRSxLQUFLO29CQUNiLFlBQVksRUFBRSxLQUFLO29CQUNuQixhQUFhLEVBQUUsS0FBSztvQkFDcEIsaUJBQWlCLEVBQUUsS0FBSztvQkFDeEIsY0FBYyxFQUFFLEtBQUs7b0JBQ3JCLGNBQWMsRUFBRSxLQUFLO29CQUNyQixrQkFBa0IsRUFBRSxLQUFLO29CQUN6QixnQkFBZ0IsRUFBRSxLQUFLO29CQUN2Qiw0QkFBNEIsRUFBRSxLQUFLO29CQUNuQyxZQUFZLEVBQUUsSUFBSTtvQkFDbEIsR0FBRyxPQUFPLEVBQUUsUUFBUSxFQUFFLGVBQWU7aUJBQ3RDO2FBQ0Y7WUFDRCxNQUFNLEVBQUUsS0FBSztZQUNiLDBDQUEwQztZQUMxQyxJQUFJLEVBQUUsT0FBTyxDQUFDLElBQUksSUFBSSxLQUFLO1lBQzNCLGdCQUFnQixFQUFFLEtBQUs7U0FDeEIsQ0FBQyxDQUFDO1FBNkVFLDZCQUF3QixHQUFHLEdBQUcsRUFBRTtZQUNyQyxPQUFPLElBQUEsOENBQXNDLEVBQUM7Z0JBQzVDLFNBQVMsRUFBRSxrQkFBa0I7Z0JBQzdCLFFBQVEsRUFBRSxJQUFJLENBQUMsT0FBTyxDQUFDLFFBQVE7Z0JBQy9CLGtCQUFrQixFQUFFLG1CQUFPLENBQUMsNEJBQTRCO2dCQUN4RCxvQkFBb0IsRUFBRTtvQkFDcEIsT0FBTyxFQUFFLElBQUksQ0FBQyxPQUFPLENBQUMsV0FBVztvQkFDakMsbUJBQW1CLEVBQUUsTUFBTTtvQkFDM0Isa0JBQWtCLEVBQUUsTUFBTTtvQkFDMUIsV0FBVyxFQUFFLE1BQU07aUJBQ3BCO2dCQUNELE1BQU0sRUFBRSxJQUFJLENBQUMsTUFBTTtnQkFDbkIsV0FBVyxFQUFFO29CQUNYLGdDQUFnQyxFQUFFLElBQUk7aUJBQ3ZDO2FBQ0YsQ0FBQyxDQUFDO1FBQ0wsQ0FBQyxDQUFDO1FBNUZBLGlFQUE2QixDQUFDLE1BQU0sQ0FBQyxJQUFJLENBQUMsQ0FBQztRQUUzQyxJQUFJLENBQUMsT0FBTyxHQUFHLE9BQU8sQ0FBQztRQUV2QixnSEFBZ0g7UUFDaEgsa0NBQWtDO1FBQ2xDLElBQUksSUFBSSxDQUFDLE9BQU8sQ0FBQyxjQUFjLEtBQUssK0JBQWtCLENBQUMsSUFBSSxFQUFFO1lBQzNELElBQUksQ0FBQyxLQUFLLENBQUMsU0FBUyxDQUFDLDBCQUEwQixFQUFFLE9BQU8sQ0FBQyxDQUFDO1NBQzNEO1FBRUQsNENBQTRDO1FBQzVDLElBQUksQ0FBQyxPQUFPLENBQUMsdUJBQXVCLENBQUMsQ0FBQztRQUN0QyxJQUFJLENBQUMsVUFBVSxDQUFDLE9BQU8sRUFBRSxjQUFjLENBQUMsQ0FBQztRQUN6QyxJQUFJLENBQUMsV0FBVyxDQUFDLE9BQU8sQ0FBQyxDQUFDO1FBRTFCLDBHQUEwRztRQUMxRyxNQUFNLHNCQUFzQixHQUFHLElBQUksMkRBQTBCLENBQUMsSUFBSSxDQUFDLENBQUM7UUFDcEUsc0JBQXNCLENBQUMsV0FBVyxDQUNoQyxHQUFHLGdDQUFnQyxDQUFDLHFCQUFxQixDQUMxRCxDQUFDO1FBQ0Ysa0ZBQWtGO1FBQ2xGLE1BQU0sWUFBWSxHQUFHLGNBQWMsQ0FBQztRQUNwQyxzQkFBc0IsQ0FBQyxXQUFXLENBQUMsWUFBWSxDQUFDLENBQUM7UUFFakQseUdBQXlHO1FBQ3pHLE1BQU0sZ0JBQWdCLEdBQUcsSUFBSSxnRkFBb0MsQ0FBQyxJQUFJLENBQUMsQ0FBQztRQUN4RSxnQkFBZ0IsQ0FBQyxXQUFXLENBQUMsSUFBSSxZQUFZLEVBQUUsQ0FBQyxDQUFDO1FBRWpELDBDQUEwQztRQUMxQywrQ0FBb0IsQ0FBQyxNQUFNLENBQUMsSUFBSSxDQUFDLENBQUMsNEJBQTRCLENBQzVELE9BQU8sQ0FBQyx5QkFBeUIsQ0FDbEMsQ0FBQztRQUVGLE1BQU0sWUFBWSxHQUFHLElBQUksQ0FBQyxPQUFPLENBQUMsVUFBVSxDQUFDLENBQUM7UUFDOUMsWUFBWSxDQUFDLElBQUksQ0FBQyxJQUFBLDZDQUFxQyxHQUFFLENBQUMsQ0FBQztRQUMzRCxZQUFZLENBQUMsSUFBSSxDQUNmLElBQUEsbUNBQTJCLEVBQ3pCLHlCQUFpQixDQUFDLFFBQVEsRUFDMUIsSUFBSSxDQUFDLHdCQUF3QixFQUFFLENBQ2hDLENBQ0YsQ0FBQztRQUVGLElBQUksQ0FBQyxjQUFjLENBQUMsS0FBSyxDQUFDLFlBQVksQ0FBQyxDQUFDO1FBRXhDLGdDQUFnQztRQUNoQyxJQUFJLENBQUMsU0FBUyxDQUFDLFdBQVcsQ0FDeEIsS0FBSyxFQUNMLFlBQVksRUFDWixXQUFXLEVBQ1gsb0JBQW9CLENBQ3JCLENBQUM7UUFFRiwrR0FBK0c7UUFDL0csV0FBVztRQUNYLElBQUksQ0FBQyxPQUFPLENBQUMsZ0JBQWdCLEVBQUU7WUFDN0IsUUFBUSxJQUFJLENBQUMsT0FBTyxDQUFDLGNBQWMsRUFBRTtnQkFDbkMsS0FBSywrQkFBa0IsQ0FBQyxHQUFHLENBQUM7Z0JBQzVCLEtBQUssK0JBQWtCLENBQUMsSUFBSSxDQUFDO2dCQUM3QixLQUFLLCtCQUFrQixDQUFDLEtBQUs7b0JBQzNCLElBQUksQ0FBQyxLQUFLO3lCQUNQLE9BQU8sQ0FBQyxTQUFTLENBQUM7d0JBQ25CLEVBQUUsSUFBSSxDQUNKLDJCQUFnQixDQUFDLE9BQU8sQ0FBQyxHQUFHLENBQUMsSUFBSSxDQUFDLE9BQU8sQ0FBQyxjQUFjLEVBQUUsTUFBTSxDQUFDLENBQ2xFLENBQUM7b0JBQ0osTUFBTTtnQkFDUixLQUFLLCtCQUFrQixDQUFDLElBQUk7b0JBQzFCLDhGQUE4RjtvQkFDOUYsTUFBTTtnQkFDUjtvQkFDRSxPQUFPLENBQUMsSUFBSSxDQUNWLDJCQUEyQixJQUFJLENBQUMsT0FBTyxDQUFDLGNBQWMscURBQXFELENBQzVHLENBQUM7YUFDTDtTQUNGO0lBQ0gsQ0FBQzs7QUF2SEgsNEVBMElDO0FBeklDOztHQUVHO0FBQ1csc0RBQXFCLEdBQWE7SUFDOUMsY0FBYztJQUNkLGVBQWU7SUFDZixtQkFBbUI7SUFDbkIsWUFBWTtDQUNiLEFBTGtDLENBS2pDIiwic291cmNlc0NvbnRlbnQiOlsiLyohIENvcHlyaWdodCBbQW1hem9uLmNvbV0oaHR0cDovL2FtYXpvbi5jb20vKSwgSW5jLiBvciBpdHMgYWZmaWxpYXRlcy4gQWxsIFJpZ2h0cyBSZXNlcnZlZC5cblNQRFgtTGljZW5zZS1JZGVudGlmaWVyOiBBcGFjaGUtMi4wICovXG5pbXBvcnQgeyBOb2RlUGFja2FnZVV0aWxzIH0gZnJvbSBcIkBhd3MvbW9ub3JlcG9cIjtcbmltcG9ydCB7IE5vZGVQYWNrYWdlTWFuYWdlciwgVHlwZVNjcmlwdEpzeE1vZGUgfSBmcm9tIFwicHJvamVuL2xpYi9qYXZhc2NyaXB0XCI7XG5pbXBvcnQgeyBUeXBlU2NyaXB0UHJvamVjdCB9IGZyb20gXCJwcm9qZW4vbGliL3R5cGVzY3JpcHRcIjtcbmltcG9ydCB7IExpYnJhcnkgfSBmcm9tIFwiLi4vLi4vbGFuZ3VhZ2VzXCI7XG5pbXBvcnQge1xuICBDb2RlR2VuZXJhdGlvblNvdXJjZU9wdGlvbnMsXG4gIEdlbmVyYXRlZFR5cGVTY3JpcHRSZWFjdFF1ZXJ5SG9va3NPcHRpb25zLFxufSBmcm9tIFwiLi4vLi4vdHlwZXNcIjtcbmltcG9ydCB7IE9wZW5BcGlHZW5lcmF0b3JIYW5kbGViYXJzSWdub3JlRmlsZSB9IGZyb20gXCIuLi9jb21wb25lbnRzL29wZW4tYXBpLWdlbmVyYXRvci1oYW5kbGViYXJzLWlnbm9yZS1maWxlXCI7XG5pbXBvcnQgeyBPcGVuQXBpR2VuZXJhdG9ySWdub3JlRmlsZSB9IGZyb20gXCIuLi9jb21wb25lbnRzL29wZW4tYXBpLWdlbmVyYXRvci1pZ25vcmUtZmlsZVwiO1xuaW1wb3J0IHsgT3BlbkFwaVRvb2xzSnNvbkZpbGUgfSBmcm9tIFwiLi4vY29tcG9uZW50cy9vcGVuLWFwaS10b29scy1qc29uLWZpbGVcIjtcbmltcG9ydCB7IFR5cGVTYWZlQXBpQ29tbWFuZEVudmlyb25tZW50IH0gZnJvbSBcIi4uL2NvbXBvbmVudHMvdHlwZS1zYWZlLWFwaS1jb21tYW5kLWVudmlyb25tZW50XCI7XG5pbXBvcnQge1xuICBidWlsZENsZWFuT3BlbkFwaUdlbmVyYXRlZENvZGVDb21tYW5kLFxuICBidWlsZEludm9rZU9wZW5BcGlHZW5lcmF0b3JDb21tYW5kQXJncyxcbiAgYnVpbGRUeXBlU2FmZUFwaUV4ZWNDb21tYW5kLFxuICBUeXBlU2FmZUFwaVNjcmlwdCxcbn0gZnJvbSBcIi4uL2NvbXBvbmVudHMvdXRpbHNcIjtcblxuLyoqXG4gKiBDb25maWd1cmF0aW9uIGZvciB0aGUgZ2VuZXJhdGVkIHR5cGVzY3JpcHQgY2xpZW50IHByb2plY3RcbiAqL1xuZXhwb3J0IGludGVyZmFjZSBHZW5lcmF0ZWRUeXBlc2NyaXB0UmVhY3RRdWVyeUhvb2tzUHJvamVjdE9wdGlvbnNcbiAgZXh0ZW5kcyBHZW5lcmF0ZWRUeXBlU2NyaXB0UmVhY3RRdWVyeUhvb2tzT3B0aW9ucyxcbiAgICBDb2RlR2VuZXJhdGlvblNvdXJjZU9wdGlvbnMge1xuICAvKipcbiAgICogV2hldGhlciB0aGlzIHByb2plY3QgaXMgcGFyZW50ZWQgYnkgYW4gbW9ub3JlcG8gb3Igbm90XG4gICAqL1xuICByZWFkb25seSBpc1dpdGhpbk1vbm9yZXBvPzogYm9vbGVhbjtcbn1cblxuLyoqXG4gKiBUeXBlc2NyaXB0IHByb2plY3QgY29udGFpbmluZyBnZW5lcmF0ZWQgcmVhY3QtcXVlcnkgaG9va3NcbiAqL1xuZXhwb3J0IGNsYXNzIFR5cGVzY3JpcHRSZWFjdFF1ZXJ5SG9va3NMaWJyYXJ5IGV4dGVuZHMgVHlwZVNjcmlwdFByb2plY3Qge1xuICAvKipcbiAgICogUGF0dGVybnMgdGhhdCBhcmUgZXhjbHVkZWQgZnJvbSBjb2RlIGdlbmVyYXRpb25cbiAgICovXG4gIHB1YmxpYyBzdGF0aWMgb3BlbkFwaUlnbm9yZVBhdHRlcm5zOiBzdHJpbmdbXSA9IFtcbiAgICBcInBhY2thZ2UuanNvblwiLFxuICAgIFwidHNjb25maWcuanNvblwiLFxuICAgIFwidHNjb25maWcuZXNtLmpzb25cIixcbiAgICBcIi5ucG1pZ25vcmVcIixcbiAgXTtcblxuICAvKipcbiAgICogT3B0aW9ucyBjb25maWd1cmVkIGZvciB0aGUgcHJvamVjdFxuICAgKiBAcHJpdmF0ZVxuICAgKi9cbiAgcHJpdmF0ZSByZWFkb25seSBvcHRpb25zOiBHZW5lcmF0ZWRUeXBlc2NyaXB0UmVhY3RRdWVyeUhvb2tzUHJvamVjdE9wdGlvbnM7XG5cbiAgY29uc3RydWN0b3Iob3B0aW9uczogR2VuZXJhdGVkVHlwZXNjcmlwdFJlYWN0UXVlcnlIb29rc1Byb2plY3RPcHRpb25zKSB7XG4gICAgc3VwZXIoe1xuICAgICAgLi4uKG9wdGlvbnMgYXMgYW55KSxcbiAgICAgIHNhbXBsZUNvZGU6IGZhbHNlLFxuICAgICAgdHNjb25maWc6IHtcbiAgICAgICAgLi4ub3B0aW9ucy50c2NvbmZpZyxcbiAgICAgICAgY29tcGlsZXJPcHRpb25zOiB7XG4gICAgICAgICAganN4OiBUeXBlU2NyaXB0SnN4TW9kZS5SRUFDVCxcbiAgICAgICAgICBsaWI6IFtcImRvbVwiLCBcImVzMjAxOVwiXSxcbiAgICAgICAgICAvLyBHZW5lcmF0ZWQgY29kZSBpc24ndCB2ZXJ5IHN0cmljdCFcbiAgICAgICAgICBzdHJpY3Q6IGZhbHNlLFxuICAgICAgICAgIGFsd2F5c1N0cmljdDogZmFsc2UsXG4gICAgICAgICAgbm9JbXBsaWNpdEFueTogZmFsc2UsXG4gICAgICAgICAgbm9JbXBsaWNpdFJldHVybnM6IGZhbHNlLFxuICAgICAgICAgIG5vSW1wbGljaXRUaGlzOiBmYWxzZSxcbiAgICAgICAgICBub1VudXNlZExvY2FsczogZmFsc2UsXG4gICAgICAgICAgbm9VbnVzZWRQYXJhbWV0ZXJzOiBmYWxzZSxcbiAgICAgICAgICBzdHJpY3ROdWxsQ2hlY2tzOiBmYWxzZSxcbiAgICAgICAgICBzdHJpY3RQcm9wZXJ0eUluaXRpYWxpemF0aW9uOiBmYWxzZSxcbiAgICAgICAgICBza2lwTGliQ2hlY2s6IHRydWUsXG4gICAgICAgICAgLi4ub3B0aW9ucz8udHNjb25maWc/LmNvbXBpbGVyT3B0aW9ucyxcbiAgICAgICAgfSxcbiAgICAgIH0sXG4gICAgICBlc2xpbnQ6IGZhbHNlLFxuICAgICAgLy8gRGlzYWJsZSB0ZXN0cyB1bmxlc3MgZXhwbGljaXRseSBlbmFibGVkXG4gICAgICBqZXN0OiBvcHRpb25zLmplc3QgPz8gZmFsc2UsXG4gICAgICBucG1pZ25vcmVFbmFibGVkOiBmYWxzZSxcbiAgICB9KTtcbiAgICBUeXBlU2FmZUFwaUNvbW1hbmRFbnZpcm9ubWVudC5lbnN1cmUodGhpcyk7XG5cbiAgICB0aGlzLm9wdGlvbnMgPSBvcHRpb25zO1xuXG4gICAgLy8gRGlzYWJsZSBzdHJpY3QgcGVlciBkZXBlbmRlbmNpZXMgZm9yIHBucG0gYXMgdGhlIGRlZmF1bHQgdHlwZXNjcmlwdCBwcm9qZWN0IGRlcGVuZGVuY2llcyBoYXZlIHR5cGUgbWlzbWF0Y2hlc1xuICAgIC8vICh0cy1qZXN0QDI3IGFuZCBAdHlwZXMvamVzdEAyOClcbiAgICBpZiAodGhpcy5wYWNrYWdlLnBhY2thZ2VNYW5hZ2VyID09PSBOb2RlUGFja2FnZU1hbmFnZXIuUE5QTSkge1xuICAgICAgdGhpcy5ucG1yYy5hZGRDb25maWcoXCJzdHJpY3QtcGVlci1kZXBlbmRlbmNpZXNcIiwgXCJmYWxzZVwiKTtcbiAgICB9XG5cbiAgICAvLyBBZGQgZGVwZW5kZW5jaWVzIG9uIHJlYWN0LXF1ZXJ5IGFuZCByZWFjdFxuICAgIHRoaXMuYWRkRGVwcyhcIkB0YW5zdGFjay9yZWFjdC1xdWVyeVwiKTtcbiAgICB0aGlzLmFkZERldkRlcHMoXCJyZWFjdFwiLCBcIkB0eXBlcy9yZWFjdFwiKTtcbiAgICB0aGlzLmFkZFBlZXJEZXBzKFwicmVhY3RcIik7XG5cbiAgICAvLyBUZWxsIE9wZW5BUEkgR2VuZXJhdG9yIENMSSBub3QgdG8gZ2VuZXJhdGUgZmlsZXMgdGhhdCB3ZSB3aWxsIGdlbmVyYXRlIHZpYSB0aGlzIHByb2plY3QsIG9yIGRvbid0IG5lZWQuXG4gICAgY29uc3Qgb3BlbmFwaUdlbmVyYXRvcklnbm9yZSA9IG5ldyBPcGVuQXBpR2VuZXJhdG9ySWdub3JlRmlsZSh0aGlzKTtcbiAgICBvcGVuYXBpR2VuZXJhdG9ySWdub3JlLmFkZFBhdHRlcm5zKFxuICAgICAgLi4uVHlwZXNjcmlwdFJlYWN0UXVlcnlIb29rc0xpYnJhcnkub3BlbkFwaUlnbm9yZVBhdHRlcm5zXG4gICAgKTtcbiAgICAvLyBJZ25vcmUgdGhlIGhvb2tzLCBzaW5jZSB0aGV5J3JlIGdlbmVyYXRlZCB3aXRoIHRoZSBoYW5kbGViYXJzIHRlbXBsYXRpbmcgZW5naW5lXG4gICAgY29uc3QgaG9va3NQYXR0ZXJuID0gXCIqKi8qSG9va3MudHNcIjtcbiAgICBvcGVuYXBpR2VuZXJhdG9ySWdub3JlLmFkZFBhdHRlcm5zKGhvb2tzUGF0dGVybik7XG5cbiAgICAvLyBUaGUgaG9va3MgYXJlIGdlbmVyYXRlZCB1c2luZyB0aGUgaGFuZGxlYmFycyB0ZW1wbGF0aW5nIGVuZ2luZSwgc28gd2UgaW5jbHVkZSBhIGhhbmRsZWJhcnMgaWdub3JlIGZpbGVcbiAgICBjb25zdCBoYW5kbGViYXJzSWdub3JlID0gbmV3IE9wZW5BcGlHZW5lcmF0b3JIYW5kbGViYXJzSWdub3JlRmlsZSh0aGlzKTtcbiAgICBoYW5kbGViYXJzSWdub3JlLmFkZFBhdHRlcm5zKGAhJHtob29rc1BhdHRlcm59YCk7XG5cbiAgICAvLyBBZGQgT3BlbkFQSSBHZW5lcmF0b3IgY2xpIGNvbmZpZ3VyYXRpb25cbiAgICBPcGVuQXBpVG9vbHNKc29uRmlsZS5lbnN1cmUodGhpcykuYWRkT3BlbkFwaUdlbmVyYXRvckNsaUNvbmZpZyhcbiAgICAgIG9wdGlvbnMub3BlbkFwaUdlbmVyYXRvckNsaUNvbmZpZ1xuICAgICk7XG5cbiAgICBjb25zdCBnZW5lcmF0ZVRhc2sgPSB0aGlzLmFkZFRhc2soXCJnZW5lcmF0ZVwiKTtcbiAgICBnZW5lcmF0ZVRhc2suZXhlYyhidWlsZENsZWFuT3BlbkFwaUdlbmVyYXRlZENvZGVDb21tYW5kKCkpO1xuICAgIGdlbmVyYXRlVGFzay5leGVjKFxuICAgICAgYnVpbGRUeXBlU2FmZUFwaUV4ZWNDb21tYW5kKFxuICAgICAgICBUeXBlU2FmZUFwaVNjcmlwdC5HRU5FUkFURSxcbiAgICAgICAgdGhpcy5idWlsZEdlbmVyYXRlQ29tbWFuZEFyZ3MoKVxuICAgICAgKVxuICAgICk7XG5cbiAgICB0aGlzLnByZUNvbXBpbGVUYXNrLnNwYXduKGdlbmVyYXRlVGFzayk7XG5cbiAgICAvLyBJZ25vcmUgYWxsIHRoZSBnZW5lcmF0ZWQgY29kZVxuICAgIHRoaXMuZ2l0aWdub3JlLmFkZFBhdHRlcm5zKFxuICAgICAgXCJzcmNcIixcbiAgICAgIFwiLm5wbWlnbm9yZVwiLFxuICAgICAgXCJSRUFETUUubWRcIixcbiAgICAgIFwiLm9wZW5hcGktZ2VuZXJhdG9yXCJcbiAgICApO1xuXG4gICAgLy8gSWYgd2UncmUgbm90IGluIGEgbW9ub3JlcG8sIHdlIG5lZWQgdG8gbGluayB0aGUgZ2VuZXJhdGVkIGNsaWVudCBzdWNoIHRoYXQgYW55IGxvY2FsIGRlcGVuZGVuY3kgb24gaXQgY2FuIGJlXG4gICAgLy8gcmVzb2x2ZWRcbiAgICBpZiAoIW9wdGlvbnMuaXNXaXRoaW5Nb25vcmVwbykge1xuICAgICAgc3dpdGNoICh0aGlzLnBhY2thZ2UucGFja2FnZU1hbmFnZXIpIHtcbiAgICAgICAgY2FzZSBOb2RlUGFja2FnZU1hbmFnZXIuTlBNOlxuICAgICAgICBjYXNlIE5vZGVQYWNrYWdlTWFuYWdlci5ZQVJOOlxuICAgICAgICBjYXNlIE5vZGVQYWNrYWdlTWFuYWdlci5ZQVJOMjpcbiAgICAgICAgICB0aGlzLnRhc2tzXG4gICAgICAgICAgICAudHJ5RmluZChcImluc3RhbGxcIilcbiAgICAgICAgICAgID8uZXhlYyhcbiAgICAgICAgICAgICAgTm9kZVBhY2thZ2VVdGlscy5jb21tYW5kLmNtZCh0aGlzLnBhY2thZ2UucGFja2FnZU1hbmFnZXIsIFwibGlua1wiKVxuICAgICAgICAgICAgKTtcbiAgICAgICAgICBicmVhaztcbiAgICAgICAgY2FzZSBOb2RlUGFja2FnZU1hbmFnZXIuUE5QTTpcbiAgICAgICAgICAvLyBOb3RoaW5nIHRvIGRvIGZvciBwbnBtLCBzaW5jZSB0aGUgcG5wbSBsaW5rIGNvbW1hbmQgaGFuZGxlcyBib3RoIHRoZSBkZXBlbmRhbnQgYW5kIGRlcGVuZGVlXG4gICAgICAgICAgYnJlYWs7XG4gICAgICAgIGRlZmF1bHQ6XG4gICAgICAgICAgY29uc29sZS53YXJuKFxuICAgICAgICAgICAgYFVua25vd24gcGFja2FnZSBtYW5hZ2VyICR7dGhpcy5wYWNrYWdlLnBhY2thZ2VNYW5hZ2VyfS4gQ2Fubm90IGxpbmsgZ2VuZXJhdGVkIHR5cGVzY3JpcHQgcnVudGltZSBwcm9qZWN0LmBcbiAgICAgICAgICApO1xuICAgICAgfVxuICAgIH1cbiAgfVxuXG4gIHB1YmxpYyBidWlsZEdlbmVyYXRlQ29tbWFuZEFyZ3MgPSAoKSA9PiB7XG4gICAgcmV0dXJuIGJ1aWxkSW52b2tlT3BlbkFwaUdlbmVyYXRvckNvbW1hbmRBcmdzKHtcbiAgICAgIGdlbmVyYXRvcjogXCJ0eXBlc2NyaXB0LWZldGNoXCIsXG4gICAgICBzcGVjUGF0aDogdGhpcy5vcHRpb25zLnNwZWNQYXRoLFxuICAgICAgZ2VuZXJhdG9yRGlyZWN0b3J5OiBMaWJyYXJ5LlRZUEVTQ1JJUFRfUkVBQ1RfUVVFUllfSE9PS1MsXG4gICAgICBhZGRpdGlvbmFsUHJvcGVydGllczoge1xuICAgICAgICBucG1OYW1lOiB0aGlzLnBhY2thZ2UucGFja2FnZU5hbWUsXG4gICAgICAgIHR5cGVzY3JpcHRUaHJlZVBsdXM6IFwidHJ1ZVwiLFxuICAgICAgICB1c2VTaW5nbGVQYXJhbWV0ZXI6IFwidHJ1ZVwiLFxuICAgICAgICBzdXBwb3J0c0VTNjogXCJ0cnVlXCIsXG4gICAgICB9LFxuICAgICAgc3JjRGlyOiB0aGlzLnNyY2RpcixcbiAgICAgIG5vcm1hbGl6ZXJzOiB7XG4gICAgICAgIEtFRVBfT05MWV9GSVJTVF9UQUdfSU5fT1BFUkFUSU9OOiB0cnVlLFxuICAgICAgfSxcbiAgICB9KTtcbiAgfTtcbn1cbiJdfQ==