"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.GeneratedJavaRuntimeProject = void 0;
/*! Copyright [Amazon.com](http://amazon.com/), Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0 */
const path = require("path");
const java_1 = require("projen/lib/java");
const languages_1 = require("../../languages");
const open_api_generator_ignore_file_1 = require("../components/open-api-generator-ignore-file");
const open_api_tools_json_file_1 = require("../components/open-api-tools-json-file");
const type_safe_api_command_environment_1 = require("../components/type-safe-api-command-environment");
const utils_1 = require("../components/utils");
const DEPENDENCIES = [
    // Required for open api generated code
    "io.swagger/swagger-annotations@1.6.8",
    "com.google.code.findbugs/jsr305@3.0.2",
    "com.squareup.okhttp3/okhttp@4.10.0",
    "com.squareup.okhttp3/logging-interceptor@4.10.0",
    "com.google.code.gson/gson@2.9.1",
    "io.gsonfire/gson-fire@1.8.5",
    "org.apache.commons/commons-lang3@3.12.0",
    "jakarta.annotation/jakarta.annotation-api@1.3.5",
    "org.openapitools/jackson-databind-nullable@0.2.4",
    "javax.ws.rs/jsr311-api@1.1.1",
    "javax.ws.rs/javax.ws.rs-api@2.1.1",
    // For handler wrappers
    "com.amazonaws/aws-lambda-java-core@1.2.1",
    "com.amazonaws/aws-lambda-java-events@3.11.0",
    // Lombok is used to add the builder pattern to models for neater construction
    "org.projectlombok/lombok@1.18.24",
    // Interceptors
    "software.amazon.lambda/powertools-logging@^1.16.1",
    "software.amazon.lambda/powertools-tracing@^1.16.1",
    "software.amazon.lambda/powertools-metrics@^1.16.1",
];
const TEST_DEPENDENCIES = [
    "org.junit.jupiter/junit-jupiter-api@5.9.1",
    "org.mockito/mockito-core@3.12.4",
];
/**
 * Java project containing types generated using OpenAPI Generator CLI
 */
class GeneratedJavaRuntimeProject extends java_1.JavaProject {
    constructor(options) {
        super({
            ...options,
            sample: false,
            junit: false,
        });
        this.buildGenerateCommandArgs = () => {
            return (0, utils_1.buildInvokeOpenApiGeneratorCommandArgs)({
                generator: "java",
                specPath: this.options.specPath,
                generatorDirectory: languages_1.Language.JAVA,
                additionalProperties: {
                    useSingleRequestParameter: "true",
                    groupId: this.pom.groupId,
                    artifactId: this.pom.artifactId,
                    artifactVersion: this.pom.version,
                    invokerPackage: this.packageName,
                    apiPackage: `${this.packageName}.api`,
                    modelPackage: `${this.packageName}.model`,
                    hideGenerationTimestamp: "true",
                    additionalModelTypeAnnotations: [
                        "@lombok.AllArgsConstructor",
                        // Regular lombok builder is not used since an abstract base schema class is also annotated
                        "@lombok.experimental.SuperBuilder",
                    ].join("\\ "),
                },
                srcDir: path.join("src", "main", "java", ...this.packageName.split(".")),
                normalizers: {
                    KEEP_ONLY_FIRST_TAG_IN_OPERATION: true,
                },
                // Do not generate map/list types. Generator will use built in HashMap, ArrayList instead
                generateAliasAsModel: false,
            });
        };
        type_safe_api_command_environment_1.TypeSafeApiCommandEnvironment.ensure(this);
        this.options = options;
        // Ignore files that we will control via projen
        const ignoreFile = new open_api_generator_ignore_file_1.OpenApiGeneratorIgnoreFile(this);
        ignoreFile.addPatterns(...GeneratedJavaRuntimeProject.openApiIgnorePatterns);
        // Add OpenAPI Generator cli configuration
        open_api_tools_json_file_1.OpenApiToolsJsonFile.ensure(this).addOpenApiGeneratorCliConfig(options.openApiGeneratorCliConfig);
        // Add dependencies
        DEPENDENCIES.forEach((dep) => this.addDependency(dep));
        TEST_DEPENDENCIES.forEach((dep) => this.addTestDependency(dep));
        this.packageName = `${this.pom.groupId}.${this.name}.runtime`;
        // Generate the java code
        const generateTask = this.addTask("generate");
        generateTask.exec((0, utils_1.buildCleanOpenApiGeneratedCodeCommand)());
        generateTask.exec((0, utils_1.buildTypeSafeApiExecCommand)(utils_1.TypeSafeApiScript.GENERATE, this.buildGenerateCommandArgs()));
        this.preCompileTask.spawn(generateTask);
        // Ignore all the generated code
        this.gitignore.addPatterns("src", "docs", "api", "README.md", ".openapi-generator");
    }
}
exports.GeneratedJavaRuntimeProject = GeneratedJavaRuntimeProject;
/**
 * Patterns that are excluded from code generation
 */
GeneratedJavaRuntimeProject.openApiIgnorePatterns = [
    "pom.xml",
    "gradle",
    "gradle/**/*",
    "gradlew",
    "gradle.properties",
    "gradlew.bat",
    "build.gradle",
    "settings.gradle",
    "build.sbt",
    ".travis.yml",
    "git_push.sh",
    "src/test",
    "src/test/**/*",
    "src/main/AndroidManifest.xml",
];
//# sourceMappingURL=data:application/json;base64,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