"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.GeneratedPythonRuntimeProject = void 0;
/*! Copyright [Amazon.com](http://amazon.com/), Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0 */
const python_1 = require("projen/lib/python");
const languages_1 = require("../../languages");
const open_api_generator_ignore_file_1 = require("../components/open-api-generator-ignore-file");
const open_api_tools_json_file_1 = require("../components/open-api-tools-json-file");
const type_safe_api_command_environment_1 = require("../components/type-safe-api-command-environment");
const utils_1 = require("../components/utils");
/**
 * Python project containing types generated using OpenAPI Generator CLI
 */
class GeneratedPythonRuntimeProject extends python_1.PythonProject {
    constructor(options) {
        super({
            ...options,
            sample: false,
            pytest: false,
            poetry: true,
            poetryOptions: {
                packages: [{ include: options.moduleName }],
                // Module must be explicitly added to include since poetry excludes everything in .gitignore by default
                include: [options.moduleName, `${options.moduleName}/**/*.py`],
            },
        });
        this.buildGenerateCommandArgs = () => {
            return (0, utils_1.buildInvokeOpenApiGeneratorCommandArgs)({
                generator: "python-nextgen",
                specPath: this.options.specPath,
                generatorDirectory: languages_1.Language.PYTHON,
                additionalProperties: {
                    packageName: this.moduleName,
                    projectName: this.name,
                },
                // Tell the generator where python source files live
                srcDir: this.moduleName,
                normalizers: {
                    KEEP_ONLY_FIRST_TAG_IN_OPERATION: true,
                },
            });
        };
        type_safe_api_command_environment_1.TypeSafeApiCommandEnvironment.ensure(this);
        this.options = options;
        // Add dependencies required by the client
        [
            "python-dateutil@~2.8.2",
            "pydantic@^1.10.5",
            "aenum@^3.1.11",
            "urllib3@~1.26.7",
            `aws-lambda-powertools@{extras=["all"],version="^2.23.0"}`,
            "python@^3.9",
        ].forEach((dep) => this.addDependency(dep));
        const openapiGeneratorIgnore = new open_api_generator_ignore_file_1.OpenApiGeneratorIgnoreFile(this);
        openapiGeneratorIgnore.addPatterns(...GeneratedPythonRuntimeProject.openApiIgnorePatterns);
        // Add OpenAPI Generator cli configuration
        open_api_tools_json_file_1.OpenApiToolsJsonFile.ensure(this).addOpenApiGeneratorCliConfig({
            version: "6.6.0",
            ...options.openApiGeneratorCliConfig,
        });
        const generateTask = this.addTask("generate");
        generateTask.exec((0, utils_1.buildCleanOpenApiGeneratedCodeCommand)());
        generateTask.exec((0, utils_1.buildTypeSafeApiExecCommand)(utils_1.TypeSafeApiScript.GENERATE, this.buildGenerateCommandArgs()));
        this.preCompileTask.spawn(generateTask);
        // Ignore all the generated code
        this.gitignore.addPatterns(this.moduleName, "docs", "README.md", ".openapi-generator");
        // The poetry install that runs as part of post synthesis expects there to be some code present, but code isn't
        // generated until build time. This means that the first install will fail when either generating the project for
        // the first time or checking out a fresh copy (since generated code is not checked in to version control). We
        // therefore add a blank __init__.py as our first install step to keep poetry happy until the generator overwrites
        // it.
        this.tasks
            .tryFind("install")
            ?.prependExec(`mkdir -p ${this.moduleName} && touch ${this.moduleName}/__init__.py`);
    }
}
exports.GeneratedPythonRuntimeProject = GeneratedPythonRuntimeProject;
/**
 * Patterns that are excluded from code generation
 */
GeneratedPythonRuntimeProject.openApiIgnorePatterns = [
    "test",
    "test/*",
    "test/**/*",
    ".github",
    ".github/*",
    ".github/**/*",
    ".gitlab-ci.yml",
    ".travis.yml",
    "git_push.sh",
    "tox.ini",
    "requirements.txt",
    "test-requirements.txt",
    "setup.py",
    "setup.cfg",
    "pyproject.toml",
];
//# sourceMappingURL=data:application/json;base64,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