"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.GeneratedTypescriptRuntimeProject = void 0;
/*! Copyright [Amazon.com](http://amazon.com/), Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0 */
const monorepo_1 = require("../../../../monorepo");
const projen_1 = require("projen");
const javascript_1 = require("projen/lib/javascript");
const typescript_1 = require("projen/lib/typescript");
const languages_1 = require("../../languages");
const open_api_generator_ignore_file_1 = require("../components/open-api-generator-ignore-file");
const open_api_tools_json_file_1 = require("../components/open-api-tools-json-file");
const type_safe_api_command_environment_1 = require("../components/type-safe-api-command-environment");
const utils_1 = require("../components/utils");
/**
 * Typescript project containing types generated using OpenAPI Generator CLI
 */
class GeneratedTypescriptRuntimeProject extends typescript_1.TypeScriptProject {
    constructor(options) {
        super({
            ...options,
            sampleCode: false,
            tsconfig: {
                ...options.tsconfig,
                compilerOptions: {
                    lib: ["dom", "es2019"],
                    // Generated code isn't very strict!
                    strict: false,
                    alwaysStrict: false,
                    noImplicitAny: false,
                    noImplicitReturns: false,
                    noImplicitThis: false,
                    noUnusedLocals: false,
                    noUnusedParameters: false,
                    strictNullChecks: false,
                    strictPropertyInitialization: false,
                    skipLibCheck: true,
                    ...options?.tsconfig?.compilerOptions,
                },
            },
            eslint: false,
            // Disable tests unless explicitly enabled
            jest: options.jest ?? false,
            npmignoreEnabled: false,
        });
        this.buildGenerateCommandArgs = () => {
            return (0, utils_1.buildInvokeOpenApiGeneratorCommandArgs)({
                generator: "typescript-fetch",
                specPath: this.options.specPath,
                generatorDirectory: languages_1.Language.TYPESCRIPT,
                additionalProperties: {
                    npmName: this.package.packageName,
                    typescriptThreePlus: "true",
                    useSingleParameter: "true",
                    supportsES6: "true",
                },
                srcDir: this.srcdir,
                normalizers: {
                    KEEP_ONLY_FIRST_TAG_IN_OPERATION: true,
                },
            });
        };
        type_safe_api_command_environment_1.TypeSafeApiCommandEnvironment.ensure(this);
        this.options = options;
        // Disable strict peer dependencies for pnpm as the default typescript project dependencies have type mismatches
        // (ts-jest@27 and @types/jest@28)
        if (this.package.packageManager === javascript_1.NodePackageManager.PNPM) {
            this.npmrc.addConfig("strict-peer-dependencies", "false");
        }
        // For event and context types
        this.addDeps("@types/aws-lambda", "@aws-lambda-powertools/tracer", "@aws-lambda-powertools/logger", "@aws-lambda-powertools/metrics");
        // Minimal .npmignore to avoid impacting OpenAPI Generator
        const npmignore = new projen_1.IgnoreFile(this, ".npmignore");
        npmignore.addPatterns("/.projen/", "/src", "/dist");
        // Tell OpenAPI Generator CLI not to generate files that we will generate via this project, or don't need.
        const openapiGeneratorIgnore = new open_api_generator_ignore_file_1.OpenApiGeneratorIgnoreFile(this);
        openapiGeneratorIgnore.addPatterns(...GeneratedTypescriptRuntimeProject.openApiIgnorePatterns);
        // Add OpenAPI Generator cli configuration
        open_api_tools_json_file_1.OpenApiToolsJsonFile.ensure(this).addOpenApiGeneratorCliConfig(options.openApiGeneratorCliConfig);
        const generateTask = this.addTask("generate");
        generateTask.exec((0, utils_1.buildCleanOpenApiGeneratedCodeCommand)());
        generateTask.exec((0, utils_1.buildTypeSafeApiExecCommand)(utils_1.TypeSafeApiScript.GENERATE, this.buildGenerateCommandArgs()));
        this.preCompileTask.spawn(generateTask);
        // Ignore all the generated code
        this.gitignore.addPatterns("src", ".npmignore", "README.md", ".openapi-generator");
        // If we're not in a monorepo, we need to link the generated client such that any local dependency on it can be
        // resolved
        if (!options.isWithinMonorepo) {
            switch (this.package.packageManager) {
                case javascript_1.NodePackageManager.NPM:
                case javascript_1.NodePackageManager.YARN:
                case javascript_1.NodePackageManager.YARN2:
                    this.tasks
                        .tryFind("install")
                        ?.exec(monorepo_1.NodePackageUtils.command.cmd(this.package.packageManager, "link"));
                    break;
                case javascript_1.NodePackageManager.PNPM:
                    // Nothing to do for pnpm, since the pnpm link command handles both the dependant and dependee
                    break;
                default:
                    console.warn(`Unknown package manager ${this.package.packageManager}. Cannot link generated typescript runtime project.`);
            }
        }
    }
}
exports.GeneratedTypescriptRuntimeProject = GeneratedTypescriptRuntimeProject;
/**
 * Patterns that are excluded from code generation
 */
GeneratedTypescriptRuntimeProject.openApiIgnorePatterns = [
    "package.json",
    "tsconfig.json",
    "tsconfig.esm.json",
    ".npmignore",
];
//# sourceMappingURL=data:application/json;base64,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