"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.ServerlessClamscan = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
// Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
// SPDX-License-Identifier: Apache-2.0
const path = require("path");
const aws_ec2_1 = require("monocdk/aws-ec2");
const aws_efs_1 = require("monocdk/aws-efs");
const aws_events_1 = require("monocdk/aws-events");
const aws_events_targets_1 = require("monocdk/aws-events-targets");
const aws_iam_1 = require("monocdk/aws-iam");
const aws_lambda_1 = require("monocdk/aws-lambda");
const aws_lambda_destinations_1 = require("monocdk/aws-lambda-destinations");
const aws_s3_1 = require("monocdk/aws-s3");
const aws_s3_notifications_1 = require("monocdk/aws-s3-notifications");
const aws_sqs_1 = require("monocdk/aws-sqs");
const monocdk_1 = require("monocdk");
const monocdk_nag_1 = require("monocdk-nag");
/**
  An [aws-cdk](https://github.com/aws/aws-cdk) construct that uses [ClamAV®](https://www.clamav.net/).
  to scan objects in Amazon S3 for viruses. The construct provides a flexible interface for a system
  to act based on the results of a ClamAV virus scan.

  The construct creates a Lambda function with EFS integration to support larger files.
  A VPC with isolated subnets, a S3 Gateway endpoint will also be created.

  Additionally creates an twice-daily job to download the latest ClamAV definition files to the
  Virus Definitions S3 Bucket by utilizing an EventBridge rule and a Lambda function and
  publishes CloudWatch Metrics to the 'serverless-clamscan' namespace.

  __Important O&M__:
  When ClamAV publishes updates to the scanner you will see “Your ClamAV installation is OUTDATED” in your scan results.
  While the construct creates a system to keep the database definitions up to date, you must update the scanner to
  detect all the latest Viruses.

  Update the docker images of the Lambda functions with the latest version of ClamAV by re-running `cdk deploy`.

  Successful Scan Event format
  ```json
  {
     "source": "serverless-clamscan",
     "input_bucket": <input_bucket_name>,
     "input_key": <object_key>,
     "status": <"CLEAN"|"INFECTED"|"N/A">,
     "message": <scan_summary>,
   }
  ```

  Note: The Virus Definitions bucket policy will likely cause a deletion error if you choose to delete
  the stack associated in the construct. However since the bucket itself gets deleted, you can delete
  the stack again to resolve the error.
 */
class ServerlessClamscan extends monocdk_1.Construct {
    /**
     * Creates a ServerlessClamscan construct.
     * @param scope The parent creating construct (usually `this`).
     * @param id The construct's name.
     * @param props A `ServerlessClamscanProps` interface.
     */
    constructor(scope, id, props) {
        var _b, _c;
        super(scope, id);
        this._efsRootPath = '/lambda';
        this._efsMountPath = `/mnt${this._efsRootPath}`;
        this._efsDefsPath = 'virus_database/';
        this.useImportedBuckets = props.acceptResponsibilityForUsingImportedBucket;
        if (!props.onResult) {
            this.resultBus = new aws_events_1.EventBus(this, 'ScanResultBus');
            this.resultDest = new aws_lambda_destinations_1.EventBridgeDestination(this.resultBus);
            this.infectedRule = new aws_events_1.Rule(this, 'InfectedRule', {
                eventBus: this.resultBus,
                description: 'Event for when a file is marked INFECTED',
                eventPattern: {
                    detail: {
                        responsePayload: {
                            source: ['serverless-clamscan'],
                            status: ['INFECTED'],
                        },
                    },
                },
            });
            this.cleanRule = new aws_events_1.Rule(this, 'CleanRule', {
                eventBus: this.resultBus,
                description: 'Event for when a file is marked CLEAN',
                eventPattern: {
                    detail: {
                        responsePayload: {
                            source: ['serverless-clamscan'],
                            status: ['CLEAN'],
                        },
                    },
                },
            });
        }
        else {
            this.resultDest = props.onResult;
        }
        if (!props.onError) {
            this.errorDeadLetterQueue = new aws_sqs_1.Queue(this, 'ScanErrorDeadLetterQueue', {
                encryption: aws_sqs_1.QueueEncryption.KMS_MANAGED,
            });
            this.errorDeadLetterQueue.addToResourcePolicy(new aws_iam_1.PolicyStatement({
                actions: ['sqs:*'],
                effect: aws_iam_1.Effect.DENY,
                principals: [new aws_iam_1.AnyPrincipal()],
                conditions: { Bool: { 'aws:SecureTransport': false } },
                resources: [this.errorDeadLetterQueue.queueArn],
            }));
            this.errorQueue = new aws_sqs_1.Queue(this, 'ScanErrorQueue', {
                encryption: aws_sqs_1.QueueEncryption.KMS_MANAGED,
                deadLetterQueue: {
                    maxReceiveCount: 3,
                    queue: this.errorDeadLetterQueue,
                },
            });
            this.errorQueue.addToResourcePolicy(new aws_iam_1.PolicyStatement({
                actions: ['sqs:*'],
                effect: aws_iam_1.Effect.DENY,
                principals: [new aws_iam_1.AnyPrincipal()],
                conditions: { Bool: { 'aws:SecureTransport': false } },
                resources: [this.errorQueue.queueArn],
            }));
            this.errorDest = new aws_lambda_destinations_1.SqsDestination(this.errorQueue);
            monocdk_nag_1.NagSuppressions.addResourceSuppressions(this.errorDeadLetterQueue, [
                { id: 'AwsSolutions-SQS3', reason: 'This queue is a DLQ.' },
            ]);
        }
        else {
            this.errorDest = props.onError;
        }
        const vpc = new aws_ec2_1.Vpc(this, 'ScanVPC', {
            subnetConfiguration: [
                {
                    subnetType: aws_ec2_1.SubnetType.PRIVATE_ISOLATED,
                    name: 'Isolated',
                },
            ],
        });
        vpc.addFlowLog('FlowLogs');
        this._s3Gw = vpc.addGatewayEndpoint('S3Endpoint', {
            service: aws_ec2_1.GatewayVpcEndpointAwsService.S3,
        });
        const fileSystem = new aws_efs_1.FileSystem(this, 'ScanFileSystem', {
            vpc: vpc,
            encrypted: props.efsEncryption === false ? false : true,
            lifecyclePolicy: aws_efs_1.LifecyclePolicy.AFTER_7_DAYS,
            performanceMode: aws_efs_1.PerformanceMode.GENERAL_PURPOSE,
            removalPolicy: monocdk_1.RemovalPolicy.DESTROY,
            securityGroup: new aws_ec2_1.SecurityGroup(this, 'ScanFileSystemSecurityGroup', {
                vpc: vpc,
                allowAllOutbound: false,
            }),
        });
        const lambda_ap = fileSystem.addAccessPoint('ScanLambdaAP', {
            createAcl: {
                ownerGid: '1000',
                ownerUid: '1000',
                permissions: '755',
            },
            posixUser: {
                gid: '1000',
                uid: '1000',
            },
            path: this._efsRootPath,
        });
        const logs_bucket = (_b = props.defsBucketAccessLogsConfig) === null || _b === void 0 ? void 0 : _b.logsBucket;
        const logs_bucket_prefix = (_c = props.defsBucketAccessLogsConfig) === null || _c === void 0 ? void 0 : _c.logsPrefix;
        if (logs_bucket === true || logs_bucket === undefined) {
            this.defsAccessLogsBucket = new aws_s3_1.Bucket(this, 'VirusDefsAccessLogsBucket', {
                encryption: aws_s3_1.BucketEncryption.S3_MANAGED,
                removalPolicy: monocdk_1.RemovalPolicy.RETAIN,
                serverAccessLogsPrefix: 'access-logs-bucket-logs',
                blockPublicAccess: {
                    blockPublicAcls: true,
                    blockPublicPolicy: true,
                    ignorePublicAcls: true,
                    restrictPublicBuckets: true,
                },
            });
            this.defsAccessLogsBucket.addToResourcePolicy(new aws_iam_1.PolicyStatement({
                effect: aws_iam_1.Effect.DENY,
                actions: ['s3:*'],
                resources: [
                    this.defsAccessLogsBucket.arnForObjects('*'),
                    this.defsAccessLogsBucket.bucketArn,
                ],
                principals: [new aws_iam_1.AnyPrincipal()],
                conditions: {
                    Bool: {
                        'aws:SecureTransport': false,
                    },
                },
            }));
        }
        else if (logs_bucket != false) {
            this.defsAccessLogsBucket = logs_bucket;
        }
        const defs_bucket = new aws_s3_1.Bucket(this, 'VirusDefsBucket', {
            encryption: aws_s3_1.BucketEncryption.S3_MANAGED,
            removalPolicy: monocdk_1.RemovalPolicy.DESTROY,
            autoDeleteObjects: true,
            serverAccessLogsBucket: this.defsAccessLogsBucket,
            serverAccessLogsPrefix: logs_bucket === false ? undefined : logs_bucket_prefix,
            blockPublicAccess: {
                blockPublicAcls: true,
                blockPublicPolicy: true,
                ignorePublicAcls: true,
                restrictPublicBuckets: true,
            },
        });
        defs_bucket.addToResourcePolicy(new aws_iam_1.PolicyStatement({
            effect: aws_iam_1.Effect.DENY,
            actions: ['s3:*'],
            resources: [defs_bucket.arnForObjects('*'), defs_bucket.bucketArn],
            principals: [new aws_iam_1.AnyPrincipal()],
            conditions: {
                Bool: {
                    'aws:SecureTransport': false,
                },
            },
        }));
        defs_bucket.addToResourcePolicy(new aws_iam_1.PolicyStatement({
            effect: aws_iam_1.Effect.ALLOW,
            actions: ['s3:GetObject', 's3:ListBucket'],
            resources: [defs_bucket.arnForObjects('*'), defs_bucket.bucketArn],
            principals: [new aws_iam_1.AnyPrincipal()],
            conditions: {
                StringEquals: {
                    'aws:SourceVpce': this._s3Gw.vpcEndpointId,
                },
            },
        }));
        defs_bucket.addToResourcePolicy(new aws_iam_1.PolicyStatement({
            effect: aws_iam_1.Effect.DENY,
            actions: ['s3:PutBucketPolicy', 's3:DeleteBucketPolicy'],
            resources: [defs_bucket.bucketArn],
            notPrincipals: [new aws_iam_1.AccountRootPrincipal()],
        }));
        this._s3Gw.addToPolicy(new aws_iam_1.PolicyStatement({
            effect: aws_iam_1.Effect.ALLOW,
            actions: ['s3:GetObject', 's3:ListBucket'],
            resources: [defs_bucket.arnForObjects('*'), defs_bucket.bucketArn],
            principals: [new aws_iam_1.AnyPrincipal()],
        }));
        this._scanFunction = new aws_lambda_1.DockerImageFunction(this, 'ServerlessClamscan', {
            code: aws_lambda_1.DockerImageCode.fromImageAsset(path.join(__dirname, '../assets/lambda/code/scan'), {
                buildArgs: {
                    // Only force update the docker layer cache once a day
                    CACHE_DATE: new Date().toDateString(),
                },
                extraHash: Date.now().toString(),
            }),
            onSuccess: this.resultDest,
            onFailure: this.errorDest,
            filesystem: aws_lambda_1.FileSystem.fromEfsAccessPoint(lambda_ap, this._efsMountPath),
            vpc: vpc,
            vpcSubnets: { subnets: vpc.isolatedSubnets },
            allowAllOutbound: false,
            timeout: monocdk_1.Duration.minutes(15),
            memorySize: 10240,
            reservedConcurrentExecutions: props.reservedConcurrency,
            environment: {
                EFS_MOUNT_PATH: this._efsMountPath,
                EFS_DEF_PATH: this._efsDefsPath,
                DEFS_URL: defs_bucket.virtualHostedUrlForObject(),
                POWERTOOLS_METRICS_NAMESPACE: 'serverless-clamscan',
                POWERTOOLS_SERVICE_NAME: 'virus-scan',
            },
        });
        if (this._scanFunction.role) {
            monocdk_nag_1.NagSuppressions.addResourceSuppressions(this._scanFunction.role, [
                {
                    id: 'AwsSolutions-IAM4',
                    reason: 'The AWSLambdaBasicExecutionRole does not provide permissions beyond uploading logs to CloudWatch. The AWSLambdaVPCAccessExecutionRole is required for functions with VPC access to manage elastic network interfaces.',
                },
            ]);
            monocdk_nag_1.NagSuppressions.addResourceSuppressions(this._scanFunction.role, [
                {
                    id: 'AwsSolutions-IAM5',
                    reason: 'The EFS mount point permissions are controlled through a condition which limit the scope of the * resources.',
                },
            ], true);
        }
        this._scanFunction.connections.allowToAnyIpv4(aws_ec2_1.Port.tcp(443), 'Allow outbound HTTPS traffic for S3 access.');
        defs_bucket.grantRead(this._scanFunction);
        const download_defs = new aws_lambda_1.DockerImageFunction(this, 'DownloadDefs', {
            code: aws_lambda_1.DockerImageCode.fromImageAsset(path.join(__dirname, '../assets/lambda/code/download_defs'), {
                buildArgs: {
                    // Only force update the docker layer cache once a day
                    CACHE_DATE: new Date().toDateString(),
                },
                extraHash: Date.now().toString(),
            }),
            timeout: monocdk_1.Duration.minutes(5),
            memorySize: 1024,
            environment: {
                DEFS_BUCKET: defs_bucket.bucketName,
                POWERTOOLS_SERVICE_NAME: 'freshclam-update',
            },
        });
        const stack = monocdk_1.Stack.of(this);
        if (download_defs.role) {
            const download_defs_role = `arn:${stack.partition}:sts::${stack.account}:assumed-role/${download_defs.role.roleName}/${download_defs.functionName}`;
            const download_defs_assumed_principal = new aws_iam_1.ArnPrincipal(download_defs_role);
            defs_bucket.addToResourcePolicy(new aws_iam_1.PolicyStatement({
                effect: aws_iam_1.Effect.DENY,
                actions: ['s3:PutObject*'],
                resources: [defs_bucket.arnForObjects('*')],
                notPrincipals: [download_defs.role, download_defs_assumed_principal],
            }));
            defs_bucket.grantReadWrite(download_defs);
            monocdk_nag_1.NagSuppressions.addResourceSuppressions(download_defs.role, [
                {
                    id: 'AwsSolutions-IAM4',
                    reason: 'The AWSLambdaBasicExecutionRole does not provide permissions beyond uploading logs to CloudWatch.',
                },
            ]);
            monocdk_nag_1.NagSuppressions.addResourceSuppressions(download_defs.role, [
                {
                    id: 'AwsSolutions-IAM5',
                    reason: 'The function is allowed to invoke the download defs Lambda function.',
                },
            ], true);
        }
        new aws_events_1.Rule(this, 'VirusDefsUpdateRule', {
            schedule: aws_events_1.Schedule.rate(monocdk_1.Duration.hours(12)),
            targets: [new aws_events_targets_1.LambdaFunction(download_defs)],
        });
        const init_defs_cr = new aws_lambda_1.Function(this, 'InitDefs', {
            runtime: aws_lambda_1.Runtime.PYTHON_3_9,
            code: aws_lambda_1.Code.fromAsset(path.join(__dirname, '../assets/lambda/code/initialize_defs_cr')),
            handler: 'lambda.lambda_handler',
            timeout: monocdk_1.Duration.minutes(5),
        });
        download_defs.grantInvoke(init_defs_cr);
        if (init_defs_cr.role) {
            monocdk_nag_1.NagSuppressions.addResourceSuppressions(init_defs_cr.role, [
                {
                    id: 'AwsSolutions-IAM4',
                    reason: 'The AWSLambdaBasicExecutionRole does not provide permissions beyond uploading logs to CloudWatch.',
                },
                {
                    id: 'AwsSolutions-IAM5',
                    reason: 'The AWSLambdaBasicExecutionRole does not provide permissions beyond uploading logs to CloudWatch.',
                },
            ], true);
        }
        new monocdk_1.CustomResource(this, 'InitDefsCr', {
            serviceToken: init_defs_cr.functionArn,
            properties: {
                FnName: download_defs.functionName,
            },
        });
        if (props.buckets) {
            props.buckets.forEach((bucket) => {
                this.addSourceBucket(bucket);
            });
        }
    }
    /**
     * @returns ArnPrincipal the ARN of the assumed role principal for the scan function
     */
    get scanAssumedPrincipal() {
        if (this._scanFunction.role) {
            const stack = monocdk_1.Stack.of(this);
            const scan_assumed_role = `arn:${stack.partition}:sts::${stack.account}:assumed-role/${this._scanFunction.role.roleName}/${this._scanFunction.functionName}`;
            return new aws_iam_1.ArnPrincipal(scan_assumed_role);
        }
        else {
            throw new Error('The scan function role is undefined');
        }
    }
    /**
     * Returns the statement that should be added to the bucket policy
       in order to prevent objects to be accessed when they are not clean
       or there have been scanning errors: this policy should be added
       manually if external buckets are passed to addSourceBucket()
     * @param bucket The bucket which you need to protect with the policy
     * @returns PolicyStatement the policy statement if available
     */
    getPolicyStatementForBucket(bucket) {
        if (this._scanFunction.role) {
            const scan_assumed_principal = this.scanAssumedPrincipal;
            return new aws_iam_1.PolicyStatement({
                effect: aws_iam_1.Effect.DENY,
                actions: ['s3:GetObject'],
                resources: [bucket.arnForObjects('*')],
                notPrincipals: [this._scanFunction.role, scan_assumed_principal],
                conditions: {
                    StringEquals: {
                        's3:ExistingObjectTag/scan-status': [
                            'IN PROGRESS',
                            'INFECTED',
                            'ERROR',
                        ],
                    },
                },
            });
        }
        else {
            throw new Error("Can't generate a valid S3 bucket policy, the scan function role is undefined");
        }
    }
    /**
     * Sets the specified S3 Bucket as a s3:ObjectCreate* for the ClamAV function.
       Grants the ClamAV function permissions to get and tag objects.
       Adds a bucket policy to disallow GetObject operations on files that are tagged 'IN PROGRESS', 'INFECTED', or 'ERROR'.
     * @param bucket The bucket to add the scanning bucket policy and s3:ObjectCreate* trigger to.
     */
    addSourceBucket(bucket) {
        bucket.addEventNotification(aws_s3_1.EventType.OBJECT_CREATED, new aws_s3_notifications_1.LambdaDestination(this._scanFunction));
        bucket.grantRead(this._scanFunction);
        this._scanFunction.addToRolePolicy(new aws_iam_1.PolicyStatement({
            effect: aws_iam_1.Effect.ALLOW,
            actions: ['s3:PutObjectTagging', 's3:PutObjectVersionTagging'],
            resources: [bucket.arnForObjects('*')],
        }));
        if (this._scanFunction.role) {
            const scan_assumed_principal = this.scanAssumedPrincipal;
            this._s3Gw.addToPolicy(new aws_iam_1.PolicyStatement({
                effect: aws_iam_1.Effect.ALLOW,
                actions: ['s3:GetObject*', 's3:GetBucket*', 's3:List*'],
                resources: [bucket.bucketArn, bucket.arnForObjects('*')],
                principals: [this._scanFunction.role, scan_assumed_principal],
            }));
            this._s3Gw.addToPolicy(new aws_iam_1.PolicyStatement({
                effect: aws_iam_1.Effect.ALLOW,
                actions: ['s3:PutObjectTagging', 's3:PutObjectVersionTagging'],
                resources: [bucket.arnForObjects('*')],
                principals: [this._scanFunction.role, scan_assumed_principal],
            }));
            const result = bucket.addToResourcePolicy(this.getPolicyStatementForBucket(bucket));
            if (!result.statementAdded && !this.useImportedBuckets) {
                throw new Error('acceptResponsibilityForUsingImportedBucket must be set when adding an imported bucket. When using imported buckets the user is responsible for adding the required policy statement to the bucket policy: `getPolicyStatementForBucket()` can be used to retrieve the policy statement required by the solution');
            }
        }
    }
}
exports.ServerlessClamscan = ServerlessClamscan;
_a = JSII_RTTI_SYMBOL_1;
ServerlessClamscan[_a] = { fqn: "monocdk-serverless-clamscan.ServerlessClamscan", version: "1.2.7" };
//# sourceMappingURL=data:application/json;base64,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