import json
import os 
import logging
import random

CONFIG_PATH = "/config.json"

_logger = logging.getLogger(__name__)

class Node():
    
    def __init__(self, path="./cetd"):
        self.path = path
        self.is_initialized = False
        self.load()
    
    def load(self):
        path = self.path + CONFIG_PATH
        if not os.path.isfile(path):
            self.is_initialized = False
            return
        with open(path, 'r') as file:
            map = json.load(file)
            for key in map:
                self.set(key, map.get(key))
    
    def save(self):
        path = self.path + CONFIG_PATH
        with open(path, 'w') as file:
            json.dump(self.__dict__, file, indent=4)
    
    def __setattr__(self, key, value):
        if callable(value):
            raise ValueError("""
            Forbidden to set a callback function for key {}.
            """.format(key))
        self.__dict__[key] = value
        return value
    
    def set(self, key, value):
        return self.__setattr__(key, value)
    
    def __getattr__(self, key):
        return self.get(key)
    
    def get(self, key, default=None):
        if key in self.__dict__:
            return self.__dict__[key]
        return default
    
    def __str__(self):
        return json.dumps(self.__dict__, indent=4)

    def load_from_args(self, args):
        """
        Copies all arguments into the node metadata
        """
        if hasattr(args, '__dict__'):
            return self.load_from_args(vars(args))
        for key in args:
            if callable(args[key]):
                continue
            self.set(key, args[key])


def is_node(path):
    if not os.path.exists(path):
        return False
    
    path = path + CONFIG_PATH
    if not os.path.isfile(path):
        return False
    
    # TODO: maso, 2022: add more condition to check if it is a relay node
    return True


def create_node(path):
    """ Creates new Relay object
    
    Relay object point to a relay node with configuration and related
    configs. This function create a new relay and return it as result.
    """
    # create folder if not exist
    if not os.path.isdir(path):
        os.mkdir(path)
    
    relay = Node(
        path=path
    )
    return relay


def check_node_initialized(relay):
    _logger.info("Check the node folder {}".format(relay.path))
    if not relay.is_initialized:
        _logger.critical(msg="Node is not initialized! use `csc-docker-pool relay --name {} init`  to initialize the node.".format(relay.name))
        exit(1)


def create_node_with_name(args):
    """Creates a non existed node
    
    A runtime exception is raised if the node is initialized before. All arguments are
    sets as initial values. A random id will be generated by default.
    
    @params args An argsparser Namespace to set default values.
    @return a new unsaved node
    """
    name = args.name
    _logger.debug("Checking the node {}".format(name))
    node = load_node_with_name(args)
    
    if node.is_initialized:
        raise RuntimeError("""
             The same node with name {} was initialized before.
             Remove node with command `genz-cetd-docker relay remove --name={}` and then
             try again.
             """.format(name, name))
    node.load_from_args(args)
    node.id = str(random.randint(100000000, 1000000000))
    return node


def load_node_with_name(args):
    """ Load a node with name
    
    Node is a place where data are stored. This may be a real physical node or
    a virtual node. For example you may use a node to store keys.
    
    @param args: command arguments
    """
    name = args.name
    if not name:
        raise RuntimeError("Name must be set to load a node")
    path = os.getcwd() + '/' + name
    _logger.debug("Loading node from {}".format(path))
    return create_node(path)



