"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const fs = require("fs");
const jsonschema = require("jsonschema");
const semver = require("semver");
const assembly = require("./schema");
// this prefix is used by the CLI to identify this specific error.
// in which case we want to instruct the user to upgrade his CLI.
// see exec.ts#createAssembly
exports.VERSION_MISMATCH = 'Cloud assembly schema version mismatch';
/**
 * Protocol utility class.
 */
class Manifest {
    constructor() { }
    /**
     * Save manifest to file.
     *
     * @param manifest - manifest.
     */
    static save(manifest, filePath) {
        fs.writeFileSync(filePath, JSON.stringify(manifest, undefined, 2));
    }
    /**
     * Load manifest from file.
     *
     * @param filePath - path to the manifest file.
     */
    static load(filePath) {
        const raw = JSON.parse(fs.readFileSync(filePath, 'UTF-8'));
        Manifest.patchStackTags(raw);
        Manifest.validate(raw);
        return raw;
    }
    /**
     * Fetch the current schema version number.
     */
    static version() {
        // eslint-disable-next-line @typescript-eslint/no-require-imports
        return require('../schema/cloud-assembly.version.json').version;
    }
    static validate(manifest) {
        function parseVersion(version) {
            const ver = semver.valid(version);
            if (!ver) {
                throw new Error(`Invalid semver string: "${version}"`);
            }
            return ver;
        }
        const maxSupported = parseVersion(Manifest.version());
        const actual = parseVersion(manifest.version);
        // first validate the version should be accepted.
        if (semver.gt(actual, maxSupported)) {
            // we use a well known error prefix so that the CLI can identify this specific error
            // and print some more context to the user.
            throw new Error(`${exports.VERSION_MISMATCH}: Maximum schema version supported is ${maxSupported}, but found ${actual}`);
        }
        // now validate the format is good.
        const validator = new jsonschema.Validator();
        const result = validator.validate(manifest, Manifest.schema, {
            // does exist but is not in the TypeScript definitions
            nestedErrors: true,
            allowUnknownAttributes: false,
        });
        if (!result.valid) {
            throw new Error(`Invalid assembly manifest:\n${result}`);
        }
    }
    /**
     * This requires some explaining...
     *
     * We previously used `{ Key, Value }` for the object that represents a stack tag. (Notice the casing)
     * @link https://github.com/aws/aws-cdk/blob/v1.27.0/packages/aws-cdk/lib/api/cxapp/stacks.ts#L427.
     *
     * When that object moved to this package, it had to be JSII compliant, which meant the property
     * names must be `camelCased`, and not `PascalCased`. This meant it no longer matches the structure in the `manifest.json` file.
     * In order to support current manifest files, we have to translate the `PascalCased` representation to the new `camelCased` one.
     *
     * Note that the serialization itself still writes `PascalCased` because it relates to how CloudFormation expects it.
     *
     * Ideally, we would start writing the `camelCased` and translate to how CloudFormation expects it when needed. But this requires nasty
     * backwards-compatibility code and it just doesn't seem to be worth the effort.
     */
    static patchStackTags(manifest) {
        for (const artifact of Object.values(manifest.artifacts || [])) {
            if (artifact.type === assembly.ArtifactType.AWS_CLOUDFORMATION_STACK) {
                for (const metadataEntries of Object.values(artifact.metadata || [])) {
                    for (const metadataEntry of metadataEntries) {
                        if (metadataEntry.type === assembly.ArtifactMetadataEntryType.STACK_TAGS && metadataEntry.data) {
                            const metadataAny = metadataEntry;
                            metadataAny.data = metadataAny.data.map((t) => {
                                return { key: t.Key, value: t.Value };
                            });
                        }
                    }
                }
            }
        }
    }
}
exports.Manifest = Manifest;
// eslint-disable-next-line @typescript-eslint/no-require-imports
Manifest.schema = require('../schema/cloud-assembly.schema.json');
//# sourceMappingURL=data:application/json;base64,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