"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const fs = require("fs");
const path = require("path");
const semver = require("semver");
// eslint-disable-next-line import/no-extraneous-dependencies
const tjs = require("typescript-json-schema");
function log(message) {
    // tslint:disable-next-line:no-console
    console.log(message);
}
function bump() {
    const metadataPath = '../schema/cloud-assembly.version.json';
    // eslint-disable-next-line @typescript-eslint/no-require-imports
    const metadata = require(metadataPath);
    const oldVersion = metadata.version;
    const newVersion = semver.inc(oldVersion, 'major');
    log(`Updating schema version: ${oldVersion} -> ${newVersion}`);
    const out = path.join(__dirname, metadataPath);
    fs.writeFileSync(out, JSON.stringify({ version: newVersion }));
}
function generate(out, shouldBump) {
    const settings = {
        required: true,
        ref: true,
        topRef: true,
        noExtraProps: false,
        out,
    };
    const compilerOptions = {
        strictNullChecks: true,
    };
    const program = tjs.getProgramFromFiles([path.join(__dirname, '../lib/schema.d.ts')], compilerOptions);
    const schema = tjs.generateSchema(program, 'AssemblyManifest', settings);
    augmentDescription(schema);
    addAnyMetadataEntry(schema);
    if (shouldBump) {
        bump();
    }
    if (out) {
        log(`Generating schema to ${out}`);
        fs.writeFileSync(out, JSON.stringify(schema, null, 4));
    }
    return schema;
}
exports.generate = generate;
/**
 * Remove 'default' from the schema since its generated
 * from the tsdocs, which are not necessarily actual values,
 * but rather descriptive behavior.
 *
 * To keep this inforamtion in the schema, we append it to the
 * 'description' of the property.
 */
function augmentDescription(schema) {
    function _recurse(o) {
        for (const prop in o) {
            if (prop === 'description' && typeof o[prop] === 'string') {
                const description = o[prop];
                const defaultValue = o.default;
                if (!defaultValue) {
                    // property doesn't have a default value
                    // skip
                    continue;
                }
                const descriptionWithDefault = `${description} (Default ${defaultValue})`;
                delete o.default;
                o[prop] = descriptionWithDefault;
            }
            else if (typeof o[prop] === 'object') {
                _recurse(o[prop]);
            }
        }
    }
    _recurse(schema);
}
/**
 * Patch the properties of MetadataEntry to allow
 * specifying any free form data. This is needed since source
 * code doesn't allow this in order to enforce stricter jsii
 * compatibility checks.
 */
function addAnyMetadataEntry(schema) {
    schema.definitions.MetadataEntry.properties.data.anyOf.push({ description: 'Free form data.' });
}
//# sourceMappingURL=data:application/json;base64,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