from __future__ import annotations

from dataclasses import dataclass, field
from typing import Any

from aduib_rpc.client import ClientRequestInterceptor
from aduib_rpc.client.auth import CredentialsProvider, InMemoryCredentialsProvider
from aduib_rpc.client.auth.interceptor import AuthInterceptor
from aduib_rpc.discover.entities import ServiceInstance
from aduib_rpc.server.rpc_execution.service_func import ServiceFunc


@dataclass(slots=True)
class RpcRuntime:
    """Holds mutable runtime registries.

    Design goals:
    - make state explicit and resettable for tests
    - allow future per-server/per-client runtime instances
    - keep behavior compatible with the existing module-level globals
    """
    service_info: ServiceInstance = field(default=None)
    service_instances: dict[str, Any] = field(default_factory=dict)
    client_instances: dict[str, Any] = field(default_factory=dict)

    service_funcs: dict[str, ServiceFunc] = field(default_factory=dict)
    client_funcs: dict[str, ServiceFunc] = field(default_factory=dict)

    interceptors: list[ClientRequestInterceptor] = field(default_factory=list)
    credentials_provider: CredentialsProvider | None = None

    def reset(self) -> None:
        # Reset all runtime registries for test isolation.
        self.service_instances.clear()
        self.client_instances.clear()
        self.service_funcs.clear()
        self.client_funcs.clear()
        self.interceptors.clear()
        self.credentials_provider = None
        self.service_info = None

    def enable_auth(self) -> None:
        if not self.credentials_provider:
            self.credentials_provider = InMemoryCredentialsProvider()
        has_auth = any(isinstance(i, AuthInterceptor) for i in self.interceptors)
        if not has_auth:
            self.interceptors.append(AuthInterceptor(self.credentials_provider))


# Initialize lazily in get_runtime().
_global_runtime: RpcRuntime | None = None


def get_runtime() -> RpcRuntime:
    """Return the default global RpcRuntime.

    This keeps backward-compatibility for existing imports and decorators.
    """
    global _global_runtime
    if _global_runtime is None:
        _global_runtime = RpcRuntime()

    return _global_runtime

def set_service_info(service_info: ServiceInstance) -> None:
    runtime = get_runtime()
    runtime.service_info = service_info
