"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.EsbuildBundler = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const fs_1 = require("fs");
const path_1 = require("path");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const esbuild_provider_1 = require("./esbuild-provider");
const utils_1 = require("./utils");
/**
 * Low-level construct that can be used where `BundlingOptions` are required.
 * This class directly interfaces with esbuild and provides almost no configuration safeguards.
 *
 * @stability experimental
 */
class EsbuildBundler {
    /**
     * @stability experimental
     */
    constructor(
    /**
     * A path or list or map of paths to the entry points of your code.
     *
     * Relative paths are by default resolved from the current working directory.
     * To change the working directory, see `buildOptions.absWorkingDir`.
     *
     * Absolute paths can be used if files are part of the working directory.
     *
     * Examples:
     *  - `'src/index.ts'`
     *  - `require.resolve('./lambda')`
     *  - `['src/index.ts', 'src/util.ts']`
     *  - `{one: 'src/two.ts', two: 'src/one.ts'}`
     *
     * @stability experimental
     */
    entryPoints, 
    /**
     * Props to change the behaviour of the bundler.
     *
     * @stability experimental
     */
    props) {
        this.entryPoints = entryPoints;
        this.props = props;
        /**
         * @deprecated This value is ignored since the bundler is always using a locally installed version of esbuild. However the property is required to comply with the `BundlingOptions` interface.
         *
         * @stability deprecated
         */
        this.image = aws_cdk_lib_1.DockerImage.fromRegistry('scratch');
        if (props?.buildOptions?.outfile && props?.buildOptions?.outdir) {
            throw new Error('Cannot use both "outfile" and "outdir"');
        }
        this.local = {
            tryBundle: (outputDir, _options) => {
                if (this.props.copyDir) {
                    const copyDir = this.getCopyDirList(this.props.copyDir);
                    copyDir.forEach(([dest, src]) => {
                        const srcDir = path_1.resolve(this.props?.buildOptions?.absWorkingDir ?? process.cwd(), src);
                        const destDir = path_1.resolve(outputDir, dest);
                        const destToOutput = path_1.relative(outputDir, destDir);
                        if (destToOutput.startsWith('..') || path_1.isAbsolute(destToOutput)) {
                            throw new Error('Cannot copy files to outside of the asset staging directory. See docs for details.');
                        }
                        fs_1.mkdirSync(destDir, { recursive: true });
                        aws_cdk_lib_1.FileSystem.copyDirectory(srcDir, destDir);
                    });
                }
                try {
                    const buildFn = this.props.buildFn ?? esbuild_provider_1.EsbuildProvider.require(props.esbuildModulePath).buildSync;
                    const buildSync = esbuild_provider_1.EsbuildProvider.withEsbuildBinaryPath(buildFn, this.props.esbuildBinaryPath);
                    buildSync({
                        entryPoints,
                        color: process.env.NO_COLOR ? Boolean(process.env.NO_COLOR) : undefined,
                        ...(this.props?.buildOptions || {}),
                        ...this.getOutputOptions(outputDir, { normalize: path_1.normalize, join: path_1.join }),
                    });
                }
                catch (error) {
                    if (utils_1.isEsbuildError(error)) {
                        throw new Error(`Esbuild failed to bundle ${entryPoints}`);
                    }
                    throw error;
                }
                return true;
            },
        };
    }
    getCopyDirList(copyDir) {
        // Nothing to copy
        if (!copyDir) {
            return [];
        }
        // List of strings
        if (Array.isArray(copyDir)) {
            return copyDir.map((src) => ['.', src]);
        }
        // A map
        if (typeof copyDir === 'object' &&
            !Array.isArray(copyDir) &&
            copyDir !== null) {
            return Object
                .entries(copyDir)
                .flatMap(([dest, sources]) => {
                if (Array.isArray(sources)) {
                    return sources.map((src) => [dest, src]);
                }
                return [[dest, sources]];
            });
        }
        // A single string
        return [['.', copyDir]];
    }
    getOutputOptions(cdkOutputDir, path = path_1.posix) {
        if (this.props?.buildOptions?.outfile) {
            return {
                outdir: undefined,
                outfile: path.normalize(path.join(...[cdkOutputDir, this.props?.buildOptions?.outfile].filter(Boolean))),
            };
        }
        return {
            outdir: path.normalize(path.join(...[cdkOutputDir, this.props?.buildOptions?.outdir].filter(Boolean))),
            outfile: undefined,
        };
    }
}
exports.EsbuildBundler = EsbuildBundler;
_a = JSII_RTTI_SYMBOL_1;
EsbuildBundler[_a] = { fqn: "@mrgrain/cdk-esbuild.EsbuildBundler", version: "3.14.0" };
//# sourceMappingURL=data:application/json;base64,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