"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.EsbuildProvider = void 0;
const aws_cdk_lib_1 = require("aws-cdk-lib");
const constructs_1 = require("constructs");
const esbuild_source_1 = require("./esbuild-source");
class EsbuildProvider {
    /**
     * Load the esbuild module according to defined rules.
     */
    static require(path) {
        const module = path || process.env.CDK_ESBUILD_MODULE_PATH || esbuild_source_1.EsbuildSource.default || esbuild_source_1.Esbuild.name;
        return this._require(this.resolve(module));
    }
    /**
     * @internal
     */
    static _require(path) {
        // eslint-disable-next-line @typescript-eslint/no-require-imports
        return require(path);
    }
    /**
     * Invoke a function with a specific `process.env.ESBUILD_BINARY_PATH`
     * and restore the env var afterwards.
     */
    static withEsbuildBinaryPath(fn, esbuildBinaryPath) {
        if (!esbuildBinaryPath) {
            return fn;
        }
        return (...args) => {
            const originalEsbuildBinaryPath = process.env.ESBUILD_BINARY_PATH;
            if (esbuildBinaryPath) {
                process.env.ESBUILD_BINARY_PATH = esbuildBinaryPath;
            }
            const result = fn(...args);
            /**
             * only reset `ESBUILD_BINARY_PATH` if it was explicitly set via the construct props
             * since `esbuild` itself sometimes sets it (eg. when running in yarn 2 plug&play)
             */
            if (esbuildBinaryPath) {
                process.env.ESBUILD_BINARY_PATH = originalEsbuildBinaryPath;
            }
            return result;
        };
    }
    /**
     * Resolve a token without context
     */
    static resolve(token) {
        if (!aws_cdk_lib_1.Token.isUnresolved(token)) {
            return token;
        }
        return aws_cdk_lib_1.Tokenization.resolve(token, {
            scope: new constructs_1.Construct(undefined, ''),
            resolver: new aws_cdk_lib_1.DefaultTokenResolver(new aws_cdk_lib_1.StringConcat()),
        });
    }
}
exports.EsbuildProvider = EsbuildProvider;
//# sourceMappingURL=data:application/json;base64,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