"use strict";
/**
 *  Copyright 2021 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
// Imports
const core_1 = require("@aws-cdk/core");
const lib_1 = require("../lib");
const assert_1 = require("@aws-cdk/assert");
require("@aws-cdk/assert/jest");
const sqs = require("@aws-cdk/aws-sqs");
const kms = require("@aws-cdk/aws-kms");
// --------------------------------------------------------------
// Pattern deployment with default props
// --------------------------------------------------------------
test('Pattern deployment with default props', () => {
    // Initial Setup
    const stack = new core_1.Stack();
    const props = {
        iotTopicRuleProps: {
            topicRulePayload: {
                ruleDisabled: false,
                description: "Processing messages from IoT devices or factory machines",
                sql: "SELECT * FROM 'test/topic/#'",
                actions: []
            }
        }
    };
    new lib_1.IotToSqs(stack, 'test-iot-sqs', props);
    expect(assert_1.SynthUtils.toCloudFormation(stack)).toMatchSnapshot();
    // Creates a default sqs queue
    expect(stack).toHaveResource("AWS::SQS::Queue", {
        KmsMasterKeyId: {
            "Fn::GetAtt": [
                "testiotsqsEncryptionKey64EE64B1",
                "Arn"
            ]
        }
    });
    // Creates a dead letter queue
    expect(stack).toHaveResource("AWS::SQS::Queue", {
        KmsMasterKeyId: "alias/aws/sqs"
    });
    // Creates an IoT Topic Rule
    expect(stack).toHaveResource("AWS::IoT::TopicRule", {
        TopicRulePayload: {
            Actions: [
                {
                    Sqs: {
                        QueueUrl: { Ref: "testiotsqsqueue630B4C1F" },
                        RoleArn: {
                            "Fn::GetAtt": [
                                "testiotsqsiotactionsrole93B1D327",
                                "Arn"
                            ]
                        }
                    }
                }
            ],
            Description: "Processing messages from IoT devices or factory machines",
            RuleDisabled: false,
            Sql: "SELECT * FROM 'test/topic/#'"
        }
    });
    // Creates an encryption key
    expect(stack).toHaveResource("AWS::KMS::Key", {
        EnableKeyRotation: true
    });
});
// --------------------------------------------------------------
// Testing with existing SQS Queue
// --------------------------------------------------------------
test('Pattern deployment with existing queue', () => {
    // Initial Setup
    const stack = new core_1.Stack();
    const queue = new sqs.Queue(stack, 'existing-queue-obj', {
        queueName: 'existing-queue-obj'
    });
    const props = {
        existingQueueObj: queue,
        iotTopicRuleProps: {
            topicRulePayload: {
                ruleDisabled: false,
                description: "Processing messages from IoT devices or factory machines",
                sql: "SELECT * FROM 'test/topic/#'",
                actions: []
            }
        }
    };
    new lib_1.IotToSqs(stack, 'test-iot-sqs', props);
    expect(assert_1.SynthUtils.toCloudFormation(stack)).toMatchSnapshot();
    // Creates a default sqs queue
    expect(stack).toHaveResource("AWS::SQS::Queue", {
        QueueName: "existing-queue-obj"
    });
});
// --------------------------------------------------------------
// Testing with passing queue and dead letter queue props
// --------------------------------------------------------------
test('Pattern deployment with queue and dead letter queue props', () => {
    // Initial Setup
    const stack = new core_1.Stack();
    const props = {
        deadLetterQueueProps: {
            queueName: 'dlq-name'
        },
        queueProps: {
            queueName: 'queue-name'
        },
        iotTopicRuleProps: {
            topicRulePayload: {
                ruleDisabled: false,
                description: "Processing messages from IoT devices or factory machines",
                sql: "SELECT * FROM 'test/topic/#'",
                actions: []
            }
        }
    };
    new lib_1.IotToSqs(stack, 'test-iot-sqs', props);
    expect(assert_1.SynthUtils.toCloudFormation(stack)).toMatchSnapshot();
    // Creates a queue using the provided props
    expect(stack).toHaveResource("AWS::SQS::Queue", {
        QueueName: "queue-name",
        RedrivePolicy: {
            deadLetterTargetArn: {
                "Fn::GetAtt": [
                    "testiotsqsdeadLetterQueue66A04E81",
                    "Arn",
                ],
            },
            maxReceiveCount: 15
        }
    });
    // Creates a dead letter queue using the provided props
    expect(stack).toHaveResource("AWS::SQS::Queue", {
        QueueName: "dlq-name"
    });
});
// --------------------------------------------------------------
// Testing with dead letter queue turned off
// --------------------------------------------------------------
test('Pattern deployment with dead letter queue turned off', () => {
    // Initial Setup
    const stack = new core_1.Stack();
    const props = {
        deployDeadLetterQueue: false,
        queueProps: {
            queueName: 'queue-name'
        },
        iotTopicRuleProps: {
            topicRulePayload: {
                ruleDisabled: false,
                description: "Processing messages from IoT devices or factory machines",
                sql: "SELECT * FROM 'test/topic/#'",
                actions: []
            }
        }
    };
    new lib_1.IotToSqs(stack, 'test-iot-sqs', props);
    expect(assert_1.SynthUtils.toCloudFormation(stack)).toMatchSnapshot();
    // Creates a queue using the provided props
    expect(stack).toHaveResource("AWS::SQS::Queue", {
        QueueName: "queue-name"
    });
    // Does not create the default dead letter queue
    expect(stack).not.toHaveResource("AWS::SQS::Queue", {
        KmsMasterKeyId: "alias/aws/sqs"
    });
});
// --------------------------------------------------------------
// Testing with custom maxReceiveCount
// --------------------------------------------------------------
test('Pattern deployment with custom maxReceiveCount', () => {
    // Initial Setup
    const stack = new core_1.Stack();
    const props = {
        deadLetterQueueProps: {
            queueName: 'dlq-name'
        },
        deployDeadLetterQueue: true,
        maxReceiveCount: 1,
        queueProps: {
            queueName: 'queue-name'
        },
        iotTopicRuleProps: {
            topicRulePayload: {
                ruleDisabled: false,
                description: "Processing messages from IoT devices or factory machines",
                sql: "SELECT * FROM 'test/topic/#'",
                actions: []
            }
        }
    };
    new lib_1.IotToSqs(stack, 'test-iot-sqs', props);
    expect(assert_1.SynthUtils.toCloudFormation(stack)).toMatchSnapshot();
    // Creates a queue using the provided props
    expect(stack).toHaveResource("AWS::SQS::Queue", {
        QueueName: "queue-name",
        RedrivePolicy: {
            deadLetterTargetArn: {
                "Fn::GetAtt": [
                    "testiotsqsdeadLetterQueue66A04E81",
                    "Arn",
                ],
            },
            maxReceiveCount: 1
        },
    });
});
// --------------------------------------------------------------
// Testing without creating a KMS key
// --------------------------------------------------------------
test('Pattern deployment without creating a KMS key', () => {
    // Initial Setup
    const stack = new core_1.Stack();
    const props = {
        enableEncryptionWithCustomerManagedKey: false,
        iotTopicRuleProps: {
            topicRulePayload: {
                ruleDisabled: false,
                description: "Processing messages from IoT devices or factory machines",
                sql: "SELECT * FROM 'test/topic/#'",
                actions: []
            }
        }
    };
    new lib_1.IotToSqs(stack, 'test-iot-sqs', props);
    expect(assert_1.SynthUtils.toCloudFormation(stack)).toMatchSnapshot();
    // Creates a default sqs queue
    expect(stack).toHaveResource("AWS::SQS::Queue", {
        KmsMasterKeyId: "alias/aws/sqs"
    });
    // Creates a dead letter queue
    expect(stack).toHaveResource("AWS::SQS::Queue", {
        KmsMasterKeyId: "alias/aws/sqs"
    });
    // Creates an IoT Topic Rule
    expect(stack).toHaveResource("AWS::IoT::TopicRule", {
        TopicRulePayload: {
            Actions: [
                {
                    Sqs: {
                        QueueUrl: { Ref: "testiotsqsqueue630B4C1F" },
                        RoleArn: {
                            "Fn::GetAtt": [
                                "testiotsqsiotactionsrole93B1D327",
                                "Arn"
                            ]
                        }
                    }
                }
            ],
            Description: "Processing messages from IoT devices or factory machines",
            RuleDisabled: false,
            Sql: "SELECT * FROM 'test/topic/#'"
        }
    });
    // Does not create an encryption key
    expect(stack).not.toHaveResource("AWS::KMS::Key");
});
// --------------------------------------------------------------
// Testing with existing KMS key
// --------------------------------------------------------------
test('Pattern deployment with existing KMS key', () => {
    // Initial Setup
    const stack = new core_1.Stack();
    const kmsKey = new kms.Key(stack, 'existing-key', {
        enableKeyRotation: false,
        alias: 'existing-key-alias'
    });
    const props = {
        encryptionKey: kmsKey,
        iotTopicRuleProps: {
            topicRulePayload: {
                ruleDisabled: false,
                description: "Processing messages from IoT devices or factory machines",
                sql: "SELECT * FROM 'test/topic/#'",
                actions: []
            }
        }
    };
    new lib_1.IotToSqs(stack, 'test-iot-sqs', props);
    expect(assert_1.SynthUtils.toCloudFormation(stack)).toMatchSnapshot();
    // Creates a default sqs queue
    expect(stack).toHaveResource("AWS::SQS::Queue", {
        KmsMasterKeyId: {
            "Fn::GetAtt": [
                "existingkey205DFC01",
                "Arn"
            ]
        }
    });
    // Creates a dead letter queue
    expect(stack).toHaveResource("AWS::SQS::Queue", {
        KmsMasterKeyId: "alias/aws/sqs"
    });
    // Creates an IoT Topic Rule
    expect(stack).toHaveResource("AWS::IoT::TopicRule", {
        TopicRulePayload: {
            Actions: [
                {
                    Sqs: {
                        QueueUrl: { Ref: "testiotsqsqueue630B4C1F" },
                        RoleArn: {
                            "Fn::GetAtt": [
                                "testiotsqsiotactionsrole93B1D327",
                                "Arn"
                            ]
                        }
                    }
                }
            ],
            Description: "Processing messages from IoT devices or factory machines",
            RuleDisabled: false,
            Sql: "SELECT * FROM 'test/topic/#'"
        }
    });
    // Uses the provided key
    expect(stack).toHaveResource("AWS::KMS::Key", {
        EnableKeyRotation: false
    });
});
// --------------------------------------------------------------
// Testing with passing KMS key props
// --------------------------------------------------------------
test('Pattern deployment passing KMS key props', () => {
    // Initial Setup
    const stack = new core_1.Stack();
    const props = {
        encryptionKeyProps: {
            enableKeyRotation: false,
            alias: 'new-key-alias-from-props'
        },
        iotTopicRuleProps: {
            topicRulePayload: {
                ruleDisabled: false,
                description: "Processing messages from IoT devices or factory machines",
                sql: "SELECT * FROM 'test/topic/#'",
                actions: []
            }
        }
    };
    new lib_1.IotToSqs(stack, 'test-iot-sqs', props);
    expect(assert_1.SynthUtils.toCloudFormation(stack)).toMatchSnapshot();
    // Creates a default sqs queue
    expect(stack).toHaveResource("AWS::SQS::Queue", {
        KmsMasterKeyId: {
            "Fn::GetAtt": [
                "testiotsqsEncryptionKey64EE64B1",
                "Arn"
            ]
        }
    });
    // Creates a dead letter queue
    expect(stack).toHaveResource("AWS::SQS::Queue", {
        KmsMasterKeyId: "alias/aws/sqs"
    });
    // Creates an IoT Topic Rule
    expect(stack).toHaveResource("AWS::IoT::TopicRule", {
        TopicRulePayload: {
            Actions: [
                {
                    Sqs: {
                        QueueUrl: { Ref: "testiotsqsqueue630B4C1F" },
                        RoleArn: {
                            "Fn::GetAtt": [
                                "testiotsqsiotactionsrole93B1D327",
                                "Arn"
                            ]
                        }
                    }
                }
            ],
            Description: "Processing messages from IoT devices or factory machines",
            RuleDisabled: false,
            Sql: "SELECT * FROM 'test/topic/#'"
        }
    });
    // Uses the props to create the key
    expect(stack).toHaveResource("AWS::KMS::Key", {
        EnableKeyRotation: false
    });
    expect(stack).toHaveResource("AWS::KMS::Alias", {
        AliasName: "alias/new-key-alias-from-props",
        TargetKeyId: {
            "Fn::GetAtt": [
                "testiotsqsEncryptionKey64EE64B1",
                "Arn",
            ]
        }
    });
});
// --------------------------------------------------------------
// Testing with passing a FIFO queue (not supported by IoT)
// --------------------------------------------------------------
test('Pattern deployment with passing a FIFO queue (not supported by IoT)', () => {
    // Initial Setup
    const stack = new core_1.Stack();
    const queue = new sqs.Queue(stack, 'existing-fifo-queue-obj', {
        queueName: 'existing-queue.fifo',
        fifo: true
    });
    const props = {
        existingQueueObj: queue,
        iotTopicRuleProps: {
            topicRulePayload: {
                ruleDisabled: false,
                description: "Processing messages from IoT devices or factory machines",
                sql: "SELECT * FROM 'test/topic/#'",
                actions: []
            }
        }
    };
    expect.assertions(1);
    try {
        new lib_1.IotToSqs(stack, 'test-iot-sqs', props);
    }
    catch (err) {
        expect(err.message).toBe('The IoT SQS action doesn\'t support Amazon SQS FIFO (First-In-First-Out) queues');
    }
});
//# sourceMappingURL=data:application/json;base64,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