# AUTOGENERATED! DO NOT EDIT! File to edit: ../nbs/m3.ipynb.

# %% auto 0
__all__ = ['M3Info', 'Yearly', 'Quarterly', 'Monthly', 'Other', 'M3']

# %% ../nbs/m3.ipynb 2
import os
from dataclasses import dataclass
from typing import Optional, Tuple

import pandas as pd

from .utils import download_file, Info

# %% ../nbs/m3.ipynb 4
@dataclass
class Yearly:
    seasonality: int = 1
    horizon: int = 6
    freq: str = 'Y'
    sheet_name: str = 'M3Year'
    name: str = 'Yearly'
    n_ts: int = 645

@dataclass
class Quarterly:
    seasonality: int = 4
    horizon: int = 8
    freq: str = 'Q'
    sheet_name: str = 'M3Quart'
    name: str = 'Quarterly'
    n_ts: int = 756

@dataclass
class Monthly:
    seasonality: int = 12
    horizon: int = 18
    freq: str = 'M'
    sheet_name: str = 'M3Month'
    name: str = 'Monthly'
    n_ts: int = 1428

@dataclass
class Other:
    seasonality: int = 1
    horizon: int = 8
    freq: str = 'D'
    sheet_name: str = 'M3Other'
    name: str = 'Other'
    n_ts: int = 174

# %% ../nbs/m3.ipynb 5
M3Info = Info((Yearly, Quarterly, Monthly, Other))

# %% ../nbs/m3.ipynb 6
def _return_year(ts: pd.DataFrame) -> int:
    year = ts.iloc[0]
    year = year if year != 0 else 1970

    return year

# %% ../nbs/m3.ipynb 7
@dataclass
class M3:
    
    source_url = 'https://forecasters.org/data/m3comp/M3C.xls'

    @staticmethod
    def load(directory: str,
             group: str) -> Tuple[pd.DataFrame, 
                                  Optional[pd.DataFrame], 
                                  Optional[pd.DataFrame]]:
        """
        Downloads and loads M3 data.

        Parameters
        ----------
        directory: str
            Directory where data will be downloaded.
        group: str
            Group name.
            Allowed groups: 'Yearly', 'Quarterly', 'Monthly', 'Other'.
            
        Returns
        -------
        df: pd.DataFrame
            Target time series with columns ['unique_id', 'ds', 'y'].
        """
        M3.download(directory)
        
        path = f'{directory}/m3/datasets/'
        file = f'{path}/M3C.xls'

        class_group = M3Info.get_group(group)

        df = pd.read_excel(file, sheet_name=class_group.sheet_name)
        df = df.rename(columns={'Series': 'unique_id',
                                'Category': 'category',
                                'Starting Year': 'year',
                                'Starting Month': 'month'})
        df['unique_id'] = [class_group.name[0] + str(i + 1) for i in range(len(df))]
        S = df.filter(items=['unique_id', 'category'])

        id_vars = list(df.columns[:6])
        df = pd.melt(df, id_vars=id_vars, var_name='ds', value_name='y')
        df = df.dropna().sort_values(['unique_id', 'ds']).reset_index(drop=True)

        freq = pd.tseries.frequencies.to_offset(class_group.freq)

        if group == 'Other':
            df['year'] = 1970

        df['ds'] = df.groupby('unique_id')['year'] \
                     .transform(lambda df: pd.date_range(f'{_return_year(df)}-01-01',
                                                         periods=df.shape[0],
                                                         freq=freq))
        df = df.filter(items=['unique_id', 'ds', 'y'])
        
        return df, None, None

    @staticmethod
    def download(directory: str) -> None:
        """
        Download M3 Dataset.
        
        Parameters
        ----------
        directory: str
            Directory path to download dataset.
        """
        path = f'{directory}/m3/datasets/'
        if not os.path.exists(path):
            download_file(path, M3.source_url)
