# AUTOGENERATED - modify shared_anyscale_util in root directory to make changes
from typing import Any, Optional, Type, TypeVar

from pydantic import BaseModel, validator
from typing_extensions import Literal


AWS_ROLE_ARN_RESOURCE_ID_PREFIX = "role/"
_AwsArnT = TypeVar("_AwsArnT", bound="AwsArn")


class AwsArn(BaseModel):
    partition: str
    service: str
    region: Optional[str]
    account_id: Optional[str]
    resource_id: str

    @validator("*", pre=True)
    def empty_str_to_none(cls, v: object) -> object:
        if v == "":
            return None
        return v

    @classmethod
    def from_string(cls: Type[_AwsArnT], arn: str) -> _AwsArnT:
        """Convert an AWS ARN string into a Pydantic Model.

        The ARN is a standard format in AWS with
        the following format:
        arn:<partition>:<service>:<region>:<account>:<resource>
        0       1          2         3        4         5

        > arn:aws:apprunner:us-west-2:959243851260:my_app

        Region & Account can be empty i.e:
        > arn:aws:s3:::bucketName
        """
        spl = arn.split(":")
        if len(spl) != 6 or spl[0] != "arn":
            raise ValueError(f"invalid arn: {arn}")
        return cls(
            partition=spl[1],
            service=spl[2],
            region=spl[3],
            account_id=spl[4],
            resource_id=spl[5],
        )

    def to_string(self) -> str:
        return "arn:{p}:{s}:{reg}:{a}:{rid}".format(
            p=self.partition,
            s=self.service,
            reg=self.region or "",
            a=self.account_id or "",
            rid=self.resource_id,
        )


class AwsRoleArn(AwsArn):
    """Role ARNs are of the format arn:aws:iam::<account>:role/<role name>."""

    partition: Literal["aws"]
    service: Literal["iam"]
    region: Literal[None] = None  # TODO: replace with types.NoneType in Python 3.10
    account_id: str

    @validator("resource_id", always=True)
    def resource_id_is_role(cls, v: object) -> object:
        if v is None:
            raise TypeError("role ARNs must have a resource ID")
        if isinstance(v, str):
            if not v.startswith(AWS_ROLE_ARN_RESOURCE_ID_PREFIX):
                raise ValueError(f"resource {v} is not a role")
        return v

    @classmethod
    def from_role_name(
        cls: Type[_AwsArnT], account_id: str, role_name: str
    ) -> _AwsArnT:
        return cls(
            partition="aws",
            service="iam",
            account_id=account_id,
            resource_id=f"role/{role_name}",
        )

    def to_role_name(self) -> str:
        """Get the role name from a role ARN.

        If there is a path, this gets the actual name (which is following the last slash).
        Role with path (`/anyscale/`): arn:aws:iam::959243851260:role/anyscale/thomas-test-role
        The Role name *is* `thomas-test-role`

        NOTE: The resource_id for a Role is "role/<role_name>".
        """
        return self.resource_id.split("/")[-1]


def get_dataplane_role_arn(
    account_id: str, cloud_resource_record: Optional[Any],
) -> AwsRoleArn:
    """ Helper method to get instance role from cloud_resource_record. """
    if cloud_resource_record and cloud_resource_record.aws_iam_role_arns:
        return AwsRoleArn.from_string(cloud_resource_record.aws_iam_role_arns[1])

    return AwsRoleArn.from_role_name(
        account_id=account_id, role_name="ray-autoscaler-v1",
    )
